/*
 * Copyright Kavita Bala, CS 6620, Cornell University
 * Contact kb@cs.cornell.edu
 */


package cs6620.scene;

import cs6620.geometry.Frame;
import cs6620.geometry.Ray;
import javax.vecmath.*;

/**
 * A simple pinhole camera with the image centered on the
 * positive Z axis and allowing independent X and Y FOVs.
 * The image space point (0,0) maps to a point in the reference frame space with a
 * z value of 1 and negative x and y values. The point (1,1) maps to a z value of
 * 1 and positive x and y values.
 */
public class Camera {
    private Vector3d eye = new Vector3d(0,0,1);
    private Vector3d target = new Vector3d(0,0,0);
    private Vector3d up = new Vector3d(0,1,0);
    
    private Frame frame =new Frame();
    
    /**
     * The tangent of half the horizontal and vertical FOV angles.
     */
    private double tanXFOV, tanYFOV;
    
    /**
     * The angle (in radians) of the X field of view.
     */
    private double xFOV;
    
    /**
     * The angle (in radians) of the Y field of view.
     */
    private double yFOV;

    private int pixelRaysPerPoint;
    private int pixelRaySamplingMethod;
    
    /**
     * Creates a new camera with a default frame of reference and default field of
     * views (pi/4,pi/4).
     */
    public Camera() {
        setDefaultFOVs();
    }
    
    /**
     * Sets the field of views to their default values (pi/4,pi/4).
     */
    public void setDefaultFOVs() {
        setFOVs(Math.toRadians(45),Math.toRadians(45));
    }
    
    /**
     * Creates a camera that is a copy of the given camera.
     * @param inCamera the camera to copy
     */
    public Camera(Camera inCamera) {
        frame = new Frame(inCamera.frame);
        xFOV = inCamera.xFOV;
        yFOV = inCamera.yFOV;
        tanXFOV = inCamera.tanXFOV;
        tanYFOV = inCamera.tanYFOV;
    }
    
    /**
     * Constructs an unnormalized vector from the origin of the reference frame of the
     * camera to a point on the image plane of the camera.
     * @param inU the u-coordinate of the point
     * @param inV the v-coordinate of the point
     * @param outVector the resulting (unnormalized) vector
     */
    public void getVector(double inU,double inV,Vector3d outVector) {
        inU = inU*2 - 1;
        inV = inV*2 - 1;
        outVector.set(tanXFOV*inU,tanYFOV*inV,1);
        frame.vectorToWorld(outVector);
    }
    
    /**
     * Sets outRay to be a ray from the origin of the reference frame of the camera to
     * a point in the image space of the camera. The direction component of the ray is
     * not normalized.
     * @param inU the u-coordinate of the point in image space
     * @param inV the v-coordinate of the point in image space
     * @param outRay the resulting ray with unnormalized direction
     */
    public void getUnnormalizedRay(double inU,double inV,Ray outRay) {
        outRay.origin.set(frame.origin);
        outRay.tStart = 0.0;
        outRay.tEnd = Double.MAX_VALUE;
        getVector(inU,inV,outRay.direction);
    }
    
    /**
     * Sets outRay to be a ray from the origin of the reference frame of the camera to
     * a point in the image space of the camera. The direction component of the ray is
     * normalized
     * @param inU the u-coordinate of the point in image space
     * @param inV the v-coordinate of the point in image space
     * @param outRay the resulting ray with a normalized direction component
     */
    public void getNormalizedRay(double inU,double inV,Ray outRay) {
        outRay.origin.set(frame.origin);
        outRay.tStart = 0.0;
        outRay.tEnd = Double.MAX_VALUE;
        getVector(inU,inV,outRay.direction);
        outRay.direction.normalize();
    }
    
    /**
     * Sets the x and y field of views, respectively (in radians).
     * @param inXFOV the X field of view
     * @param inYFOV the Y field of view
     */
    public void setFOVs(double inXFOV, double inYFOV) {
        xFOV = inXFOV;
        yFOV = inYFOV;
        tanXFOV = (double)Math.tan(xFOV/2);
        tanYFOV = (double)Math.tan(yFOV/2);
    }
    
    /**
     * Sets the X field of view.
     * @param inXFOV the X field of view
     */
    public void setXFOV(double inXFOV) {
        xFOV = inXFOV;
        setFOVs(xFOV, yFOV);
    }
    
    /**
     * Returns the X field of view.
     * @return the X field of view
     */
    public double getXFOV() {
        return xFOV;
    }
    
    /**
     * Sets the Y field of view
     * @param inYFOV the Y field of view
     */
    public void setYFOV(double inYFOV) {
        yFOV = inYFOV;
        setFOVs(xFOV, yFOV);
    }
    
    /**
     * Returns the Y field of view
     * @return the Y field of view
     */
    public double getYFOV() {
        return yFOV;
    }
    
    /** Alternate way to set the fields of view.  Sets them to match a virtual screen
     * of specified size at the specified distance from the camera center of projection.
     * @param width width of virtual screen
     * @param height height of virtual screen
     * @param distance distance from camera to virtual screen
     */
    public void setVirtualScreen(double width,double height,double distance) {
        tanXFOV = width*0.5/distance;
        tanYFOV = height*0.5/distance;
        xFOV = Math.atan2(width*0.5,distance);
        yFOV = Math.atan2(height*0.5,distance);
    }
    
    /**
     * Sets the frame of the camera to look from the eye parameter to the
     * target parameter with an up vector given by the up parameter.
     * This method works like glLookAt.
     * @param inEye where the camera is looking from
     * @param inTarget where the camera is looking at
     * @param inUp the up vector
     */
    public void lookAt(Vector3d inEye, Vector3d inTarget, Vector3d inUp) {
        eye.set(inEye);
        target.set(inTarget);
        up.set(inUp);
        computeFrameFromLookatParameters();
    }
    
    private void computeFrameFromLookatParameters() {
        frame.z.sub(target,eye);
        frame.z.normalize();
        frame.y.set(up);
        frame.y.scale(-1);
        frame.x.cross(frame.y,frame.z);
        frame.x.normalize();
        frame.y.cross(frame.z,frame.x);
        frame.origin.set(eye);
    }
    
    public void setEye(Vector3d inEye) {
        eye.set(inEye);
        computeFrameFromLookatParameters();
    }
    
    public void setTarget(Vector3d inTarget) {
        target.set(inTarget);
        computeFrameFromLookatParameters();
    }
    
    public void setUp(Vector3d inUp) {
        up.set(inUp);
        computeFrameFromLookatParameters();
    }

    public void setPixelRaysPerPoint (int n) {
	pixelRaysPerPoint = n;
    }
    public int getPixelRaysPerPoint () {
	return pixelRaysPerPoint;
    }

    public void setPixelRaySamplingMethod (String s) {
    }
    public int getPixelRaySamplingMethod () {
	return pixelRaySamplingMethod;
    }
}


