/*
 * Copyright Kavita Bala, CS 6620, Cornell University
 * Contact kb@cs.cornell.edu
 */


package cs6620.primitive;

import cs6620.geometry.Frame;
import cs6620.geometry.IntersectionRecord;
import cs6620.geometry.Ray;
import javax.vecmath.*;

/**This class defines the Triangle primitive.  Stores barycentric coordinates
 * in PositionSampleRecord.primitiveUV.
 */
public class Triangle extends Primitive {
    protected Vector3d v0;
    protected Vector3d v1;
    protected Vector3d v2;
    //note the origin of this frame is never used
    protected Frame frame;
    protected double	surfaceArea;
    
    /**
     * An empty constructor that can be used by subclasses when they have
     * their own constructors. This should not be made public as it leaves
     * the Triangle uninitialized. Subclasses are responsible for ensuring
     * that a completely initialized Triangle is constructed.
     */
    protected Triangle() {
    }
    
    /**Constructor.
     */
    public Triangle(Vector3d inV0, Vector3d inV1, Vector3d inV2) {
        v0 = new Vector3d(inV0);
        v1 = new Vector3d(inV1);
        v2 = new Vector3d(inV2);
        frame = new Frame();
        initialize();
        frame.renormalizeFromZ();
    }
    
    /**Copy constructor.
     */
    public Triangle(Triangle inTriangle) {
        v0 = new Vector3d(inTriangle.v0);
        v1 = new Vector3d(inTriangle.v1);
        v2 = new Vector3d(inTriangle.v2);
        material = inTriangle.material;
        frame = new Frame();
        frame.setOrigin(v0);
        initialize();
        frame.renormalizeFromZ();
    }
    
    /**initialization routine. called at the bottom of constructors and after
     * transformation. this is private.  Warning: frames z vector is set
     * in direction of surface normal, but frame is invalid until it is
     * renormalized!
     */
    private void initialize() {
        //compute v1 - v0
        double x1 = v1.x - v0.x;
        double y1 = v1.y - v0.y;
        double z1 = v1.z - v0.z;
        //compute v2 - v0
        double x2 = v2.x - v0.x;
        double y2 = v2.y - v0.y;
        double z2 = v2.z - v0.z;
        //compute cross(v1-v0, v2-v0) (unnormalized normal)
        double normx = y1*z2-z1*y2;
        double normy = z1*x2-x1*z2;
        double normz = x1*y2-y1*x2;
        //surface area is equal to 0.5*length of the cross product
        surfaceArea = 0.5*Math.sqrt(normx*normx + normy*normy + normz*normz);
        //set the z axis of the frame to the unormalized normal
        frame.z.set(normx,normy,normz);
    }
    
    
    /**Shamelessly stolen from Dave's raytracer.
     */
    public boolean intersects(Ray inRay, IntersectionRecord outRecord) {
        double G, H, I, J, K, L;
        double EIHF, GFDI, DHEG, AKJB, JCAL, BLKC;
        double inv_denom, x, y, t;
        //these could be precomputed, but then we have to save and transform
        //more state with each triangle
        double A = v0.x - v1.x;
        double B = v0.y - v1.y;
        double C = v0.z - v1.z;
        double D = v0.x - v2.x;
        double E = v0.y - v2.y;
        double F = v0.z - v2.z;
        
        G = inRay.direction.x;
        H = inRay.direction.y;
        I = inRay.direction.z;
        
        EIHF = E*I-H*F;
        GFDI = G*F-D*I;
        DHEG = D*H-E*G;
        
        inv_denom = 1.0 / (A*EIHF + B*GFDI + C*DHEG);
        
        J = v0.x - inRay.origin.x;
        K = v0.y - inRay.origin.y;
        L = v0.z - inRay.origin.z;
        
        x = inv_denom * (J*EIHF + K*GFDI + L*DHEG);
        
        if (x < 0 || x > 1)
            return false;
        
        AKJB = A*K - J*B;
        JCAL = J*C - A*L;
        BLKC = B*L - K*C;
        
        y = inv_denom * (I*AKJB + H*JCAL + G*BLKC);
        if (y < 0 || x + y > 1)
            return false;
        
        t = -inv_denom * (F*AKJB + E*JCAL + D*BLKC);
        boolean inBounds = (t > inRay.tStart)&&(t<inRay.tEnd);
        if ( !inBounds ) //test is done this way so that any NAN values resulting from
            return false;                               //degenerate triangles will fail the test and return false
        
        outRecord.t = t;
        outRecord.primitiveUV.x = x;
        outRecord.primitiveUV.y = y;
        getBasis(outRecord.frame);
        inRay.evaluate(t, outRecord.frame.origin);
        outRecord.primitive = this;
        return true;
    }
    
    
    /**
     */
    public void getBasis(Frame outFrame) {
        outFrame.x.set(frame.x);
        outFrame.y.set(frame.y);
        outFrame.z.set(frame.z);
    }
    
    /**
     */
    public boolean getFrame(Frame outFrame) {
        outFrame.set(frame);
        return true;
    }
    
    /**returns the i'th vertex of the triangle.
     * @param i the required vertex, 0 1 or 2
     */
    public Vector3d getVertex(int i) {
        switch(i) {
            case 0:
                return v0;
            case 1:
                return v1;
            case 2:
                return v2;
            default:
                throw new Error("Bad vertex " + i + " passed to getVertex");
        }
    }
    
    /**get the edges as rays.
     * @return an array of rays of size 3, v0->v1, v1->v2 and v2->v0 are its contents.
     */
    private Ray [] getEdgeRays() {
        Ray [] edges = new Ray[3];
        edges[0] = new Ray(v0,v1);
        edges[1] = new Ray(v1,v2);
        edges[2] = new Ray(v2,v0);
        return edges;
    }
    
    /**toString function.
     * @see System.out
     */
    public String toString() {
        return "Triangle:\n"+v0+"\n"+v1+"\n"+v2+"\n";
    }
    
    /**get vertex 0
     */
    public Vector3d getVertex0() {
        return v0;
    }
    
    /**get vertex 1
     */
    public Vector3d getVertex1() {
        return v1;
    }
    
    /**get vertex 2
     */
    public Vector3d getVertex2() {
        return v2;
    }
}


