/*
 * Copyright Kavita Bala, CS 6620, Cornell University
 * Contact kb@cs.cornell.edu
 */


package cs6620.primitive;

import cs6620.geometry.Frame;
import cs6620.geometry.IntersectionRecord;
import cs6620.geometry.Ray;
import javax.vecmath.*;

/**
 * A sphere.
 * Local texture UV coordinates are a stretching of (phi,theta) on the sphere
 * (U = phi/2pi and goes from 0 to 1, V = theta/pi+0.5 and goes from
 * 0 to 1). The sphere has a coordinate frame (accessed
 * through get/setFrame) that serves as both the center of the
 * sphere and as a mapping to/from local UV coordinates to global
 * 3D points. The equator of the sphere (theta = 0) lies in the XZ
 * plane of the coordinate frame. The north pole (theta = pi/2)
 * lies on the positive Y axis.
 */
public class Sphere extends Primitive {
    protected Frame frame;
    protected double radius;
    
    /**Constructor for a unit sphere located at origin.
     */
    public Sphere() {
        frame = new Frame();
        radius = 1.0;
    }
    
    /**Constructor.
     */
    public Sphere(Vector3d c, double r) {
        frame = new Frame();
        frame.setOrigin(c);
        radius = r;
        frame.constructFromZ(Frame.standardZ);
    }
    
    /**@return radius of sphere
     */
    public double getRadius() {
        return radius;
    }
    
    /**sets the radius of this sphere
     * @param inRadius set this.radius to inRadius
     */
    public void setRadius(double inRadius) {
        radius = inRadius;
    }
    
    public Vector3d getCenter() {
        return frame.getOrigin();
    }
    
    /**
     * Sets the center of the sphere to a copy of the input radius.
     * @param inCenter sets center to inCenter
     */
    public void setCenter(Vector3d inCenter) {
        frame.setOrigin(inCenter);
    }
    
    /**returns the frame of this sphere.
     */
    public boolean getFrame(Frame outFrame) {
        outFrame.set(frame);
        return true;
    }
    
    /**sets the frame to inFrame.
     * @param inFrame set frame to inFrame
     */
    public void setFrame(Frame inFrame) {
        frame.set(inFrame);
    }
    
    
    /**
     * Assumes (x,y,z) form a normalized vector in local space and places an orthonormal
     * basis and origin based at that direction on the sphere in outFrame in world space.
     */
    private void getFrame(double x, double y, double z, Frame outFrame) {
        outFrame.origin.set(x*radius,y*radius,z*radius);
        frame.pointToWorld(outFrame.origin);
        outFrame.z.set(x,y,z);
        outFrame.y.set(0.0,1.0,0.0);
        outFrame.x.cross(outFrame.y,outFrame.z);
        if (Math.abs(outFrame.x.lengthSquared()) > 1.0e-10) {
            outFrame.x.normalize();
            outFrame.y.cross(outFrame.z,outFrame.x);
        } else {
            outFrame.y.set(1.0,0.0,0.0);
            outFrame.x.cross(outFrame.y,outFrame.z);
        }
        frame.vectorToWorld(outFrame.x);
        frame.vectorToWorld(outFrame.y);
        frame.vectorToWorld(outFrame.z);
    }
    
    /**
     */
    public boolean intersects(Ray inRay, IntersectionRecord outRecord) {
        Vector3d d = inRay.direction;
        Vector3d c = frame.origin;
        Vector3d o = inRay.origin;
        double qx = o.x - c.x;
        double qy = o.y - c.y;
        double qz = o.z - c.z;
        double dd = d.lengthSquared();
        double qd = qx*d.x + qy*d.y + qz*d.z;
        double qq = qx*qx + qy*qy + qz*qz;
        
        // solving the quadratic equation for t at the pts of intersection
        // dd*t^2 + (2*qd)*t + (qq-r^2) = 0
        double discriminantsqr = (qd*qd-dd*(qq-radius*radius));
        if (discriminantsqr < 0) {
            return false;
        }
        double discriminant = Math.sqrt(discriminantsqr);
        double t1 = (-qd-discriminant)/dd;
        double t2 = (-qd+discriminant)/dd;
        double t = 0;
        if (t1 > inRay.tStart && t1 < inRay.tEnd) {
            t = t1;
        } else if (t2 > inRay.tStart && t2 < inRay.tEnd) {
            t = t2;
        } else {
            return false;
        }
        //note you are not allowed to alter the intersection record unless you actually intersect the primitive
        Vector3d p = outRecord.frame.origin;
        inRay.evaluate(t,p);
        frame.pointToLocal(p);
        p.normalize();
        getFrame(p.x,p.y,p.z,outRecord.frame);
        outRecord.t = t;
        outRecord.primitive = this;
        outRecord.primitiveUV.set(-1,-1);
        return true;
    }
}




