/*
 * Copyright Kavita Bala, CS 6620, Cornell University
 * Contact kb@cs.cornell.edu
 */

package cs6620.io;

import cs6620.material.RGBColor;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import javax.imageio.ImageIO;

/**
 * An image, with each pixel being represented by an RGBColor.
 * The image is stored in a one dimensional array, in scanline order.
 */
public class RGBColorImage {
    private int width;
    private int height;
    private RGBColor [] pixels;
    
    public RGBColorImage() {}
    
    public RGBColorImage(int inWidth, int inHeight) {
        width = inWidth;
        height = inHeight;
        createPixels();
    }
    
    /**
     * Generates the array of pixels and initializes them to 0. Sets
     * the array of pixels to null if width or height are invalid.
     */
    private void createPixels() {
        if (width >= 0 && height >= 0) {
            pixels = new RGBColor[width*height];
            for (int i = 0; i < pixels.length; i++) {
                pixels[i] = new RGBColor();
            }
        } else {
            pixels = null;
        }
    }
    
    /**
     * Copies the color of the input pixel into the pixel at the given
     * x and y coordinates.
     */
    public void setPixel(int x, int y, RGBColor pixel) {
        getPixel(x,y).set(pixel);
    }
    
    /**
     * Returns a reference to the given pixel
     */
    public RGBColor getPixel(int x, int y) {
        return pixels[y*width+x];
    }
    
    /**
     * Returns a reference to the entire array of pixels
     */
    public RGBColor [] getPixels() {
        return pixels;
    }
    
    /** Getter for property width.
     * @return Value of property width.
     *
     */
    public int getWidth() {
        return width;
    }
    
    /** Setter for property width. Recreates the array of pixels when called.
     * @param width New value of property width.
     *
     */
    public void setWidth(int width) {
        this.width = width;
        createPixels();
    }
    
    /** Getter for property height.
     * @return Value of property height.
     *
     */
    public int getHeight() {
        return height;
    }
    
    /** Setter for property height. Recreates the array of pixels when called.
     * @param height New value of property height.
     *
     */
    public void setHeight(int height) {
        this.height = height;
        createPixels();
    }
    
    /**
     * Generates a byte BGR BufferedImage from this image. BufferedImages
     * are useful for display within AWT/Swing applications and for saving
     * to various image formats. The byte values for the RGB components are
     * generated by multiplying the RGBColor pixels by 256 and converting to
     * integers. The result is clamped to be in [0,255] for each channel.
     * Thus the value 1 in the RGBColor image maps to the brightest possible
     * pixel value and the value 1/256 maps to the dimmest possible non-zero
     * pixel value. Use scale if a different range is desired.
     */
    public BufferedImage convertToBufferedImage() {
        BufferedImage result = new BufferedImage(width,height,BufferedImage.TYPE_3BYTE_BGR);
        int currentIndex = 0;
        RGBColor pixelCopy = new RGBColor();
        for (int y = 0; y < height; y++) {
            for (int x = 0; x < width; x++, currentIndex++) {
                RGBColor pixel = pixels[currentIndex];
                pixelCopy.set(pixel);
                pixelCopy.clamp(0.0,1.0);
                pixelCopy.scale(256);
                int r = (int)(pixelCopy.getR());
                int g = (int)(pixelCopy.getG());
                int b = (int)(pixelCopy.getB());
                r = (r >= 256 ? 255 : r);
                g = (g >= 256 ? 255 : g);
                b = (b >= 256 ? 255 : b);
                int packed = (r << 16) | (g << 8) | (b << 0);
                result.setRGB(x,y,packed);
            }
        }
        return result;
    }
    
    /**
     * Multiplies the entire image (all three channels) by a single scalar value.
     * Useful when the image is too bright or too dim.
     */
    public void scale(double scale) {
        for (int i = 0; i < pixels.length; i++) {
            pixels[i].scale(scale);
        }
    }
  
    /**
     * Saves this image to a file with the given filename in PNG format.
     * The image is first converted to a BufferedImage
     */
    public void save(String filename) throws IOException {
        BufferedImage bufImg = convertToBufferedImage();
        ImageIO.write(bufImg,"png",new File(filename));
    }
}
