/*
 * Copyright Kavita Bala, CS 6620, Cornell University
 * Contact kb@cs.cornell.edu
 */


package cs6620.io;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.StringTokenizer;
import javax.vecmath.Tuple3d;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/** Reads in XML files and constructs Java objects. Starting with a root Java object
 * and a root XML node, the node's child nodes are analyzed. A node with a name
 * of xxx (as in <xxx>) is turned into a call to the setXxx method of the parent
 * object. An object is created of type equal to the type that the setXxx method
 * takes as a parameter. The children of the <xxx> (such as <yyy>) node are then
 * analyzed and converted into setYyy methods on the new object.
 *
 * As a base case, the parser will generate integer arrays, double arrays, Strings,
 * and subclasses of javax.vecmath.Tuple3D.
 *
 * XML nodes can have 3 attributes. The "name" attribute will name the
 * resulting object. Named objects can be obtained with the "ref" attribute. The
 * "type" attribute will generate an object of the given type instead of the type
 * specified by the setter method. The given type must be a subclass or interface
 * of the type of the argument to the setter method.
 *
 * If a class does not have a setter method with the specified name (setXxx for
 * an XML node named <xxx>) the parser will look for a method named addXxx before
 * reporting an error.
 *
 * Example:
 * The xml file
 *
 * <task>
 * <camera>
 *  <up> 0 1 0 </up>
 *  <target> 0 0 0 </target>
 *  <xFOV> 0.5 </xFOV>
 * </camera>
 * </task>
 *
 * Will inspect the RenderingTask class (assuming the parser has been called with
 * this class as the root class). It will then look for a setCamera method and
 * find that it takes an object of type cs665.scene.Camera. A new instance of this
 * class will be created and its fields will be inspected recursively. A method
 * named setUp will be found in the Camera class which takes a Vector3d. 0.0 2.0
 * 5.0 will then be parsed to construct a new Vector3d and setUp will be called
 * with this new object. Thus the XML file will generate the following calls
 * <CODE>
 * RenderingTask t = new RenderingTask();
 * Camera c = new Camera();
 * c.setUp(new Vector3d(0,1,0));
 * c.setTarget(new Vector3d(0,0,0));
 * c.setXFOV(0.5);
 * t.setCamera(c);
 * </CODE>
 */
public class Parser {
    private DocumentBuilder db;
    
    private HashMap references = new HashMap();
    /** Creates a new Parser. */    
    public Parser() throws ParserConfigurationException {
        db = DocumentBuilderFactory.newInstance().newDocumentBuilder();
    }
    
    
    /** Parses the String and generates either an Integer object or a Double object,
     * depending on the given class.
     * @param c the class to parse the string as
     *
     * @param text the text to parse
     * @return a new object of the given type
     */    
    private Object parsePrimitive(Class c, String text) {
        if (c == Integer.TYPE) {
            return new Integer(text);
        } else if (c == Double.TYPE) {
            return new Double(text);
        } else {
            throw new Error("Cannot parse primitive of type "+c);
        }
    }
    
    private ArrayList parseArray(Class componentType, String text) {
        StringTokenizer t = new StringTokenizer(text);
        ArrayList result = new ArrayList();
        while (t.hasMoreTokens()) {
            String token = t.nextToken();
            result.add(parsePrimitive(componentType,token));
        }
        return result;
    }
    
    private Object parseObject(Class c, String text) throws InstantiationException, IllegalAccessException {
        if (c == String.class) {
            return text;
        } else if (c == Integer.class) {
            return new Integer(text);
        } else if (c == Double.class) {
            return new Double(text);
        } else if ((c.isArray() && c.getComponentType().isPrimitive())) {
            ArrayList tempArray = parseArray(c.getComponentType(), text);
            Object result = Array.newInstance(c.getComponentType(),tempArray.size());
            for (int i = 0; i < tempArray.size(); i++) {
                Array.set(result,i,tempArray.get(i));
            }
            return result;
        } else if (Tuple3d.class.isAssignableFrom(c)) {
            ArrayList tempArray = parseArray(Double.TYPE, text);
            if (tempArray.size() != 3) {
                throw new Error("Tuple3d is not of length 3 ("+tempArray.size()+")");
            }
            Tuple3d result = (Tuple3d)c.newInstance();
            result.x = ((Double)tempArray.get(0)).doubleValue();
            result.y = ((Double)tempArray.get(1)).doubleValue();
            result.z = ((Double)tempArray.get(2)).doubleValue();
            return result;
        } else {
            throw new Error("Cannot parse type "+c);
        }
    }
    
    private Object parseObject(Class c, Node n) throws Exception {
        try {
            Object resultingObject = null;
            NamedNodeMap attributes = n.getAttributes();
            Node nameAttribute = attributes.getNamedItem("name");
            Node typeAttribute = attributes.getNamedItem("type");
            Node refAttribute = attributes.getNamedItem("ref");
            
            if (typeAttribute != null) {
                String className = typeAttribute.getNodeValue();
                try {
                    Class possibleClass = Class.forName(className);
                    if (c.isAssignableFrom(possibleClass)) {
                        c = possibleClass;
                    } else {
                        throw new Error("Type " +className+" does not extend or implement "+c.getName());
                    }
                }
                catch (ClassNotFoundException e) {
                    throw new Error("Class could not  be found: "+className);
                }
            }
            
            if (c.isArray() && !c.getComponentType().isPrimitive()) {
                throw new Error("Cannot parse arrays of non-primitive types");
            }
            
            NodeList children = n.getChildNodes();
            
            if (refAttribute != null) {
                String name = refAttribute.getNodeValue();
                resultingObject = references.get(name);
                if (resultingObject == null) {
                    throw new Error("Unresolved reference: "+name);
                }
            } else if ((c.isArray() && c.getComponentType().isPrimitive()) ||
            c == String.class ||
            c == Integer.class ||
            c == Double.class ||
            Tuple3d.class.isAssignableFrom(c)) {
                for (int i = 0; i < children.getLength(); i++) {
                    Node child = children.item(i);
                    if (child.getNodeType() == Node.TEXT_NODE) {
                        resultingObject = parseObject(c, child.getNodeValue());
                    } else {
                        throw new Error("Found a non-text node while trying to parse a "+c.getName());
                    }
                }
            } else {
                resultingObject = c.newInstance();
                
                // Iterate through XML children elements, setting properties of the
                // object based on the recursively parsed value of the child elements
                for (int i = 0; i < children.getLength(); i++) {
                    Node child = children.item(i);
                    
                    if (child.getNodeType() != Node.ELEMENT_NODE) {
                        continue;
                    }
                    
                    // Turn XML Node name into setter method name
                    String childName = child.getNodeName();
                    StringBuffer methodNameBuffer = new StringBuffer(childName);
                    methodNameBuffer.setCharAt(0, Character.toUpperCase(methodNameBuffer.charAt(0)));
                    String methodName = methodNameBuffer.insert(0,"set").toString();
                    
                    // Look for setXXX method
                    Method [] methods = c.getMethods();
                    Method foundMethod = null;
                    for (int j = 0; j < methods.length && foundMethod == null; j++) {
                        Method m = methods[j];
                        if (methodName.equals(m.getName())) {
                            foundMethod = m;
                        }
                    }
                    
                    // If can't find setXXX method, look for addXXX method instead
                    if (foundMethod == null) {
                        methodName = methodNameBuffer.replace(0,3, "add").toString();
                        for (int j = 0; j < methods.length && foundMethod == null; j++) {
                            Method m = methods[j];
                            if (methodName.equals(m.getName())) {
                                foundMethod = m;
                            }
                        }
                    }
                    
                    if (foundMethod == null) {
                        throw new Error("Must have set or add method for property "+childName+" in class "+c.getName());
                    }
                    Class [] parameterTypes = foundMethod.getParameterTypes();
                    if (parameterTypes.length != 1) {
                        throw new Error("Method "+methodName+" must take exactly one parameter");
                    }
                    Class parameterType = parameterTypes[0];
                    
                    // If the type is primitive, switch to corresponding Object type
                    // to parse. Method invocation will automatically take care of
                    // converting Object types back into primitives.
                    if (parameterType.isPrimitive()) {
                        if (parameterType == Integer.TYPE) {
                            parameterType = Integer.class;
                        } else if (parameterType == Float.TYPE) {
                            parameterType = Float.class;
                        } else if (parameterType == Double.TYPE) {
                            parameterType = Double.class;
                        } else {
                            throw new Error("Cannot parse primitives of type "+parameterType);
                        }
                    }
                    
                    // Recursively parse value of child element
                    Object childValue = parseObject(parameterType, child);
                    
                    // Call the setter method with the parsed value;
                    foundMethod.invoke(resultingObject, new Object [] {childValue});
                }
            }
            
            if (nameAttribute != null) {
                String name = nameAttribute.getNodeValue();
                references.put(name,resultingObject);
            }
            return resultingObject;
        }
        catch (Error e) {
            System.out.println("Error while parsing node "+n.getNodeName());
            throw e;
        }
        catch (Exception e) {
            System.out.println("Exception while parsing node "+n.getNodeName());
            throw e;
        }
    }
    
    /** Parses a given file to generate an object of the given class. See documentation
     * at the top of this class for usage.
     * @param filename the name of the XML file to parse
     * @param c the class of the object to parse
     * @return a new object of the given class
     */    
    public Object parse(String filename, Class c) throws SAXException, IOException {
        Document doc = db.parse(new File(filename));
        Element root = doc.getDocumentElement();
        Object result = null;
        try {
            result = parseObject(c,root);
        }
        catch (Error e) {
            System.out.println("Parsing error:"+e);
        }
        catch (Exception e) {
            System.out.println("Exception occurred while parsing:"+e);
        }
        return result;
        
    }
    
}

