/*
 * Copyright Kavita Bala, CS 6620, Cornell University
 * Contact kb@cs.cornell.edu
 */


package cs6620.geometry;

import javax.vecmath.*;

/**
 * A line segment represented by an origin, a direction and two
 * parameters. The points in the line segment are those in
 * origin + t*direction, where t is in [tStart,tEnd]. The direction
 * is NOT NECESSARILY normalized.
 */
public final class Ray {
    public static final double EPSILON=1e-5;
    /** The origin of the ray coordinate system. Note that this is <b>not</b> the start
     * of the ray.
     */
    public final Vector3d origin =new Vector3d();
    /** The direction of the ray. Note that this is not necessarily normalized.
     */
    public final Vector3d direction =new Vector3d();
    /** The start of the ray in ray parameter space.
     */
    public double tStart;
    /** The end of the "ray" in ray parameter space.
     */
    public double tEnd;
    
    /** Creates a ray with the origin at the center and no defined direction. The
     * start parameter is 0 and the end parameter is "infinite".
     */
    public Ray() {
        tStart = 0;
        tEnd = Double.MAX_VALUE;
    }
    
    /** Creates a segment between two points. The origin is set to the first point, the
     * direction to the difference between the points, and the start and end parameters
     * to 0 and 1 respectively.
     * @param inStart the start point
     * @param inEnd the end point
     */
    public Ray(Vector3d inStart,Vector3d inEnd) {
        origin.set(inStart);
        direction.sub(inEnd,inStart);
        tStart = 0;
        tEnd = 1;
    }
    
    /** Creates a ray with the given parameters. The parameters are copied exactly as
     * they are to the members of the ray.
     * @param inOrigin the origin of the ray coordinate system
     * @param inDirection the direction of the ray
     * @param inStart the start of the ray segment in ray parameter space
     * @param inEnd the end of the ray segment in ray parameter space
     */
    public Ray(Vector3d inOrigin,Vector3d inDirection,double inStart,double inEnd) {
        origin.set(inOrigin);
        direction.set(inDirection);
        tStart = inStart;
        tEnd = inEnd;
    }
    
    /** Creates a copy of the given ray
     * @param inRay the ray to copy
     */
    public Ray(Ray inRay) {
        origin.set(inRay.origin);
        direction.set(inRay.direction);
        tStart = inRay.tStart;
        tEnd = inRay.tEnd;
    }
    
    /** Set the start parameter to be a small offset from the origin and the end
     * parameter to be at "infinity".
     */
    public void makeOffsetRay() {
        tStart = EPSILON;
        tEnd = Double.MAX_VALUE;
    }
    
    /** Set the start parameter to a small offset from the origin and the end parameter
     * to a small offset before 1.
     */
    public void makeOffsetSegment() {
        tStart = EPSILON;
        tEnd = 1.0-EPSILON;
    }
    
    /** Set the start parameters to a small offset from the origin and the end to
     * a small offset before the passed param
     * @param inEnd end of the ray
     */
    public void makeOffsetSegment(double inEnd) {
        tStart = EPSILON;
        tEnd = inEnd - EPSILON;
    }
    
    /** Sets the start of the ray to be at the start point and the end of the ray to
     * be at the end point.
     * @param inStart where the ray should start
     * @param inEnd where the ray should end
     */
    public void makeSegment(Vector3d inStart,Vector3d inEnd) {
        origin.set(inStart);
        direction.sub(inEnd,inStart);
        tStart = 0;
        tEnd = 1;
    }
    
    /** Sets this ray to an exact copy of the input ray.
     * @param inRay the ray to copy
     */
    public void set(Ray inRay) {
        tStart = inRay.tStart;
        tEnd = inRay.tEnd;
        origin.set(inRay.origin);
        direction.set(inRay.direction);
    }
    
    /** Given a parameter t, returns a point origin + t*direction. Note that this point
     * may or may not lie on the ray, depending on whether t is in [tStart,tEnd]
     * @param inT a parameter in ray parameter space
     * @return a point colinear with the ray
     */
    public Vector3d evaluate(double inT) {
        Vector3d p = new Vector3d();
        p.scaleAdd(inT,direction,origin);
        return p;
    }
    
    /** Given a parameter t, returns a point origin + t*direction. Note that this point
     * may or may not lie on the ray, depending on whether t is in [tStart,tEnd]
     * @param inT the parameter in ray parameter space
     * @param outPoint a point colinear with the ray
     */
    public void evaluate(double inT,Vector3d outPoint) {
        outPoint.scaleAdd(inT,direction,origin);
    }
    
    /** Returns the x-coordinate of the start of the ray.
     * @return the x-coordinate of the start of the ray
     */
    public final double getStartX() {
        return origin.x + tStart*direction.x;
    }
    
    /** Returns the y-coordinate of the start of the ray
     * @return the y-coordinate of the start of the ray
     */
    public final double getStartY() {
        return origin.y + tStart*direction.y;
    }
    
    /** Returns the z-coordinate of the start of the ray.
     * @return the z-coordinate of the start of the ray
     */
    public final double getStartZ() {
        return origin.z + tStart*direction.z;
    }
    
    /** Returns the x-coordinate of the end of the ray.
     * @return the x-coordinate of the end of the ray
     */
    public final double getEndX() {
        return origin.x + tEnd*direction.x;
    }
    
    /** Returns the y-coordinate of the end of the ray.
     * @return the y-coordinate of the end of the ray
     */
    public final double getEndY() {
        return origin.y + tEnd*direction.y;
    }
    
    /** Returns the z-coordinate of the end of the ray.
     * @return the z-coordinate of the end of the ray
     */
    public final double getEndZ() {
        return origin.z + tEnd*direction.z;
    }
    
    /** Returns a point at the start of the ray.
     * @return a point at the start of the ray
     */
    public Vector3d getStart() {
        return evaluate(tStart);
    }
    
    /** Returns a point at the end of the ray.
     * @return a point at the end of the ray
     */
    public Vector3d getEnd() {
        return evaluate(tEnd);
    }
    
    /** Creates a string representation of the ray for debugging purposes.
     * @return a string representation of the ray
     */
    public String toString() {
        return "origin:"+origin+"\ndirection:"+direction+"\ntS:"+tStart+"\ntE:"+tEnd;
    }
}

