/*
 * Copyright Kavita Bala, CS 6620, Cornell University
 * Contact kb@cs.cornell.edu
 */

package cs6620.geometry;

import javax.vecmath.*;

/** Represents an orthonormal basis with an origin. This class has methods that
 * operate on points and some methods that operate on vectors. The operations
 * are nearly identical, the only difference being that points are translated
 * by the frame origin while vectors are not.
 */
public class Frame {
    /** The standard x axis (1,0,0).
     *
     */
    public static final Vector3d standardX =new Vector3d(1,0,0);
    /** The standard y axis, (0,1,0).
     *
     */
    public static final Vector3d standardY =new Vector3d(0,1,0);
    /** The standard z axis, (0,0,1).
     */
    public static final Vector3d standardZ =new Vector3d(0,0,1);
    /** The standard origin, (0,0,0).
     */
    public static final Vector3d standardOrigin =new Vector3d(0,0,0);
    
    /** The x axis of this frame. This vector is always normalized and orthogonal to the
     * y and z axes.
     */
    public final Vector3d x =new Vector3d();
    /** The y axis of this frame. This vector is always normalized and orthogonal to the
     * x and z axes.
     */
    public final Vector3d y =new Vector3d();
    /** The z axis of this frame. This vector is always normalized and orthogonal to the
     * x and y axes.
     */
    public final Vector3d z =new Vector3d();
    /** The origin of this frame.
     */
    public final Vector3d origin =new Vector3d();
    
    /**
     * A default frame of reference with standard x,y,z axes and
     * standard origin.
     */
    public Frame() {
        x.set(standardX);
        y.set(standardY);
        z.set(standardZ);
        origin.set(standardOrigin);
    }
    
    /** A copy of the input frame.
     * @param inFrame the frame to copy.
     */
    public Frame(Frame inFrame) {
        x.set(inFrame.x);
        y.set(inFrame.y);
        z.set(inFrame.z);
        origin.set(inFrame.origin);
    }
    
    /** Creates a frame out of three vectors and a point. These vectors should form an
     * orthonormal basis. Whether they do or not is not checked.
     * @param inX the x axis (should be normalized)
     * @param inY the y axis (should be normalized)
     * @param inZ the z axis (should be normalized)
     * @param inOrigin the origin
     */
    public Frame(Vector3d inX,Vector3d inY,Vector3d inZ,Vector3d inOrigin) {
        x.set(inX);
        y.set(inY);
        z.set(inZ);
        origin.set(inOrigin);
    }
    
    /** Copy the frame.
     * @param inFrame the frame to copy
     */
    public void set(Frame inFrame) {
        x.set(inFrame.x);
        y.set(inFrame.y);
        z.set(inFrame.z);
        origin.set(inFrame.origin);
    }
    
    /** Sets the axes of this frame. The input vectors should already be normalized and
     * form an orthonormal basis.
     * @param inX the x axis
     * @param inY the y axis
     * @param inZ the z axis
     */
    public void set(Vector3d inX,Vector3d inY,Vector3d inZ) {
        x.set(inX);
        y.set(inY);
        z.set(inZ);
    }
    
    /** Sets the axes and the origin of this frame. The axes should form an orthonormal
     * basis.
     * @param inX the x axis
     * @param inY the y axis
     * @param inZ the z axis
     * @param inOrigin the origin
     */
    public void set(Vector3d inX,Vector3d inY,Vector3d inZ,Vector3d inOrigin) {
        x.set(inX);
        y.set(inY);
        z.set(inZ);
        origin.set(inOrigin);
    }
    
    /** Checks origin and x, y, and z axes for exact equality
     * @param inCompare Frame to compare to
     * @return true if frames are exactly equal because their origins and axes are numerically identical.
     */
    public boolean equals(Frame inCompare) {
        return origin.equals(inCompare.origin) && x.equals(inCompare.x) && y.equals(inCompare.y) && z.equals(inCompare.z);
    }
    
    /** Takes a vector in the frame's coordinate system and transforms
     * it into world coordinates. Since this is a vector, the coordinates
     * will <b>not</b> be translated according to the frame's origin.
     * @param modVector the vector to transform
     */
    public void vectorToWorld(Vector3d modVector) {
        // Must copy original coordinates because we're changing
        // vector in place.
        double vx = modVector.x;
        double vy = modVector.y;
        double vz = modVector.z;
        modVector.scale(vx,x);
        modVector.scaleAdd(vy,y,modVector);
        modVector.scaleAdd(vz,z,modVector);
    }
    
    /** Takes the x, y, z coordinates of a vector in local coordinates
     * and returns its world coordinates in the outPoint parameter.
     * @param inX the x coordinate of the vector
     * @param inY the y coordinate of the vector
     * @param inZ the z coordinate of the vector
     * @param outVector the result of the transformation
     */
    public void vectorToWorld(double inX, double inY, double inZ, Vector3d outVector) {
        outVector.set(0,0,0);
        outVector.scaleAdd(inX,x,outVector);
        outVector.scaleAdd(inY,y,outVector);
        outVector.scaleAdd(inZ,z,outVector);
    }
    
    /** Takes the x, y, z coordinates of a point in local coordinates
     * and returns its world coordinates in the outPoint parameter. Since this is
     * a point, it will be translated by the origin of this frame.
     * @param inX the x coordinate of the point
     * @param inY the y coordinate of the point
     * @param inZ the z coordinate of the point
     * @param outPoint a point in world coordinates
     */
    public void pointToWorld(double inX, double inY, double inZ, Vector3d outPoint) {
        outPoint.set(origin);
        outPoint.scaleAdd(inX,x,outPoint);
        outPoint.scaleAdd(inY,y,outPoint);
        outPoint.scaleAdd(inZ,z,outPoint);
    }
    
    /** Converts a point in local coordinates to a point in world coordinates. Since
     * the parameter is a point, it will be affected by the origin of this frame.
     * @param inPoint a point in local coordinates
     * @param outPoint the result of the transformation: a point in world coordinates
     */
    public void pointToWorld(Vector3d inPoint,Vector3d outPoint) {
        outPoint.set(origin);
        outPoint.scaleAdd(inPoint.x,x,outPoint);
        outPoint.scaleAdd(inPoint.y,y,outPoint);
        outPoint.scaleAdd(inPoint.z,z,outPoint);
    }
    
    /** Converts a point in local coordinates to a point in world coordinates. Since
     * the parameter is a point, it will be affected by the frame's origin.
     * @param modPoint the point to modify as well as the result of that modification
     */
    public void pointToWorld(Vector3d modPoint) {
        double x = modPoint.x;
        double y = modPoint.y;
        double z = modPoint.z;
        modPoint.set(origin);
        modPoint.scaleAdd(x,this.x,modPoint);
        modPoint.scaleAdd(y,this.y,modPoint);
        modPoint.scaleAdd(z,this.z,modPoint);
    }
    
    /** Transforms a point in world coordinates to a point in local coordinates.
     * @param modPoint the point to transform, as well as the result of that transformation
     */
    public void pointToLocal(Vector3d modPoint) {
        modPoint.sub(this.origin);
        vectorToLocal(modPoint);
    }
    
    /** Transforms a point in world coordinates to a point in local coordinates. Since
     * the parameter is a point, the point will be affected by the origin of this frame.
     * @param inPoint the point to transform (world coordinates)
     *
     * @param outPoint the transformed point (local coordinates)
     */
    public void pointToLocal(Vector3d inPoint,Vector3d outPoint) {
        outPoint.sub(inPoint,origin);
        vectorToLocal(outPoint);
    }
    
    /** Transform a vector in world coordinates to local coordinates. Since this is a
     * vector transformation, it will not be affected by the origin of this frame.
     * @param modVector the vector to transform as well as the result of the transformation
     */
    public void vectorToLocal(Vector3d modVector) {
        double x = this.x.dot(modVector);
        double y = this.y.dot(modVector);
        double z = this.z.dot(modVector);
        modVector.x = x;
        modVector.y = y;
        modVector.z = z;
    }
    
    /** Transform a vector in world coordinates to local coordinates. Since this is a
     * vector transformation, it will not be affected by the origin of this frame.
     * @param inVector the vector to transform
     * @param outVector the result of the transformation
     */
    public void vectorToLocal(Vector3d inVector, Vector3d outVector) {
        double x = this.x.dot(inVector);
        double y = this.y.dot(inVector);
        double z = this.z.dot(inVector);
        outVector.x = x;
        outVector.y = y;
        outVector.z = z;
    }
    
    /** Copies the coordinates of the input parameter into the origin of
     * this coordinate frame.
     * @param inPoint the new origin
     */
    public void setOrigin(Vector3d inPoint) {
        origin.set(inPoint);
    }
    
    /** Returns the origin of this frame. Note that this is not a copy and changing the
     * value of this vector will change the origin of the frame.
     * @return the origin of the frame.
     */
    public Vector3d getOrigin() {
        return origin;
    }
    
    /** Gets the origin of this frame and copies it onto the given parameter.
     * @param outPoint the origin of this frame
     */
    public void getOrigin(Vector3d outPoint) {
        outPoint.set(origin);
    }
    
    /** Constructs an orthonormal basis from two vectors.
     * The Z vector is copied, normalized, and used as the Z axis.
     * The Y vector is used to construct the X axis.
     * @param inZ the direction of the z axis for the new frame
     * @param inY a vector that is orthogonal to the desired x axis and that is not colinear with
     * the given z direction
     */
    public void constructFromZY(Vector3d inZ, Vector3d inY) {
        z.set(inZ);
        z.normalize();
        x.cross(inY,inZ);
        x.normalize();
        y.cross(z,x);
    }
    /** Renormalizes and orthogonalizes the frame based on the current Z and
       Y axes.  The Z axis is just normalized.  It is used along with the
       current Y axis to find an orthonormal X axis and then the y axis is
       orthogonalized and renormalized.
     */
    public void renormalizeFromZY() {
        z.normalize();
        x.cross(y,z);
        x.normalize();
        y.cross(z,x);
    }
    /** Constructs an orthonormal basis from two vectors.
     * The Z vector is copied, normalized, and used as the Z axis.
     * The X vector is used to construct the Y axis.
     * @param inZ the desired direction of the z axis for the new frame
     * @param inX a vector that is orthogonal to the desired y axis and is not colinear with the
     * given z axis
     */
    public void constructFromZX(Vector3d inZ, Vector3d inX) {
        z.set(inZ);
        z.normalize();
        y.cross(inZ,inX);
        y.normalize();
        x.cross(y,z);
    }
    
    /** Construct an orthonormal basis from one vector.
     * The input vector is copied, normalized, and used as the Z axis.
     * Two vectors that are orthonormal to each other and to Z
     * are then found.
     * @param inZ the desired direction of the z axis of the new frame
     */
    public void constructFromZ(Vector3d inZ) {
        z.set(inZ);
        renormalizeFromZ();
    }
    /** Renormalize and orthogonalize an orthonormal basis based on the
      current value of z.  The current Z axis is normalized and two
      orthonormal vectors (to Z and each other) are put in X and Y
     */
    public void renormalizeFromZ() {
        z.normalize();
        double zDotSX = z.dot(standardX);
        zDotSX = (zDotSX < 0 ? -zDotSX : zDotSX)-1;
        if (Math.abs(zDotSX) < 1e-3) {
            x.set(standardY);
        } else {
            x.set(standardX);
        }
        y.cross(z,x);
        y.normalize();
        x.cross(y,z);
    }
    
    /** Given the local cartesian coordinates of a vector, transform it to world
     * cartesian coordinates.
     * @param inX the x coordinate of the vector
     * @param inY the y coordinate of the vector
     * @param inZ the z coordinate of the vector
     * @param outVector the resulting world cartesian coordinates
     */
    public void cartesianVectorToWorld(double inX,double inY,double inZ,Vector3d outVector) {
        vectorToWorld(inX,inY,inZ,outVector);
    }
    
    /** Given the local cylindrical coordinates of a vector, transform it to world
     * cartesian coordinates.
     * @param inR the cylindrical radius
     * @param inPhi the angle
     * @param inZ the height
     * @param outVector the resulting world-coordinate cartesian vector
     */
    public void cylindricalVectorToWorld(double inR,double inPhi,double inZ,Vector3d outVector) {
        vectorToWorld(inR*Math.cos(inPhi), inR*Math.sin(inPhi), inZ, outVector);
    }
    
    /** Given the local spherical coordinates of a vector, transform it to world
     * cartesian coordinates.
     * @param inR the spherical radius
     * @param inTheta the theta angle
     * @param inPhi the phi angle
     * @param outVector the resulting world cartesian coordinates
     */
    public void sphericalVectorToWorld(double inR,double inTheta,double inPhi,Vector3d outVector) {
        vectorToWorld(inR*Math.sin(inTheta)*Math.cos(inPhi),
        inR*Math.sin(inTheta)*Math.sin(inPhi),
        inR*Math.cos(inTheta),
        outVector);
    }
    
    /** Given a point in local cartesian coordinates, transform is to world cartesian
     * coordinates.
     * @param inX the x coordinate of the point
     * @param inY the y coordinate of the point
     * @param inZ the z coordinate of the point
     * @param outPoint the transformed point in world cartesian coordinates
     */
    public void cartesianPointToWorld(double inX,double inY,double inZ,Vector3d outPoint) {
        cartesianVectorToWorld(inX,inY,inZ,outPoint);
        outPoint.add(origin);
    }
    
    /** Given the local cylindrical coordinates of a point, transform it to world
     * cartesian coordinates.
     * @param inR the cylindrical radius
     * @param inPhi the phi angle
     * @param inZ the height
     * @param outPoint the resulting point in world cartesian coordinates
     */
    public void cylindricalPointToWorld(double inR,double inPhi,double inZ,Vector3d outPoint) {
        cylindricalVectorToWorld(inR,inPhi,inZ,outPoint);
        outPoint.add(origin);
    }
    
    /** Given the local spherical coordinates of a point, transform it to world
     * cartesian coordinates.
     * @param inR the spherical radius
     * @param inTheta the theta angle
     * @param inPhi the phi angle
     * @param outPoint the resulting point in world cartesian coordinates
     */
    public void sphericalPointToWorld(double inR,double inTheta,double inPhi,Vector3d outPoint) {
        sphericalVectorToWorld(inR,inTheta,inPhi,outPoint);
        outPoint.add(origin);
    }
    
    /**
     * Get the specular direction from the normal and the incoming one.
     * @param outExiting Exiting reflected direction
     * @param inIncoming Incoming direction (points towards the surface).
     */
    public void reflectedDirection(Vector3d outExiting, Vector3d inIncoming) {
        outExiting.scaleAdd(-2*inIncoming.dot(z), z, inIncoming);
    }
    
    /** Returns a string representation of this frame for debugging purposes.
     * @return a string representation of this frame
     */
    public String toString() {
        String s = x.toString() + "\n";
        s += y.toString() + "\n";
        s += z.toString() + "\n";
        s += origin.toString() + "\n";
        return s;
    }
}
