/*
 * Copyright Kavita Bala, CS 6620, Cornell University
 * Contact kb@cs.cornell.edu
 */


package cs6620.geometry;

import javax.vecmath.*;
import java.util.*;

/** Represents an axis aligned box in three-space. The representation is made public
 * through the minPoint and maxPoint field. These represent the minimum and maximum
 * coordinates for the box. For example, the box [1,5]x[2,3]x[-2,4] would be
 * represented by minPoint=(1,2,-2) and maxPoint = (5,3,4).
 */
public final class AxisAlignedBox  {
  /** A vector containing the minimum coordinates in each of the three dimensions.
   */
  public final Vector3d minPoint =new Vector3d();

  /** A vector containing the maximum coordinate in each of the three dimensions.
   */
  public final Vector3d maxPoint =new Vector3d();

  /** Creates a new box with the given minimum and maximum coordinates.
   * @param inMin the minimum coordinates for this box
   * @param inMax the maximum coordinates for this box
   */
  public AxisAlignedBox(Vector3d inMin,Vector3d inMax) {
    minPoint.set(inMin);
    maxPoint.set(inMax);
  }

  /** Creates a new box which is an exact copy of the given box.
   * @param inBox the box to make a copy of
   */
  public AxisAlignedBox(AxisAlignedBox inBox) {
    minPoint.set(inBox.minPoint);
    maxPoint.set(inBox.maxPoint);
  }

  /** Generates a degenerate box, with min and max points at max and
   * min infinity. Useful as an initial box when generating bounding
   * volumes. For example, one would create a degenerate box and then
   * add points to it so that the box would be the tightest box containing
   * the added points.
   */
  public AxisAlignedBox() {
    reset();
  }

  /** Reset bounding box to default empty values */
  public void reset() {
    minPoint.set(Double.MAX_VALUE,Double.MAX_VALUE,Double.MAX_VALUE);
    maxPoint.set(-Double.MAX_VALUE,-Double.MAX_VALUE,-Double.MAX_VALUE);
  }
  /** Set bounding box to given min and max values */
  public void set(Vector3d inMin,Vector3d inMax) {
    minPoint.set(inMin);
    maxPoint.set(inMax);
  }

  /** Makes the box large enough to contain its current size and the
   * new point.
   * @param inPoint a point that should fit in the box
   */
  public void add(Vector3d inPoint) {
    minPoint.set(Math.min(minPoint.x,inPoint.x),
    Math.min(minPoint.y,inPoint.y),
    Math.min(minPoint.z,inPoint.z));
    maxPoint.set(Math.max(maxPoint.x,inPoint.x),
    Math.max(maxPoint.y,inPoint.y),
    Math.max(maxPoint.z,inPoint.z));
  }
  
  /** Makes the box large enough to contain its current size and the
   * new point specified by its three coordinates.
   */
  public void addPoint(double inX, double inY, double inZ) {
    minPoint.set(Math.min(minPoint.x,inX),Math.min(minPoint.y,inY),Math.min(minPoint.z,inZ));
    maxPoint.set(Math.max(maxPoint.x,inX),Math.max(maxPoint.y,inY),Math.max(maxPoint.z,inZ));
  }
    

  /** Makes the box large enough to contain its current volume and the
   * volume of the provided box.
   * @param inBox the box whose volume should be added to this box
   */
  public void add(AxisAlignedBox inBox) {
    add(inBox.minPoint);
    add(inBox.maxPoint);
  }

  /** Set this box to the intersection of itself and the specified box */
  public void intersect(AxisAlignedBox inBox) {
    minPoint.x = Math.max(minPoint.x,inBox.minPoint.x);
    minPoint.y = Math.max(minPoint.y,inBox.minPoint.y);
    minPoint.z = Math.max(minPoint.z,inBox.minPoint.z);
    maxPoint.x = Math.min(maxPoint.x,inBox.maxPoint.x);
    maxPoint.y = Math.min(maxPoint.y,inBox.maxPoint.y);
    maxPoint.z = Math.min(maxPoint.z,inBox.maxPoint.z);
  }
  
  /** Returns true iff ray intersects this box.
   * @return true iff the ray intersects this box.
   * @param inRay the ray to check for intersection. This parameter is
   * changed temporarily while within this method, but all
   * values are restored when the method returns.
   */
  public boolean intersects(Ray inRay) {
    double oldTStart = inRay.tStart;
    double oldTEnd = inRay.tEnd;
    if (clipRay(inRay)) {
      // Restore ray parameters (changed by clipRay)
      inRay.tStart = oldTStart;
      inRay.tEnd = oldTEnd;
      return true;
    } else {
      return false;
    }
  }

  /** If the segment intersects this box, then it changes the start and end parameters
   * of the segment so that it is the intersection of the interior of the box
   * and itself, and the method returns true.
   * If the segment does not intersect the box, false is returned and the ray
   * is unchanged.
   * @param modRay the ray to clip against this box
   * @return true iff the ray intersected this box
   */
  public boolean clipRay(Ray modRay) {
    boolean xDirNotZero = (modRay.direction.x!=0);
    boolean yDirNotZero = (modRay.direction.y!=0);
    boolean zDirNotZero = (modRay.direction.z!=0);
    double maxEntry = -Double.MAX_VALUE;
    double minExit = Double.MAX_VALUE;
    if (xDirNotZero) {
      double minXIntersect = (minPoint.x - modRay.origin.x)/modRay.direction.x;
      double maxXIntersect = (maxPoint.x - modRay.origin.x)/modRay.direction.x;
      if (minXIntersect < maxXIntersect) {
        maxEntry = minXIntersect;
        minExit = maxXIntersect;
      } else {
        maxEntry = maxXIntersect;
        minExit = minXIntersect;
      }
    } else {
      if ((modRay.origin.x < minPoint.x)||(modRay.origin.x > maxPoint.x))
      return false;
    }

    if (yDirNotZero) {
      // This can be optimized by factoring out the divisions
      double minYIntersect = (minPoint.y - modRay.origin.y)/modRay.direction.y;
      double maxYIntersect = (maxPoint.y - modRay.origin.y)/modRay.direction.y;
      if (minYIntersect < maxYIntersect) {
        maxEntry = (maxEntry > minYIntersect) ? maxEntry : minYIntersect;
        minExit = (minExit < maxYIntersect) ? minExit : maxYIntersect;
      } else {
        maxEntry = (maxEntry > maxYIntersect) ? maxEntry : maxYIntersect;
        minExit = (minExit < minYIntersect) ? minExit : minYIntersect;
      }
    } else {
      if ((modRay.origin.y < minPoint.y)||(modRay.origin.y > maxPoint.y))
      return false;
    }

    if (zDirNotZero) {
      double minZIntersect = (minPoint.z - modRay.origin.z)/modRay.direction.z;
      double maxZIntersect = (maxPoint.z - modRay.origin.z)/modRay.direction.z;
      if (minZIntersect < maxZIntersect) {
        maxEntry = (maxEntry > minZIntersect) ? maxEntry : minZIntersect;
        minExit = (minExit < maxZIntersect) ? minExit : maxZIntersect;
      } else {
        maxEntry = (maxEntry > maxZIntersect) ? maxEntry : maxZIntersect;
        minExit = (minExit < minZIntersect) ? minExit : minZIntersect;
      }
    } else {
      if ((modRay.origin.z < minPoint.z)||(modRay.origin.z > maxPoint.z))
      return false;
    }

    if (minExit < maxEntry) {
      // The line does not intersect the box
      return false;
    } else if (minExit >= modRay.tStart && maxEntry < modRay.tEnd) {
      // The line AND the ray intersect the box
      modRay.tStart = (modRay.tStart > maxEntry) ? modRay.tStart : maxEntry;
      modRay.tEnd = (modRay.tEnd < minExit) ? modRay.tEnd : minExit;
      return true;
    } else {
      return false;
    }
  }

  /** Returns the surface area of this box.
   * @return the surface area of this box.
   */
  public double surfaceArea() {
    double xSize = maxPoint.x-minPoint.x;
    double ySize = maxPoint.y-minPoint.y;
    double zSize = maxPoint.z-minPoint.z;
    return 2*xSize*ySize + 2*xSize*zSize + 2*ySize*zSize;
  }

  public Vector3d getCenter() {
    Vector3d result = new Vector3d();
    getCenter(result);
    return result;
  }
  
  public void getCenter(Vector3d outCenter) {
    outCenter.x = 0.5*(minPoint.x + maxPoint.x);
    outCenter.y = 0.5*(minPoint.y + maxPoint.y);
    outCenter.z = 0.5*(minPoint.z + maxPoint.z);
  }
  
  /** Returns the size of the x-dimension of this box.
   * @return the size of the x-dimension of this box.
   */
  public double getXSize() {
    return maxPoint.x-minPoint.x;
  }

  /** Returns the size of the y-dimension of this box.
   * @return the size of the y-dimension of this box.
   */
  public double getYSize() {
    return maxPoint.y-minPoint.y;
  }

  /** Returns the size of the z-dimension of this box.
   * @return Returns the size of the z-dimension of this box.
   */
  public double getZSize() {
    return maxPoint.z-minPoint.z;
  }

  /** Computes the four diagonals of the box (line segments connecting opposing
   * corners).
   * @return an array of four ray segments representing the four diagonals
   */
  public Ray [] getDiagonals() {
    Ray [] result = new Ray[4];
    result[0] = new Ray(new Vector3d(minPoint), new Vector3d(maxPoint));
    result[1] = new Ray(new Vector3d(maxPoint.x,minPoint.y,minPoint.z),
    new Vector3d(minPoint.x,maxPoint.y,maxPoint.z));
    result[2] = new Ray(new Vector3d(minPoint.x,maxPoint.y,minPoint.z),
    new Vector3d(maxPoint.x,minPoint.y,maxPoint.z));
    result[3] = new Ray(new Vector3d(minPoint.x,minPoint.y,maxPoint.z),
    new Vector3d(maxPoint.x,maxPoint.y,minPoint.z));
    return result;
  }
  /* get individual diagonals in place (no allocation) */
  /* uses ugly hack where endpoint is temporarily stored in ray.direction */
  public void getDiagonal(Ray outRay, int index) {
    if (index == 0) {
      outRay.makeSegment(minPoint,maxPoint);
    } else if (index == 1) {
      outRay.origin.set(maxPoint.x,minPoint.y,minPoint.z);
      outRay.direction.set(minPoint.x,maxPoint.y,maxPoint.z);
      outRay.makeSegment(outRay.origin,outRay.direction);
    } else if (index == 2) {
      outRay.origin.set(minPoint.x,maxPoint.y,minPoint.z);
      outRay.direction.set(maxPoint.x,minPoint.y,maxPoint.z);
      outRay.makeSegment(outRay.origin,outRay.direction);
    } else {
      outRay.origin.set(minPoint.x,minPoint.y,maxPoint.z);
      outRay.direction.set(maxPoint.x,maxPoint.y,minPoint.z);
      outRay.makeSegment(outRay.origin,outRay.direction);
    }
  }
  /** Get the diagonal which is closest to the given direction */
  public void getDiagonal(Ray outRay, Vector3d direction) {
    //we initially store the end points in the outRay.origin and direction
    //and then set the ray using them (a bit hacky but requires no allocation)
    if (direction.x > 0) {
      outRay.origin.x = minPoint.x;
      outRay.direction.x = maxPoint.x;
    } else {
      outRay.origin.x = maxPoint.x;
      outRay.direction.x = minPoint.x;
    }
    if (direction.y > 0) {
      outRay.origin.y = minPoint.y;
      outRay.direction.y = maxPoint.y;
    } else {
      outRay.origin.y = maxPoint.y;
      outRay.direction.y = minPoint.y;
    }
    if (direction.z > 0) {
      outRay.origin.z = minPoint.z;
      outRay.direction.z = maxPoint.z;
    } else {
      outRay.origin.z = maxPoint.z;
      outRay.direction.z = minPoint.z;
    }
    outRay.makeSegment(outRay.origin,outRay.direction);
  }

  /** Returns a string representation of this box for debugging purposes.
   * @return a string representation of this box.
   */
  public String toString() {
    return minPoint + "\n" +maxPoint;
  }

  /** Writes the min point into the given array of floats.
   * @param outFloats the array into which the min point should be written.
   */
  public void getMin(float[] outFloats) {
    outFloats[0] = (float) minPoint.x;
    outFloats[1] = (float) minPoint.y;
    outFloats[2] = (float) minPoint.z;
  }

  /** Writes the max point into the given array of floats.
   * @param outFloats the array of floats into which the max point should be written.
   */
  public void getMax(float[] outFloats) {
    outFloats[0] = (float) maxPoint.x;
    outFloats[1] = (float) maxPoint.y;
    outFloats[2] = (float) maxPoint.z;
  }

  /** Returns true iff this box contains the parameter
   * box. That is, iff the space contained by the
   * parameter box is a subset of that contained by this
   * box.
   * @param inBox the box to check for containment
   * @return true iff this box contains inBox.
   */
  public boolean contains(AxisAlignedBox inBox) {
    return
    (inBox.maxPoint.x <= maxPoint.x) &&
    (inBox.maxPoint.y <= maxPoint.y) &&
    (inBox.maxPoint.z <= maxPoint.z) &&
    (inBox.minPoint.x >= minPoint.x) &&
    (inBox.minPoint.y >= minPoint.y) &&
    (inBox.minPoint.z >= minPoint.z);
  }

  public boolean contains(Vector3d inPoint) {
    return
    (inPoint.x <= maxPoint.x) &&
    (inPoint.x >= minPoint.x) &&
    (inPoint.y <= maxPoint.y) &&
    (inPoint.y >= minPoint.y) &&
    (inPoint.z <= maxPoint.z) &&
    (inPoint.z >= minPoint.z);
  }
  public boolean contains(double inX, double inY, double inZ) {
    return
    (inX <= maxPoint.x) &&
    (inX >= minPoint.x) &&
    (inY <= maxPoint.y) &&
    (inY >= minPoint.y) &&
    (inZ <= maxPoint.z) &&
    (inZ >= minPoint.z);
  }

  /**
   * Returns true iff the volume of this box intersects the volume of the given
   * box.
   * @param inBox the box to check for intersection
   * @return true iff there is an intersection
   */
  public boolean intersects(AxisAlignedBox inBox) {
    boolean xNotIntersect = inBox.minPoint.x > maxPoint.x || inBox.maxPoint.x < minPoint.x;
    boolean yNotIntersect = inBox.minPoint.y > maxPoint.y || inBox.maxPoint.y < minPoint.y;
    boolean zNotIntersect = inBox.minPoint.z > maxPoint.z || inBox.maxPoint.z < minPoint.z;
    if (xNotIntersect || yNotIntersect || zNotIntersect) {
      return false;
    } else {
      return true;
    }
  }

  /** Returns the volume of this axis-aligned box
   * @return the volume of this box
   */
  public double getVolume() {
    return (maxPoint.x-minPoint.x)*(maxPoint.y-minPoint.y)
    *(maxPoint.z-minPoint.z);
  }

  /**
   * Scales the box while keeping the same center.
   * @param inScale the scaling factors (> 1 makes it larger)
   */
  public void scale(Vector3d inScale) {
    double centerX = (minPoint.x + maxPoint.x)*0.5;
    double centerY = (minPoint.y + maxPoint.y)*0.5;
    double centerZ = (minPoint.z + maxPoint.z)*0.5;
    maxPoint.x = (maxPoint.x-centerX)*inScale.x + centerX;
    minPoint.x = (minPoint.x-centerX)*inScale.x + centerX;
    maxPoint.y = (maxPoint.y-centerY)*inScale.y + centerY;
    minPoint.y = (minPoint.y-centerY)*inScale.y + centerY;
    maxPoint.z = (maxPoint.z-centerZ)*inScale.z + centerZ;
    minPoint.z = (minPoint.z-centerZ)*inScale.z + centerZ;
  }
  /**
   * Scales the box while keeping the same center.
   * @param inScale the scaling factor (> 1 makes it larger)
   */
  public void scale(double inScale) {
    double centerX = (minPoint.x + maxPoint.x)*0.5;
    double centerY = (minPoint.y + maxPoint.y)*0.5;
    double centerZ = (minPoint.z + maxPoint.z)*0.5;
    maxPoint.x = (maxPoint.x-centerX)*inScale + centerX;
    minPoint.x = (minPoint.x-centerX)*inScale + centerX;
    maxPoint.y = (maxPoint.y-centerY)*inScale + centerY;
    minPoint.y = (minPoint.y-centerY)*inScale + centerY;
    maxPoint.z = (maxPoint.z-centerZ)*inScale + centerZ;
    minPoint.z = (minPoint.z-centerZ)*inScale + centerZ;
  }

  public void translate(Vector3d inTranslation) {
    maxPoint.add(inTranslation);
    minPoint.add(inTranslation);
  }
}



