import java.net.*;
import java.io.*;
import java.util.*;
import java.math.*;
import java.nio.*;
import java.nio.channels.*;
import java.nio.channels.spi.SelectorProvider;
import java.nio.charset.*;

public class PeerConnection {
	private HashSet<Node> peer_set; //for all known peers
	private HashSet<Node> peer_set_con; //for pending connection
	private Selector sel;
	private MessageQueue msg_q;
	private Logger logger;
	private long con_timer_begin;
	
	public PeerConnection(MessageQueue msg_q, ServerSocketChannel ssChannel,Logger logger) {		
		this.msg_q = msg_q;
		this.logger= logger;
		peer_set = new HashSet<Node>();
		peer_set_con= new HashSet<Node>();
		try {			
			sel = SelectorProvider.provider().openSelector();
			ssChannel.register(sel, SelectionKey.OP_ACCEPT); //accept connection
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * access message queue to get more peers. getQueue() might block, if tracker hasn't fetched
	 * the queue in time
	 * @throws IOException
	 */
	private void getPeers() throws IOException{
		// MessageQueue contains holds the response from the tracker
		byte[] allPeers = msg_q.getQueue();
		if(allPeers==null){
			return;
		}
			
		final int pinfo_len = 6;
		int nPeer = Math.min(msg_q.getTotalPeerNum(), allPeers.length / pinfo_len);
		
		for(int i = 0; i < nPeer; i++){
			// Fix this when there is time
			byte[] ip= {allPeers[i*pinfo_len],allPeers[i*pinfo_len+1],allPeers[i*pinfo_len+2],allPeers[i*pinfo_len+3]}; 
			byte[] port= {allPeers[i*pinfo_len+4],allPeers[i*pinfo_len+5]}; //network order
			int portn = Helper.uByteShortToInt(port); //network order->host
			byte[] key= new byte[pinfo_len]; //key for PeerMap
			System.arraycopy(ip, 0, key, 0, ip.length);
			System.arraycopy(port, 0, key, 4, port.length);
						
			InetAddress cur_inet = InetAddress.getByAddress(ip);
			Node cur_node= null;
			try{
				cur_node = new Node(new InetSocketAddress(cur_inet, portn));
			} catch(IOException e){
				continue;
			}
			if(peer_set.contains(cur_node)){
				cur_node.sc.close();
			}
			else{
				putHandshake(msg_q.tcn.tf, cur_node.handshake);
				peer_set.add(cur_node);
				peer_set_con.add(cur_node);
			}
		}
	}
	
	public void putHandshake(TorrentFile tf, ByteBuffer handshake){
		//handshake
		byte[] hs= new byte[68]; //1+19+8+20+20=68
		byte[] pstr={'B','i','t','T','o','r','r','e','n','t',' ',
				'p','r','o','t','o','c','o','l'};
		byte[] pstrlen={19};
		byte[] reserved={0,0,0,0,0,16,0,0}; //extension_protocol (bit 44 set)
		byte[] info_hash= tf.info_hash_as_binary;
		byte[] peer_id= TorrentCrawler.rid;
		System.arraycopy(pstrlen, 0, hs, 0, 1);
		System.arraycopy(pstr,0,hs,1,19);
		System.arraycopy(reserved,0,hs,20,8);
		System.arraycopy(info_hash,0,hs,28,20);
		System.arraycopy(peer_id,0,hs,48,20);
		handshake.put(hs);
		handshake.position(0);
	}
	
	public void writeConInfo(){
		int connected= 0;
		int dropped= 0;
		
		Iterator cur_it= peer_set.iterator();
		while(cur_it.hasNext()){
			Node cur_node= (Node) cur_it.next();
			if(cur_node.connected)
				connected++;
			else if(cur_node.dropped)
				dropped++;
		}
		
		logger.writeLine(logger.con,System.currentTimeMillis(),"connected:"+connected+
				" dropped:"+dropped+" known:"+peer_set.size()+" seed:"+msg_q.tcn.tf.seed+" leech:"+
				msg_q.tcn.tf.leech);
		
	}
	
	public int connectToNodes(long measure_period) {
		writeConInfo();
		if (peer_set_con.isEmpty()) {
			try {
				getPeers();						
			} catch (IOException e) {
				System.err.println("IOException in connectToNodes 1: " + e);						
			}
		}
		
		MessageHandler mh= new MessageHandler(msg_q.tcn.tf,logger);
		
		//time out period
		final int max_time_ms = 3000;
		long measure_start_time = System.currentTimeMillis();			
		while (System.currentTimeMillis() - measure_start_time < measure_period) {	
			
			// Purge old timeout for OP_CONNECT
			long cur_time = System.currentTimeMillis();
			for(Iterator<SelectionKey> i = sel.keys().iterator(); i.hasNext();) {
				SelectionKey sk = i.next();
				if(!sk.isValid() || sk.interestOps()!=sk.OP_CONNECT)
					continue;
				Node cur_node = (Node)sk.attachment();
				
				if (cur_time - cur_node.begin_time > max_time_ms) {
					sk.cancel();						
					if (cur_node.reconnect()) {
						peer_set_con.add(cur_node);								
					}
				}
			}
			
			if(cur_time - con_timer_begin > 30000){
				writeConInfo();
				con_timer_begin= System.currentTimeMillis();
			}
			
			int num_ready;
			try {				
				num_ready = sel.select(500);				
			} catch (IOException e) {
				System.err.println("IOException at select: " + e);
				return -1;				
			}
			
			if (num_ready == 0) {
				if (peer_set_con.isEmpty()) {
					try {
						getPeers();						
					} catch (IOException e) {
						System.err.println("IOException in connectToNodes 2: " + e);						
					}
				}
				
				Iterator cur_it = peer_set_con.iterator();
				if (cur_it.hasNext()) {	
					Node cur_node = (Node)cur_it.next();
					cur_it.remove();	// Remove it from the map
					
					long begin_time = System.currentTimeMillis();
					boolean connected = false;
					try {
						connected = cur_node.connectToNode();
					} catch (IOException e) {
						peer_set_con.add(cur_node);
						continue;
					}
					//begin time is used for latency measure if OP_CONNECT, and for timeout
					cur_node.addBeginTime(begin_time);
					try{
						if (connected) {
							long end_time = System.currentTimeMillis();
							//System.out.println("Latency to Node " + cur_node.addr + " is " + (end_time - cur_node.begin_time));
							cur_node.addLatency(end_time-cur_node.begin_time);
							cur_node.connected= true;
							cur_node.dropped= false;
							cur_node.sc.register(sel, SelectionKey.OP_WRITE, cur_node);
							continue;				
						}
						cur_node.sc.register(sel, SelectionKey.OP_CONNECT, cur_node);
					} catch (ClosedChannelException e) {
						System.err.println("ClosedChannelException: " + e);						
					}
				}		
				continue;
			}
			
			long end_time = System.currentTimeMillis(); 
			for(Iterator<SelectionKey> i= sel.selectedKeys().iterator(); i.hasNext();){
				try{
					SelectionKey sk = i.next();
					
					if(sk.interestOps()==sk.OP_ACCEPT && sk.isAcceptable()){
						ServerSocketChannel ssc= (ServerSocketChannel) sk.channel();
						SocketChannel sc= ssc.accept();
						if(sc!=null){
							sc.configureBlocking(false);
							System.out.println("accept an incoming connection");
							Socket s= sc.socket();
							InetAddress inet= s.getInetAddress();
							int portn= s.getPort();
							Node cur_node = new Node(new InetSocketAddress(inet,portn),sc);
							cur_node.incoming= true;
							cur_node.connected= true;
							cur_node.dropped= false;
							if(peer_set.contains(cur_node)){
								cur_node.sc.close();
							}
							else{
								putHandshake(msg_q.tcn.tf, cur_node.handshake);
								peer_set.add(cur_node);
								//peer_set_con.add(cur_node);
								sc.register(sel,SelectionKey.OP_WRITE,cur_node);
							}
						}
						continue;
					}
					
					Node cur_node = (Node)sk.attachment();
					cur_node.addDownrate(end_time, msg_q.tcn.tf.piece_length);
					/*
					if (end_time - cur_node.begin_time > max_time_ms) {
						System.out.println("Timed out in event loop");
						sk.cancel();
						if (cur_node.reconnect()) {
							peer_set.add(cur_node);
						}
						continue;
					}
					*/
					if (sk.isConnectable() && sk.interestOps() == sk.OP_CONNECT) {						
						try {
							if (cur_node.sc.finishConnect()) {
								cur_node.connected= true;
								cur_node.dropped= false;
								System.out.println("Finish connect Latency to Node " + cur_node.addr + " is " + (end_time - cur_node.begin_time));
								cur_node.addLatency(end_time-cur_node.begin_time);
								sk.interestOps(sk.OP_WRITE);
							}
						} catch (IOException e) {
							//System.err.println("Finish connect exception: " + e);
							sk.cancel();
							if (cur_node.reconnect()) {
								peer_set_con.add(cur_node);								
							}
						}
						continue;
					} 
					if (sk.isWritable() && sk.interestOps() == sk.OP_WRITE) {
						try{
							cur_node.sc.write(cur_node.handshake);
							if(!cur_node.handshake.hasRemaining()){
								//System.out.println(cur_node.addr+" sent handshake");
								cur_node.handshaked= true;
								cur_node.handshake.position(0);
								sk.interestOps(sk.OP_READ);
							}
						}
						catch(NotYetConnectedException e){
							System.out.println("NotYetConnectedException in OP_WRITE "+e);
							sk.cancel();
							if (cur_node.reconnect()) {
								peer_set_con.add(cur_node);								
							}
						}
						catch(IOException e){
							System.out.println("IOException in OP_WRITE "+e);
							sk.cancel();
							if (cur_node.reconnect()) {
								peer_set_con.add(cur_node);								
							}
						}
						continue;
					}
					if (sk.isReadable() && sk.interestOps() == sk.OP_READ) {
						cur_node.bf= mh.readPeerResponse(cur_node,sk,peer_set_con);
						if(cur_node.dr_measured)
							sk.cancel();
						continue;
					}
				} catch(CancelledKeyException e){
					System.err.println("CancelledKeyException" + e);
					continue;
				} catch(IOException e){
					//System.err.println("connectToNodes 3:" + e);
					continue;
				}
			}
		}
		msg_q.finish();
		System.out.println("measure time expired");
		writeConInfo();
		logger.writeAll(msg_q.tcn.tf,peer_set);
		return 0;
	}
}
