require 'common.rb'
require 'S3'

class Silt
	RECHECK_TIME = 0.05

	def initialize(conn, bucket, gossip_port, locker_address = "", tracker_address = "", port = nil)
		@conn = conn
		@bucket = bucket
		@keys_hashes = {}
		@keys_timestamps = {}
		@read_locks = {}
		@write_locks = {}
		@keys_hashes_semaphore = Mutex.new		# applies to keys_hashes and write_locks
		@locker = [TCPSocket.new(locker_address, LOCKER_PORT, "", port)]
		@in_queue = []
		@out_queue = []
		@last_out_backup = nil

		@transaction_num = 0
		@transaction_keys = {}

		# Gossip Thread
		Thread.new do
			gossiper(tracker_address, gossip_port, @keys_hashes, @write_locks, @keys_hashes_semaphore)
		end

		# send/recv thread .. greatly reduced in complexity because only one op is
		# waiting for 'send' usually. The same is for 'recv' except for expirey
		# notices
		Thread.new do
			loop do
				writer = nil; writer = @locker if @out_queue.length > 0
				res = select(@locker, writer, nil, RECHECK_TIME)
				if res != nil
					if res[0].length > 0
						buf = @locker[0].recv(PACK_LEN)
						if buf.length == 0
							# NOTE: faulty server
							@locker[0].close
							@locker.delete_at(0)
							log "Silt: lock server down!"

							# keep trying to establish a connection with a new locker
							loop do
								sleep 1
								begin
									@locker << TCPSocket.new(locker_address, LOCKER_PORT, "", port)
									break
								rescue
								end
							end
							log "Silt: lock server is back up!"
							if @last_out_backup != nil
								log "Silt: resending last message to server"
								@out_queue.insert(0, @last_out_backup)
							end
						else
							len = buf.unpack(PACKER)[0].to_i
							msg = Marshal.restore(@locker[0].recv(len))

							if msg.is_expire?
								@keys_hashes_semaphore.synchronize do
									msg.keys_hashes.each do |k,h|
										@read_locks.delete(k)
										@write_locks.delete(k)
										@keys_hashes[k] = h
									end
									log "Silt: keys expired [#{s = ""; msg.keys_hashes.each {|k,h| s << k + ", "}; s}]"
								end
							else
								@in_queue << msg
								@last_out_backup = nil		# since locker only sends back messages to client after receiving a request/release
							end
						end
					end
					if res[1].length > 0
						begin
							@last_out_backup = @out_queue.delete_at(0)
							@locker[0].send(@last_out_backup, 0)
						rescue
						end
					end
				end
			end
		end
	end

	def begin_transaction(keys)
		keys = [keys] if keys.class != Array
		lock_for_write(keys)
		@transaction_num += 1
		@transaction_keys[@transaction_num] = {}
		keys.each do |k|
		end
		return @transaction_num
	end

	def commit(transaction_id)
		@transaction_keys[transaction_id].each do |k,tmpK|
			value = @conn.get(@bucket, tmpK).object.data
			@conn.put(@bucket, k, value)
			@conn.delete(@bucket, tmpK)
		end
		release_locks(@transaction_keys[transaction_id].keys)
	end

	def abort(transaction_id)
		@transaction_keys[transaction_id].each do |k,tmpK|
			@conn.delete(@bucket, tmpK)
		end
		release_locks(@transaction_keys[transaction_id].keys)
	end

	def transaction_read(transaction_id, key)
		tmpK = @transaction_keys[transaction_id][key] || key
		return @conn.get(@bucket, tmpK) if tmpK
	end

	def transaction_write(transaction_id, key, value)
		@transaction_keys[transaction_id][key] ||= k + "-" + rand_string(10)
		tmpK = @transaction_keys[transaction_id][key]
		return @conn.put(@bucket, tmpK, value) if tmpK
	end

	def lock_for_read(keys, lease = nil)
		keys = [keys] if keys.class != Array
		needed_keys = keys - (@read_locks.keys + @write_locks.keys)
		if needed_keys.length == 0
			log "Silt: needed keys already acquired"
			return
		end

		request_locks( needed_keys, RD_LOCK_CMD, lease )
		wait_for_grant()
		needed_keys.each {|k| @read_locks[k] = true}
	end

	def lock_for_write(keys, lease = nil)
		keys = [keys] if keys.class != Array
		needed_keys = keys - @write_locks.keys
		if needed_keys.length == 0
			log "Silt: needed keys already acquired"
			return
		end

		request_locks( needed_keys, WR_LOCK_CMD, lease )
		wait_for_grant()
		@keys_hashes_semaphore.synchronize do
			needed_keys.each {|k| @write_locks[k] = true}
		end
	end

	def read(key)
		# check if we have the read lock on the key
		return -1 if @read_locks[key].nil? && @write_locks[key].nil?
		# read the key's value
		value = nil
		loop do
			begin
				value = @conn.get(@bucket, key).object.data
			rescue
				next
			end
			if @keys_hashes[key] && @keys_hashes[key][0] != value.hash.to_s
				log "Warning: value read does not match latest hash!"
				sleep 0.1
				next
			end
			break
		end
		return value
	end

	def write(key, value)
		# check if we have the write lock on the key
		return -1 if @write_locks[key].nil?
		# write the value
		loop do
			begin
				@conn.put(@bucket, key, value)
				break
			rescue
			end
		end
		# update the key's hash, the timestamp should be either 0 or the latest
		# grant timestamp from lock server on that key
		@keys_hashes_semaphore.synchronize do
			@keys_hashes[key] = [value.to_s.hash.to_s, @keys_timestamps[key]]
		end
		return 1		# success
	end

	def release_locks(keys)
		keys = [keys] if keys.class != Array
		held_keys = keys & (@read_locks.keys + @write_locks.keys)
		if held_keys.length == 0
			log "Silt: no keys locked to release"
			return
		end

		kh = {}
		held_keys.each {|k| kh[k] = (@keys_hashes[k] || ["",0])}
		msg = Marshal.dump(Message.new(UNLOCK_CMD, kh))

		@out_queue << ([msg.length].pack(PACKER) << msg)
		wait_for_grant(true)		# to get updated hash/timestamp from lock server
		# delete the locks
		@keys_hashes_semaphore.synchronize do
			held_keys.each {|k| @read_locks.delete(k); @write_locks.delete(k)}
		end
		log "Silt: released [#{s = ""; held_keys.each {|k| s << "{" + k + " => [" + @keys_hashes[k][0] + ", " + @keys_hashes[k][1].to_s + "]}, "}; s}]"
	end

	private

	def request_locks(keys, locks_type, lease)
		kh = {}
		keys.each { |k| kh[k] = (@keys_hashes[k] || ["",0]) }
		msg = Marshal.dump(Message.new(locks_type, kh, lease))

		@out_queue << ([msg.length].pack(PACKER) << msg)
		log "Silt: requested [#{s = ""; kh.each {|k,h| s << "{" + k + " => [" + h[0] + ", " + h[1].to_s + "]}, "}; s}]"
	end

	def wait_for_grant(is_for_release = false)
		loop do
			if @in_queue.length > 0 && @in_queue[0].is_grant?
				msg = @in_queue.delete_at(0)
				# update known keys hashes
				@keys_hashes_semaphore.synchronize do
					msg.keys_hashes.each do |k,h|
						@keys_hashes[k] = h
						@keys_timestamps[k] = msg.timestamp
					end
				end
				if is_for_release == false
					log "Silt: was granted [#{s = ""; msg.keys_hashes.each {|k,h| s << "{" + k + " => [" + h[0] + ", " + h[1].to_s + "]}, "}; s}]"
				end
				msg.keys_hashes.each do |k,h|
					if h.length == 3 && h[2] == 1
						log "Silt: **inconsistent key {#{k} => [#{h[0]}, #{h[1]}]}"
					end
				end
				break
			else
				sleep RECHECK_TIME
			end
		end
	end

	def rand_string(len)
		alphabet = ('a'..'z').to_a
		ret = ""
		len.times { ret << alphabet[rand(alphabet.length)] }
		return ret
	end
end
