#!/usr/bin/ruby
require 'common.rb'

if ARGV.length < 2
	puts "USAGE: locker <tracker address> <gossip port>"
	exit(1)
elsif ARGV[1].to_i < 5001 || ARGV[1].to_i > 6000
	puts "Gossip port number must be between 5001 and 6000"
	exit(1)
end

$rd_sockets = []
$wr_sockets = nil

$read_locks  = {}
$write_locks = {}
$locks_leases = {}
$keys_hashes = {}
$wait_queue = []
$out_queue = {}

$keys_hashes_semaphore = Mutex.new

# setup socket
$server = TCPServer.new("", LOCKER_PORT)
$server.setsockopt(Socket::SOL_SOCKET, Socket::SO_REUSEADDR, 1)
$rd_sockets << $server

# Gossip thread
Thread.new do
	gossiper(ARGV[0], ARGV[1].to_i, $keys_hashes, $write_locks, $keys_hashes_semaphore)
end


def accept_connection
	new_client = $server.accept
	$rd_sockets << new_client
	log "Locker: accepted connection from #{new_client.peeraddr[3] + ":" + new_client.peeraddr[1].to_s}"
end

def grant_request(sock, req, timestamp)
	# create table of key-hash updates that client does not know about
	kh = {}
	# NOTE: A semaphore is not needed here because they keys_hashes table can not
	# be modified by another thread when we are here
	req.keys_hashes.each do |k, ht|
		if ($keys_hashes[k] && ($keys_hashes[k][0,2] != ht[0,2]))
			kh[k] = $keys_hashes[k]
		else
			kh[k] = ht
		end
	end
	# create a message to send back to the client
	msg = Marshal.dump(Message.new(GRANT_CMD, kh, timestamp))
	$out_queue[sock] ||= []
	$out_queue[sock] << ([msg.length].pack(PACKER) << msg)
	# enable this socket for writing
	$wr_sockets ||= []
	$wr_sockets << sock

	log "Locker: granted #{req.is_lock? ? "lock of" : "release of"} [#{s = ""; kh.each {|k,h| s << "{" + k + " => [" + h[0] + ", " + h[1].to_s + "]}, "}; s}] to #{sock.peeraddr[3] + ":" + sock.peeraddr[1].to_s}"
end

# process blocked requests from waiting queue
def process_requests
	loop do
		return unless $wait_queue.length > 0
		# this request can proceed if none of its keys are locked
		$wait_queue[0][1].keys_hashes.each_key {|k| return if $write_locks.has_key?(k)}
		if $wait_queue[0][1].is_write_lock?
			$wait_queue[0][1].keys_hashes.each_key {|k| return if $read_locks.has_key?(k)}
		end
		# request can proceed, remove it from waiting queue
		sock, req = $wait_queue.delete_at(0)
		issued_on = Time.now.to_i
		grant_request(sock, req, issued_on)
		# lock the requested keys
		if req.is_read_lock?
			req.keys_hashes.each_key {|k| $read_locks[k] ||= [] ; $read_locks[k] << sock}
		else
			$keys_hashes_semaphore.synchronize do
				req.keys_hashes.each_key {|k| $write_locks[k] ||= [] ; $write_locks[k] << sock}
			end
		end
		# register the lease expiration timestamp for the locks
		lease_period = [MAX_LEASE]
		lease_period << req.timestamp if req.timestamp
		expires_on = lease_period.min + issued_on
		$locks_leases[sock] ||= {}
		req.keys_hashes.each_key {|k| $locks_leases[sock][k] = [issued_on, expires_on]}
	end
end

def release_key(sock, release_request, key, hash, timestamp)
	if $keys_hashes[key] && (($keys_hashes[key][1] > hash[1]))
		expire_key(sock, key, nil, timestamp, true, true)
		grant_request(sock, release_request, timestamp)
		return
	end

	$locks_leases.keys.each do |s|
		if s != sock && $locks_leases[s].has_key?(key)
			# if some other socket has a lock on this key that was issued after
			# this request's timestamp, then that earlier lock grant was a mistake,
			# expire that key
			if hash[1] < $locks_leases[s][key][0]
				expire_key(s, key, nil, timestamp, true, true)
			end
		end
	end

	# update the corresponding key hash
	$keys_hashes_semaphore.synchronize do
		if hash[0].length > 0	# client made modification
			new_hash = [hash[0], timestamp]
			if hash.length == 3 && $keys_hashes[key][0] == hash[0]	# inconsistent data
				new_hash[2] = 1
			end
			$keys_hashes[key] = new_hash
		end

		# remove write lock if any and clean up
		if $write_locks[key]
			$write_locks[key].delete(sock)
			$write_locks.delete(key) if $write_locks[key].length == 0
		end
	end
	# remove read lock if any and clean up
	if $read_locks[key]
		$read_locks[key].delete(sock)
		$read_locks.delete(key) if $read_locks[key].length == 0
	end
	# remove lock lease
	if $locks_leases[sock]
		$locks_leases[sock].delete(key)
	end
end

# Expire all the keys for a given socket
def expire_socket_keys(sock, now, force = false, send_notice = true)
	return unless $locks_leases[sock]

	expired_keys = {}
	$locks_leases[sock].each do |k,t|
		key_hash = expire_key(sock, k, t, now, force)
		expired_keys[k] = key_hash unless key_hash.nil?
	end
	if expired_keys.size > 0 && send_notice
		# notify client of the expiry of his locks
		send_expiry_notice(sock, expired_keys)
	end
end

# Expire a particular key for a given socket
def expire_key(sock, key, timestamps, now, force, send_notice = false)
	if force || timestamps[1] < now
		if $read_locks[key]
			$read_locks[key].delete(sock)
			$read_locks.delete(key) if $read_locks[key].length == 0
		else
			$keys_hashes_semaphore.synchronize do
				$keys_hashes[key][2] = 1 if $keys_hashes[key]		# flag key hash for possible inconsistency
				if $write_locks[key] != nil
					$write_locks[key].delete(sock)
					$write_locks.delete(key) if $write_locks[key].length == 0
				end
			end
		end
		$locks_leases[sock].delete(key)
		if send_notice
			send_expiry_notice(sock, {key => $keys_hashes[key]})
		end
		return $keys_hashes[key]
	end
	return nil
end

# Queue up a message notifying a socket of the expiry of (some of) its keys
def send_expiry_notice(sock, expired_keys_hashes)
	$wr_sockets ||= []
	$wr_sockets << sock
	msg = Marshal.dump(Message.new(EXPIRE_CMD, expired_keys_hashes))
	$out_queue[sock] ||= []
	$out_queue[sock] << ([msg.length].pack(PACKER) << msg)
	log "Locker: expired keys [#{s = ""; expired_keys_hashes.each_key {|k| s << k + ", "}; s}]"
end

def check_for_expired_keys
	now = Time.now.to_i
	$locks_leases.each_key {|s| expire_socket_keys(s, now)}
end

log "Locker up! Listening for clients.."
n = 0
loop do
	res = select($rd_sockets, $wr_sockets, nil, 0.5)
	if res != nil
		for sock in res[0]
			if sock == $server
				accept_connection
			else
				buf = sock.recv(PACK_LEN)
				if buf.length == 0
					# faulty client
					$rd_sockets.delete(sock)
					expire_socket_keys(sock, 0, true, false)
					sock.close
				else
					len = buf.unpack(PACKER)[0].to_i
					req = Marshal.restore(sock.recv(len))

					if req.is_lock?
						log "Locker: received lock request on [#{s = ""; req.keys_hashes.each {|k,h| s << "{" + k + " => [" + h[0] + ", " + h[1].to_s + "]}, "}; s}] from #{sock.peeraddr[3] + ":" + sock.peeraddr[1].to_s}"
						$wait_queue << [sock, req]
					elsif req.is_release?
						log "Locker: received release request of [#{s = ""; req.keys_hashes.each {|k,h| s << "{" + k + " => [" + h[0] + ", " + h[1].to_s + "]}, "}; s}] from #{sock.peeraddr[3] + ":" + sock.peeraddr[1].to_s}"
						now = Time.now.to_i		# this is to timestamp key hashes
						req.keys_hashes.each {|k, h| release_key(sock, req, k, h, now)}
						grant_request(sock, req, now)
					end
					process_requests()
				end
			end
		end
		for sock in res[1]
			begin
				sock.send($out_queue[sock].delete_at(0), 0)
				if $out_queue[sock].length == 0
					$wr_sockets.delete(sock)
					$wr_sockets = nil if $wr_sockets.length == 0
				end
			rescue
			end
		end
	end
	# HACK: stupid hack, ideally would use a timer interrupt and a separate
	# thread
	n += 1
	if n == 10
		check_for_expired_keys
		n = 0
	end
end
