require 'socket'
require 'thread'

TRACKER_PORT = 4000
LOCKER_PORT  = 5000
N_PEERS = 3

RD_LOCK_CMD = 1
WR_LOCK_CMD = 2
UNLOCK_CMD  = 3
GRANT_CMD   = 4
EXPIRE_CMD  = 5

PACKER   = "S"
PACK_LEN = 2
MAX_LEASE = 100

class Node
	attr_accessor :addr, :port
	def initialize(addr, port)
		@addr = addr
		@port = port
	end
end

# key_hashes = {key => hash}
class Message
	attr_reader :cmd, :keys_hashes, :timestamp
	def initialize(cmd, keys_hashes, timestamp = nil)
		@cmd = cmd
		@keys_hashes = keys_hashes
		@timestamp = timestamp
	end

	def is_read_lock?
		@cmd == RD_LOCK_CMD
	end

	def is_write_lock?
		@cmd == WR_LOCK_CMD
	end

	def is_lock?
		(@cmd == RD_LOCK_CMD) || (@cmd == WR_LOCK_CMD)
	end

	def is_release?
		@cmd == UNLOCK_CMD
	end

	def is_grant?
		@cmd == GRANT_CMD
	end

	def is_expire?
		@cmd == EXPIRE_CMD
	end
end

def log(msg)
	# NOTE: comment the line below if you want to turn off scree logging
	puts msg
end

def gossiper(tracker_address, gossip_port, keys_hashes, write_locks, keys_hashes_semaphore)
	rd_fds = []
	wr_fds = []
	peers = []

	tracker_socket = TCPSocket.new(tracker_address, TRACKER_PORT)
	gossip_socket = TCPServer.new("", gossip_port)
	tracker_socket.setsockopt(Socket::SOL_SOCKET, Socket::SO_REUSEADDR, 1)
	gossip_socket.setsockopt(Socket::SOL_SOCKET, Socket::SO_REUSEADDR, 1)

	rd_fds << gossip_socket
	rd_fds << tracker_socket
	wr_fds << tracker_socket

	log "Gossiper: up and ready!.."
	loop do
		sleep rand(0)		# to create a little delay between multiple starting nodes
		# select random key/hash to gossip about
		gossip_kh = {}
		keys_hashes_semaphore.synchronize do
			avail_keys = keys_hashes.keys - write_locks.keys
			if avail_keys.length > 0
				i = rand(avail_keys.length)
				key = avail_keys[i]
				gossip_kh[key] = keys_hashes[key]
			end
		end

		# select random subset of peers to gossip with
		num_peers = [peers.length, N_PEERS].min	# maximum number of peers to send rumor to
		indices = (0..peers.length-1).to_a
		if peers.length > 0 && gossip_kh.size > 0
			wr_fds ||= []
			num_peers.times { wr_fds << peers[indices.delete_at(rand(indices.length))] }
		end

		# run select & read/write
		begin
			res = select(rd_fds + peers, wr_fds, nil, 0.5)
			if res != nil
				for sock in res[0]
					if sock == gossip_socket
						client = gossip_socket.accept
						log "Gossiper: new peer @  #{client.peeraddr[3] + ":" + client.peeraddr[1].to_s}"
						peers << client
					else
						buf = sock.recv(PACK_LEN)
						if buf.length == 0
							wr_fds.delete(sock) if wr_fds
							res[1].delete(sock)
							peers.delete(sock)
							sock.close
							next
						else
							len = buf.unpack(PACKER)[0].to_i
							obj = Marshal.restore(sock.recv(len))

							if sock == tracker_socket
								# Tracker announced new peer
								log "Gossiper: connecting to peer @ #{obj.addr + ":" + obj.port.to_s}"
								peers << TCPSocket.new(obj.addr, obj.port)
							else
								# Gossip Rumor Received
								key = obj.keys[0]
								hash = obj.values[0]
								# update local key hash value if it is older than incoming
								keys_hashes_semaphore.synchronize do
									if write_locks.has_key?(keys_hashes.keys[0]) == false
										if keys_hashes[key].nil? || ((hash[1] > keys_hashes[key][1]) && (keys_hashes[key][1] != 0))	# timestamp 0 denotes unstable hash after write
											keys_hashes[key] = hash
											log "Gossiper: received new info {#{key + " => [" + hash[0] + ", " + hash[1].to_s + "]"}} from #{sock.peeraddr[3] + ":" + sock.peeraddr[1].to_s}"
										elsif (hash[1] < keys_hashes[key][1])
											reply_gossip_kh = {}
											reply_gossip_kh[key] = keys_hashes[key]
											reply_msg = Marshal.dump(reply_gossip_kh)
											reply_msg = [reply_msg.length].pack(PACKER) << reply_msg
											sock.send(reply_msg, 0)
										end
									end
								end
							end
						end
					end
				end
				for sock in res[1]
					if sock == tracker_socket
						# send gossip port information to the tracker
						sock.send(gossip_port.to_s, 0)
						wr_fds.delete(sock)
						log "Gossiper: contacted tracker"
					else
						msg = Marshal.dump(gossip_kh)
						msg = [msg.length].pack(PACKER) << msg
						sock.send(msg, 0)
						#log "Gossiper: gossiped {#{gossip_kh.keys[0] + " => [" + gossip_kh.values[0][0] + ", " + gossip_kh.values[0][1].to_s + "]"}} to #{sock.peeraddr[3] + ":" + sock.peeraddr[1].to_s}"
						# remove from write list
						wr_fds.delete(sock)
					end
				end
				wr_fds = nil if wr_fds && wr_fds.length == 0
			end
		rescue
			sleep 1
		end
	end
end

