(* ====================================================== *)
(* Basics *)
(* ====================================================== *)
(* example: Module (Feel free to comment this out if you are not using multiple files!)*)
module P = Practice
let _ = print_endline (P.greetings ^ (string_of_int P.cid))

(* example: if *)
let grad_student = 4110 + if P.f then 2000 else 0
let _ = assert (grad_student = 4110)

(* example: let *)
let x = (let x = 5 in ((let x = 6 in x) + x))
let _ = assert (x = 11)

(* example: function *)
let id x = x
let _ = assert (id false = false)

let add3 x y z = x + y + z
let _ = assert (add3 0 1 2 = 3)

let id (x : int) : int = x
let _ = assert (id 0 = 0)

let id (x : 'a) : 'a = x
let _ = assert (id 0 = 0);
assert (id 'a' = 'a');
assert (id true = true)

let id = fun x -> x 
let _ = assert (id 0 = 0);
assert (id 'a' = 'a');
assert (id true = true)

(* example: recursion *)
let rec mul a b = 
  if (b = 0) 
    then 0 
    else if (b < 0)
      then -a + (mul a (b+1))
      else a + (mul a (b-1))

let _ = assert (mul 2 3 = 6);
assert (mul (-2) 3 = -6);
assert (mul 2 (-3) = -6);
assert (mul (-2) (-3) = 6)

(** exercise: [fib].
1, 1, 2, 3, 5, 8...
*)
(*
let rec fib a = ...

let _ = assert (fib (-1) = 1);
assert (fib 0 = 1);
assert (fib 1 = 1);
assert (fib 2 = 2);
assert (fib 3 = 3);
assert (fib 4 = 5);
assert (fib 5 = 8);
assert (fib 6 = 13);
assert (fib 7 = 21)
let _ = print_endline "fib passed!"
*)

(* ====================================================== *)
(* Types *)
(* ====================================================== *)
(* ?sian *)
type grade = A of int | B of int | C of int | F
type transcript = grade * grade * grade

let asian = (A 100, A 100, A 100)
let bsian = (B 85, B 80, B 75)
let csian = (C 70, C 65, C 60)
let fsian = (F, F, F)

let grade (g: grade) : int =
  match g with
  | A g -> g
  | B g -> g
  | C g -> g
  | F -> 0 

let _ = assert (grade (A 80) = 80);
assert (grade (B 100) = 100);
assert (grade F = 0)

let higher (g1 : grade) (g2: grade) : bool =
  (grade g1) >= (grade g2)

let prouder (s1 : transcript) (s2: transcript) : bool =
  match s1 with
  | g1, g2, g3 -> 
    (match s2 with
    | f1, f2, f3 -> (higher g1 f1) && (higher g2 f2) && (higher g3 f3)
  )

let _ = assert (prouder asian bsian);
assert (prouder bsian csian);
assert (prouder csian fsian)

(** exercise: [total].
Takes a transcript and returns the sum of its grades.
*)
(*
let total (s1 : transcript) : int = ...

let _ = assert (total asian = 300);
assert (total bsian = 240);
assert (total csian = 195);
assert (total fsian = 0)
let _ = print_endline "?sian passed!"
*)

(* example: recursive types *)
type my_int = Z | S of my_int

let zero = Z
(* AST
    S
    |
    Z
*)
let one = S Z
(* AST
    S
    |
    S
    |
    Z
*)
let two = S (S Z)
(* AST
    S
    |
    S
    |
    S
    |
    Z
*)
let three = S (S (S Z))

(* example: int_to_myint *)
let rec int_to_myint (i : int) : my_int =
  if (i <= 0) 
    then Z 
    else S (int_to_myint (i-1))

let _ = assert (int_to_myint 0 = zero);
assert (int_to_myint (-1) = zero);
assert (int_to_myint 1 = one);
assert (int_to_myint 2 = two);
assert (int_to_myint 3 = three)

(** exercise : [myint_to_int]
Translate any my_int to an int. *)
(*
let rec myint_to_int (m : my_int) : int = ...

let _ = assert (myint_to_int zero = 0);
assert (myint_to_int one = 1);
assert (myint_to_int two = 2);
assert (myint_to_int three = 3)
let _ = print_endline "myint_to_int passed!"
*)


(** exercise: [add] *)
(*
let rec add (m : my_int) (n : my_int) : my_int = ...

let _ = assert (add zero zero = zero);
assert (add zero one = one);
assert (add one zero = one);
assert (add one two = three)
let _ = print_endline "add passed!"
*)

(** exercise: [halve].
Divide a my_int by 2, round down.
*)
(*
let rec halve (m : my_int) : my_int = ...

let _ = assert (halve zero = zero);
assert (halve one = zero);
assert (halve two = one);
assert (halve (add two three) = two)
let _ = print_endline "halve passed!"
*)

(* example: lambda calculus *)
type var = string
type exp = Var of var
  | App of exp * exp
  | Lambda of var * exp

  (* AST
      Lambda
      /   |
     x    x
  *)
  let id_ = Lambda("x", Var("x"));;
  (* AST
      Lambda
    /   |
  x   Lambda
    /   |
  y     x
  *)
  let true_ = Lambda("x", Lambda("y", Var("x")));;
  (* AST
      Lambda
    /   |
  x   Lambda
    /   |
  y     y
  *)
  let false_ = Lambda("x", Lambda("y", Var("y")));;
  (* AST
      Lambda
    /   |
  x   Lambda
    /   |
  y   Lambda
    /   |
  z    App
    /   |
  App   z
  / \
 x  y
  *)
  let if_ = Lambda("x", Lambda("y", Lambda("z",
             App(App(Var("x"), Var("y")), Var("z")))));;

let _ = assert (id_ = id_);
assert (true_ = true_);
assert (false_ = false_);
assert (if_ = if_)

(* example: lists. In standard library, list is defined as:
type 'a list = [] | (::) of 'a * 'a list
*)
(* AST
    ::
  /  |
 1  ::
  /  |
 4  :: 
  /  |
 8  ::
  /  |
 5  ::
  /  |
 0  []
*)
let ithaca = [1;4;8;5;0]
let roosevelt_island = [1;0;0;4;4]
let worriormine = [2;4;8;9;4]
let greensboro = [0;5;8;4;1]
let amherst = [4;4;0;0;1]
let upperco = [2;1;1;5;5]
let saint_paul = [5;5;1;1;2]

(* example: sum *)
let rec length (l : 'a list) : 'a =
  match l with
  | [] -> 0
  | _ :: t -> 1 + (length t)

let _ = assert (length [] = 0);
assert (length ithaca = 5);
assert (length roosevelt_island = 5);
assert (length worriormine = 5);
assert (length greensboro = 5);
assert (length amherst = 5);
assert (length upperco = 5);
assert (length saint_paul = 5)

(** exercise: [reverse]. Reverse a list. *)
(*

let reverse (l : int list) : int list = ...

let _ = assert (reverse [] = []);
assert (reverse ithaca = greensboro);
assert (reverse roosevelt_island = amherst);
assert (reverse upperco = saint_paul)
let _ = print_endline "reverse passed!"
*)

(** exercise: [map].
Takes a function [f] and a list [l], and return a list whose
elements are applied by the function.
*)
(*
let rec map (f : 'a -> 'b) (l : 'a list) : 'b list = ...

let _ = assert (map id [] = []);
assert (map id ithaca = ithaca)
let _ = print_endline "map passed!"

(* example: partial application *)
let plus1 = map (fun x -> x + 1)

let _ = assert (plus1 [] = []);
assert (plus1 roosevelt_island = upperco);
assert (plus1 amherst = saint_paul)
*)

(** exercise: [zip].
Takes a pair of lists and return a list of pairs.
The shorter list is padded with 0s at the end.
*)
(*
let rec zip (lsts : 'a list * 'a list) : ('a * 'a) list = ...

let nowhere = ([], []) 
let cornell = (ithaca, roosevelt_island) 

let _ = assert (zip nowhere = []);
assert (zip (ithaca, []) = map (fun x -> (x, 0)) ithaca);
assert (zip ([], ithaca) = map (fun x -> (0, x)) ithaca);
assert (zip cornell = [(1, 1); (4, 0); (8, 0); (5, 4); (0, 4)])
let _ = print_endline "zip passed!"
*)


(** exercise: 'a tree.

1. Define ['a tree]. A tree is either a Leaf, which has type ['a];
or a [Node], which is a pair of subtrees of the same type.

2. Define the [map] function over your tree that takes
a polymorphic function [f], a tree [t] and 
returns a tree whose leaves are applied by f.

3. (Optional) play around your tree!
  Some ideas:
  function [flip] that swap left and right of your tree
  function [to_list] that turns a tree into a list
  function [from_list] that turns a list into a tree
*)