open Ast
open Extract
open Proof

type premise = int * formula
type task = int * premise list * formula
type state = task list * proof * formula

let task_name = ref 0
let premise_name = ref 0
let undo_stack : state list ref = ref []

let next_task_name() : int =
  let name = !task_name in incr task_name; name
  
let next_premise_name() : int =
  let name = !premise_name in incr premise_name; name

let premise_to_string (p : premise) : string =
  let (name, premise) = p in
    "    " ^ (Extract.to_string (Var name)) ^ ": " ^ (Ast.to_string premise)
    
let task_to_string (t : task) : string =
  let (name, premise_list, lemma) = t in
  let s =
    (Extract.to_string (Extract.Task name) ^ ":\n  Premises:") ::
    (if premise_list = [] then ["    no premises"] else List.map premise_to_string premise_list)
    @ ["  Goal: " ^ (Ast.to_string lemma)]
  in String.concat "\n" s
  
let start_state (e : formula) : state =
  task_name := 0;
  premise_name := 0;
  undo_stack := [];
  let name = next_task_name() in
  let task = (name, [], e) in
    ([task], Task name, e)

let to_string (state : state) : string =
  let (tasks, proof, theorem) = state in
  let s =
    ["Theorem: " ^ (Ast.to_string theorem);
     "Proof: " ^ Extract.to_string (extract proof)] in
  let t =
    match tasks with
      [] -> ["No tasks"]
    | task :: _ ->
        [string_of_int (List.length tasks) ^ " task(s)";
         "Current task " ^ (task_to_string task)] in
  String.concat "\n" (s @ t)
  