structure LambdaSimpleInterpretation : LAMBDA_SIMPLE_INTERPRETATION = struct
    structure LambdaSimpleAst = LambdaSimpleAst
    structure PP = LambdaSimplePPrint

    exception RuntimeError of string

    local open LambdaSimpleAst in
    (* Contexts map variables to expressions.  
     * They should be helpful when translating letrec expressions. 
     *)
    datatype ctxt = CTXT of (var * expr * ctxt) list;
      
    (* Adds a new binding for a variable in a context. 
     *)
    fun add (CTXT ctxt : ctxt) (v', e', ctxt') = CTXT ((v',e',ctxt')::ctxt);
      
    (* Looks up a variable in a context 
     *)
    fun lookup v (CTXT []) = NONE
      | lookup v (CTXT ((v', e', ctxt')::ctxt))
      = if eqvar v v'
	  then SOME (e',ctxt')
	  else lookup v (CTXT ctxt);
	    
    fun interpret e = 
      let
	fun interp ctxt e =
	  case e of
	    Num i => (Num i, ctxt)
	  | IsZero e => let
			  val (n,_) = interp ctxt e
			in 
			  case n
			    of Num(0) => (True,ctxt)
			     | _ => (False,ctxt)
			end
	  | Op(p,e1,e2) => let
			     val (n1,_) = interp ctxt e1
			     val (n2,_) = interp ctxt e2
			   in
			     case (n1,n2)
			       of (Num(i1),Num(i2))
				=> (case p
				      of Plus => (Num(i1+i2),ctxt)
				       | Times => (Num(i1*i2),ctxt)
				       | Minus => (Num(i1-i2),ctxt))
				| _ => raise (RuntimeError 
					      "Primop applied to non Num's")
			   end
	  | True => (True,ctxt)
	  | False => (False,ctxt)
	  | If(e0,e1,e2) => let
			      val (b,_) = interp ctxt e0
			    in
			      case b
				of True => interp ctxt e1
				 | False => interp ctxt e2
				 | _ => raise (RuntimeError
					       "If applied to non True/False")
			    end
	  | Var v => (case lookup v ctxt
			of SOME (e',ctxt') => (e',ctxt')
			 | NONE => raise (RuntimeError "Unbound variable"))
	  | App(e1,e2) => let
			    val (f,ctxtf) = interp ctxt e1
			    val (a,ctxta) = interp ctxt e2
			  in
			    case f
			      of Lambda(x,e) 
			       => interp (add ctxtf (x,a,ctxta)) e
			       | _ => raise (RuntimeError 
                                             "App applied to non Lambda")
			  end
	  | Lambda(x,e) => (Lambda(x,e),ctxt)
      in
	(#1 (interp (CTXT []) e)) 
	handle (RuntimeError s) => (print (s^"\n"); Num 0)
      end
   end
end