structure CPSEvaluation : CPS_EVALUATION =
  struct
    open CPSAst
    structure PP = CPSPPrint

    datatype value = Lam' of var * stm
                   | Halt'
                   | Tuple' of value list
                   | Num' of int
    fun toExp (Lam'(v,s)) = Lam(v,s)
      | toExp (Halt') = Halt
      | toExp (Tuple' _) = Tuple []
      | toExp (Num'(n)) = Num(n)

    fun is_value (Lam(x,s)) = true
      | is_value (Halt) = true
      | is_value (Num(n)) = true
      | is_value (Tuple(vs)) = true
      | is_value _ = false

    (* type value = exp such that is_value = true *)

    exception CpsFailure of string

    (* type env = var -> value *)
    val empty_env = []
    fun extend env x v = (x,v)::env
    fun lookup [] (x as (xs,xi)) 
      = raise CpsFailure ("Unbound variable " ^ xs ^ (Int.toString xi))   
      | lookup ((y,v)::env) x
      = if eqvar x y
	  then v
	  else lookup env x

    (* val step : env -> exp -> val *)
    fun step env (exp as Op(Plus,x0,x1)) = 
      (case (lookup env x0, lookup env x1)
	 of (Num'(n),Num'(m)) => Num'(n+m)
          | _ => (PP.ppE exp; 
		  print "\n";
		  raise CpsFailure "Added non-integer"))
      | step env (exp as Op(Minus,x0,x1)) = 
      (case (lookup env x0, lookup env x1)
	 of (Num'(n),Num'(m)) => Num'(n-m)
          | _ => (PP.ppE exp; 
		  print "\n";
		  raise CpsFailure "Subtracted non-integer"))
      | step env (exp as Op(Times,x0,x1)) = 
      (case (lookup env x0, lookup env x1)
	 of (Num'(n),Num'(m)) => Num'(n*m)
          | _ => (PP.ppE exp; 
		  print "\n";
		  raise CpsFailure "Multiplied non-integer"))
      | step env (exp as Index (x,n)) = 
      (case lookup env x 
	 of Tuple'(vs) => (List.nth (vs,n)
			   handle Subscript
			   => (PP.ppE exp;
			      print "\n";
			      raise CpsFailure "Index out of range"))
          | v => (PP.ppE exp;
		  print "\n";
		  raise CpsFailure "Indexed non-tuple"))
      | step env (Var(x)) = lookup env x
      | step env (Lam(v,s)) = Lam'(v,s)
      | step env Halt = Halt'
      | step env (Tuple(vs)) = Tuple' (List.map (fn v => lookup env v) vs)
      | step env (Num(n)) = Num'(n)

    (* val eval : env -> stm -> val *)
    and eval env (stm as App(x0,x1)) =
      (case lookup env x0
	 of Lam'(x,s) => eval (extend empty_env x (lookup env x1)) s
	  | Halt' => lookup env x1
          | _ => (PP.ppS stm;
		  print "\n";
		  raise CpsFailure "Attempt to apply non-function"))
      | eval env (stm as Ifz (x,s1,s2)) =
      (case lookup env x
	 of Num'(0) => eval env s1
	  | Num'(n) => eval env s2
	  | _ => (PP.ppS stm;
		  raise CpsFailure "Attempt to ifz non-integer"))
      | eval env (stm as Let (x,e,s)) = 
      let val v = step env e 
      in eval (extend env x v) s
      end
	 
    (* val evaluate : stm -> val *)
    fun evaluate s = toExp (eval empty_env s)
  end