type state = Ast.loc -> int
type config = Ast.com * state
type breakConfig = Ast.com * Ast.com * Ast.com list * state

let initialState : state = fun _ -> 0
    
let updateState (s, l, n) =
    (fun l' -> if l' = l then n else (s l'))
    
let lookupState (s, l) = (s l)

exception BadBreak
    
open Ast

let rec interpretAexp = function
  | (N(n), s) -> n
  | (Loc(l), s) -> lookupState(s, l)
  | (Plus(a1, a2), s) -> 
    let v1 = interpretAexp(a1, s) in
    let v2 = interpretAexp(a2, s) in
    v1 + v2
  | (Minus(a1, a2), s) ->
    let v1 = interpretAexp(a1, s) in
    let v2 = interpretAexp(a2, s) in
    v1 - v2
  | (Times(a1, a2), s) ->
    let v1 = interpretAexp(a1, s) in
    let v2 = interpretAexp(a2, s) in
    v1 * v2

let rec interpretBexp = function
  | (True, s) -> true
  | (False, s) -> false
  | (Equals(a1, a2), s) -> 
    let v1 = interpretAexp(a1, s) in
    let v2 = interpretAexp(a2, s) in
    v1 = v2
  | (LessEq(a1, a2), s) ->
    let v1 = interpretAexp(a1, s) in
    let v2 = interpretAexp(a2, s) in
    v1 <= v2
  | (Not(b), s) ->
    let b = interpretBexp(b, s) in
    not b
  | (And(b1, b2), s) ->
    let b1 = interpretBexp(b1, s) in
    let b2 = interpretBexp(b2, s) in
    b1 && b2
  | (Or(b1, b2), s) ->
    let b1 = interpretBexp(b1, s) in
    let b2 = interpretBexp(b2, s) in
    b1 || b2

let rec interpret = function
  | (Skip, s) -> s
  | (Assign (l, aexp), s) -> 
    let v = interpretAexp(aexp, s) in
    updateState(s, l, v)
  | (Seq(c1, c2), s) -> 
    let s' = interpret(c1, s) in
    interpret(c2, s')
  | (Cond(bexp, c1, c2), s) ->
    let b = interpretBexp(bexp, s) in
    if b then interpret(c1, s) else interpret(c2, s)
  | (While(bexp, c), s) ->
    let b = interpretBexp(bexp, s) in
    if b then 
	let s'' = interpret(c, s) in
	interpret(While(bexp, c), s'')
    else
	s
  | (Break, s) -> raise BadBreak

(* CS611: implement this function *)
let rec interpretBreak _ = raise BadBreak
