exception RuntimeError of string
open LambdaSimpleAst
    (* Contexts map variables to expressions.  
     * They should be helpful when translating letrec expressions. 
     *)
type ctxt = CTXT of (var * expr * ctxt) list

    (* Adds a new binding for a variable in a context. 
     *)
let add (CTXT ctxt : ctxt) (v', e', ctxt') 
    = CTXT ((v',e',ctxt')::ctxt)
  
    (* Looks up a variable in a context 
     *)
let rec lookup v = function
    (CTXT []) -> None
  | (CTXT ((v', e', ctxt')::ctxt)) -> 
      if eqvar v v'
      then Some (e',ctxt')
      else lookup v (CTXT ctxt)
    
let interpret e = 
  let rec interp ctxt e =
    match e with
      Num i -> (Num i, ctxt)
    | ZeroP e -> 
	let (n,_) = interp ctxt e
	in 
	(match n with
	  Num(0) -> interp ctxt transTrue
	| _ -> interp ctxt transFalse)
    | IsZero e -> 
	let (n,_) = interp ctxt e
	in 
	(match n with
	  Num(0) -> (True,ctxt)
	| _ -> (False,ctxt))
    | Op(p,e1,e2) -> 
	let (n1,_) = interp ctxt e1 in
	let(n2,_) = interp ctxt e2 in
	(match (n1,n2) with
	  (Num(i1),Num(i2))-> 
	    (match p with
	      Plus -> (Num(i1+i2),ctxt)
	    | Times -> (Num(i1*i2),ctxt)
	    | Minus -> (Num(i1-i2),ctxt))
	| _ -> raise (RuntimeError 
			"Primop applied to non Num's"))
    | True -> (True,ctxt)
    | False -> (False,ctxt)
    | If(e0,e1,e2) -> 
	let (b,_) = interp ctxt e0 in
	(match b with
	  True -> interp ctxt e1
	| False -> interp ctxt e2
	| _ -> raise (RuntimeError
			"If applied to non True/False"))
    | Var v -> 
	(match lookup v ctxt with
	  Some (e',ctxt') -> (e',ctxt')
	| None -> raise (RuntimeError "Unbound variable"))
    | App(e1,e2) -> 
	let (f,ctxtf) = interp ctxt e1 in
	let (a,ctxta) = interp ctxt e2 in
	(match f with
	  Lambda(x,e) -> interp (add ctxtf (x,a,ctxta)) e
	| _ -> raise (RuntimeError 
			"App applied to non Lambda"))
    | Lambda(x,e) -> (Lambda(x,e),ctxt)
  in
  (try (fst (interp (CTXT []) e))
  with (RuntimeError s) -> (print_string (s^"\n"); Num 0))

