type var = LambdaAst.var

let eqvar = LambdaAst.eqvar
let newvar = LambdaAst.newvar

type primop = Plus | Times | Minus
type expr = Var of var
  | Num of int
  | ZeroP of expr
  | IsZero of expr
  | Lambda of var * expr
  | App of expr * expr
  | Op of primop * expr * expr
  | True
  | False
  | If of expr * expr * expr

exception Not_found
let alpha_vary expr =
  let add ctxt str i = (str, i)::ctxt in
  let rec lookup x = function
      [] -> raise Not_found
    | (str, i)::rest -> 
	if x = str 
	then (str, i) 
	else lookup x rest
  in

  let bindnew (str,_) ctxt = 
    let (_, i) = newvar str in
    ((str, i), add ctxt str i)
  in
  
  let rec alpha = (fun ctxt -> 
    (function
	Var (str, i) -> 
	  (try Var(lookup str ctxt) 
	  with Not_found -> Var(str, i))
      | Num i -> Num i
      | ZeroP(e) -> ZeroP(alpha ctxt e)
      | IsZero(e) -> IsZero(alpha ctxt e)
      | Lambda (v, e)->
	  let (v', ctxt') = bindnew v ctxt in
	  Lambda(v', alpha ctxt' e)
      | App (e1, e2) -> App (alpha ctxt e1, alpha ctxt e2)
      | Op (p, e1, e2) -> Op (p, alpha ctxt e1, alpha ctxt e2)
      | True -> True
      | False -> False
      | If (b,e1,e2) -> 
	  If (alpha ctxt b, alpha ctxt e1, alpha ctxt e2)))
  in
  alpha [] expr

let alpha_equiv expr =
  let add ctxt i j = (i, j)::ctxt in
  let rec lookup x = function
      [] -> raise Not_found
    | (i, j)::rest ->
	if x = i 
	then j 
	else lookup x rest
  in

  let bindnew (_,i) (_,j) ctxt = add ctxt i j in

  let rec alpha ctxt e1 e2=
    (match (e1, e2) with
      (Var (_, i), Var (_,j)) -> 
	(try (lookup i ctxt = j) with Not_found -> false)
    | (Num i, Num j) -> i = j
    | (ZeroP(e1), ZeroP(e2)) -> alpha ctxt e1 e2
    | (IsZero(e1), IsZero(e2)) -> alpha ctxt e1 e2
    | (Lambda (v1,e1), Lambda (v2,e2))-> 
	let ctxt' = bindnew v1 v2 ctxt in
	alpha ctxt' e1 e2
    | (App (e11,e12), App(e21,e22))-> 
	(alpha ctxt e11 e21) &&
	(alpha ctxt e12 e22)
    | (Op (p1,e11,e12), Op (p2,e21,e22))-> 
	(p1 = p2) &&
	(alpha ctxt e11 e21) &&
	(alpha ctxt e12 e22)
    | (True, True) -> true
    | (False, False) -> true
    | (If (b1,e11,e12), If (b2,e21,e22)) -> 
	(alpha ctxt b1 b2) &&
	(alpha ctxt e11 e21) &&
	(alpha ctxt e12 e22) 
    | _ -> false)
  in
  alpha [] expr

exception ConvertFailure of string

let rec convert = function 
    LambdaAst.Var v -> Var v
  | LambdaAst.Num n -> Num n
  | LambdaAst.Lambda ([v],e) -> Lambda(v, convert e)
  | LambdaAst.Lambda _ -> 
      raise (ConvertFailure "Lambda has > 1 arguments")
  | LambdaAst.App [e1;e2] -> App(convert e1, convert e2)
  | LambdaAst.App [e0;e1;e2] ->
      let primop = match e0 with
	LambdaAst.Op p -> 
	  (match p with
	    LambdaAst.Plus -> Plus
	  | LambdaAst.Times -> Times
	  | LambdaAst.Minus -> Minus
	  | _ -> raise (ConvertFailure
		"App head is not arithmetic"))
      | _ -> raise (ConvertFailure
	    "App head is not primop")
      in
      Op (primop, convert e1, convert e2)
  | LambdaAst.App _ -> raise (ConvertFailure
      "App has > 2 expressions")
  | LambdaAst.Op LambdaAst.ZeroP ->
      let z = newvar "z"
      in 
      Lambda(z, ZeroP(Var z))
  | LambdaAst.Op _ -> raise (ConvertFailure
      "Op not in simple language")
  | LambdaAst.Pair _ -> raise (ConvertFailure
      "Pair not in simple language")
  | LambdaAst.Case _ -> raise (ConvertFailure
      "Case not in simple language")
  | LambdaAst.Left _ -> raise (ConvertFailure
      "Left not in simple language")
  | LambdaAst.Right _ -> raise (ConvertFailure
      "Right not in simple language")
  | LambdaAst.Letrec _ -> raise (ConvertFailure
      "Letrec not in simple language")

let rec thunk = function 
    Num n -> Num n
  | ZeroP e -> ZeroP (thunk e)
  | IsZero e -> IsZero (thunk e)
  | Op (primop, e1, e2) -> Op (primop, thunk e1, thunk e2)
  | True -> True
  | False -> False
  | If (e0, e1, e2) -> If (thunk e0, thunk e1, thunk e2)
  | Var x -> App(Var x, Num 0)
  | Lambda(x, e) -> Lambda(x, thunk e)
  | App(e1, e2) ->
      let u = newvar "u"
      in 
      App(thunk e1, Lambda(u, thunk e2))

let fromLambdaAst x = thunk (convert x)
let transTrue = fromLambdaAst (LambdaTranslate.transTrue)
let transFalse = fromLambdaAst (LambdaTranslate.transFalse)

let rec toLambdaAst = function
    Var v -> LambdaAst.Var v
  | Num i -> LambdaAst.Num i
  | ZeroP(e) -> 
      LambdaAst.App [LambdaAst.Op LambdaAst.ZeroP;
		     toLambdaAst e]
  | IsZero(e) ->
      LambdaAst.App [toLambdaAst 
		       (fromLambdaAst 
			  (LambdaAst.Op LambdaAst.ZeroP)); 
		     toLambdaAst e]
  | Lambda (v,e) -> LambdaAst.Lambda ([v], toLambdaAst e)
  | App (e1,e2) -> LambdaAst.App [toLambdaAst e1; toLambdaAst e2]
  | Op (p,e1,e2) -> 
      let primop = match p with 
	Plus -> LambdaAst.Plus
      | Times -> LambdaAst.Times
      | Minus -> LambdaAst.Minus
      in
      LambdaAst.App [LambdaAst.Op primop;
		   toLambdaAst e1;
		   toLambdaAst e2]
  | True -> toLambdaAst (fromLambdaAst LambdaTranslate.transTrue)
  | False -> toLambdaAst (fromLambdaAst LambdaTranslate.transFalse)
  | If (b,e1,e2) -> 
      LambdaAst.App
      [LambdaAst.App
	 [LambdaAst.App 
	    [LambdaTranslate.translate LambdaAst.astIf;
	     toLambdaAst b];
	  toLambdaAst e1];
       toLambdaAst e2]

let ftLambdaAst x= toLambdaAst ( fromLambdaAst x)
