type var = string * int

let eqvar (s1, i1) (s2, i2) = i1 = i2

type primop = 
    Plus                   
    | Times
    | Minus
    | ZeroP
    | Fst
    | Snd

type expr =
    Var of var
    | Num of int
    | Op of primop
    | Lambda of var list * expr
    | App of expr list
    | Pair of expr * expr
    | Case of expr * (var * expr) * (var * expr)
    | Left of expr
    | Right of expr
    | Letrec of (var * expr) list * expr


exception Not_found
let vctr = ref 0
let newvar s = 
  let v = !vctr in
  let _ = vctr := v+1 in
  (s, v)


let alpha_vary expr =
  let add ctxt str i = (str, i)::ctxt in

  let rec lookup x = function
      [] -> raise Not_found
    | ((str, i)::rest) -> 
	if x = str then 
	  (str, i) 
	else 
	  lookup x rest 
  in

  let bindnew vl ctxt=
    List.fold_right
      (fun (str,_) (vl', ctxt) -> 
	let (_, i) = newvar str in
	((str, i)::vl', add ctxt str i))
      vl ([], ctxt)
  in
  
  let rec alpha ctxt e = 
    match e with
      Var (str, i) -> (try Var(lookup str ctxt) with Not_found -> Var(str, i))
    | Lambda (vl, e) -> 
	let (vl', ctxt') = bindnew vl ctxt in
	Lambda(vl', alpha ctxt' e)
    | App el -> App(List.map (alpha ctxt) el)
    | Pair(e1,e2) -> Pair((alpha ctxt) e1, (alpha ctxt) e2)
    | Letrec (vel, e) -> 
	let (vl, el) = List.split vel in
	let (vl', ctxt') = bindnew vl ctxt in
	let el' = List.map (alpha ctxt') el in
	let (vel') = List.combine vl' el' in
	Letrec(vel', alpha ctxt' e)
    | Left e -> Left ((alpha ctxt) e)
    | Right e -> Right ((alpha ctxt) e)
    | Case (e,(v1,e1),(v2,e2)) -> 
	let e' = alpha ctxt e in
	let ([v1'],ctxt1) = bindnew [v1] ctxt in
	let e1' = alpha ctxt1 e1 in
	let ([v2'],ctxt2) = bindnew [v2] ctxt in
	let e2' = alpha ctxt2 e2 in 
	Case(e',(v1',e1'),(v2',e2'))
    | e -> e
  in
  alpha [] expr

let alpha_equiv expr =
  let add ctxt i j = (i, j)::ctxt in
  let rec lookup x = function
      [] -> raise Not_found
    | (i, j)::rest -> if x = i then j else
      lookup x rest
  in

  let bindnew vl1 vl2 ctxt =
    List.fold_right
    (fun ((_,i),(_,j)) (ctxt) -> add ctxt i j)
    (List.combine vl1 vl2) ctxt
  in
          
  let rec alpha ctxt e1 e2 =
    match (e1,e2) with
      (Var (_, i),Var(_,j)) -> 
	(try (lookup i ctxt = j) with Not_found -> false)
    | (Lambda(vl1, e1),Lambda(vl2,e2)) -> 
	let ctxt' = bindnew vl1 vl2 ctxt in
	alpha ctxt' e1 e2 
    | (App el1,App el2) -> 
	List.fold_left (fun y (x1, x2) -> 
	  (alpha ctxt x1 x2)&& y)
	  true
	  (List.combine el1 el2)
    | (Pair(e1,e1'),Pair(e2,e2')) -> 
	alpha ctxt e1 e2 && alpha ctxt e1' e2' 
    | (Letrec (vel1, e1),Letrec(vel2,e2)) -> 
	let (vl1, el1) = List.split vel1 in
	let (vl2,el2) = List.split vel2 in
	let (ctxt') = bindnew vl1 vl2 ctxt in
	List.fold_left 
	  (fun y (x1,x2) -> (alpha ctxt x1 x2) && y)
	   true
	  (List.combine el1 el2)
    | (Left e1,Left e2) -> alpha ctxt e1 e2
    | (Right e1, Right e2) -> alpha ctxt e1 e2
    | (Case (e1,(v11,e11),(v12,e12)),Case(e2,(v21,e21),(v22,e22))) -> 
	alpha ctxt e1 e2 
          &&
        alpha (bindnew [v11] [v21] ctxt) e11 e21
          &&
        alpha (bindnew [v12] [v22] ctxt) e12 e22
    | (e1,e2) -> e1=e2
  in
  alpha [] expr

let astTrue = Left(Num 0)

let astFalse = Right(Num 0)
    
let astIf = 
  let e = newvar "e"  in
  let e1 = newvar "e1" in
  let e2 = newvar "e2"
  in Lambda([e;e1;e2],
	    Case(Var e,
		 (newvar "t",Var e1),(newvar "f",Var e2)))

let astError = App[Num 0; Num 0]

let astHead = 
  let l = newvar "l" in
  let h = newvar "h" in
  let err = newvar "error" in
  Lambda([l],Case(Var l,(err,astError),
                  (h,App [(Op Fst);Var h])))

let astTail = 
  let l = newvar "l" in
  let t = newvar "t" in
  let err = newvar "error" in
  Lambda([l],Case(Var l,(err,astError),
                  (t,App[Op Snd;Var t])))

let astNilP = 
  let l = newvar "l"
  in 
  Lambda([l],App [astIf; Var l; astTrue; astFalse])

let astCons = 
  let h = newvar "h" in
  let t = newvar "t" in
  Lambda([h;t],Right(Pair(Var h, Var t)))

let astNil = Left(Num 0)
