package rubik.geometry;

/**
 * Representation of a point or vector in 3-dimensional Euclidean space, along
 * with several useful operations on them.
 */
public class ThreeDPoint {
   
   public double x;
   public double y;
   public double z;

   public static final ThreeDPoint ZERO = new ThreeDPoint(0, 0, 0);

   public ThreeDPoint(double x, double y, double z) {
      this.x = x;
      this.y = y;
      this.z = z;
   }

   public ThreeDPoint plus(ThreeDPoint p) {
      return new ThreeDPoint(x + p.x, y + p.y, z + p.z);
   }

   public ThreeDPoint minus(ThreeDPoint p) {
      return new ThreeDPoint(x - p.x, y - p.y, z - p.z);
   }
   
   public double dotProduct(ThreeDPoint p) {
      return (x * p.x + y * p.y + z * p.z);
   }

   public double length() {
      return Math.sqrt(dotProduct(this));
   }

   public double cos(ThreeDPoint p) {
      return (dotProduct(p) / length() * p.length());
   }

   // project relative to a given
   public TwoDPoint project(Matrix orientation, ThreeDPoint viewerPosition) {
      return orientation.transpose().mult(this).project(viewerPosition);
   }
   
   // project this onto 2D space from point of view determined by given viewer position
   // used for rendering objects on the screen
   public TwoDPoint project(ThreeDPoint viewerPosition) {
      double retinalDistance = 1;
      double alpha = retinalDistance / (viewerPosition.x - x);
      return new TwoDPoint(y * alpha, z * alpha);
   }
   
   // does the given 3D point occur on the positive side of the plane
   // determined by this normal vector?
   public boolean behind(ThreeDPoint p) {
      assert !this.equals(ThreeDPoint.ZERO);
      return p.minus(this).dotProduct(this) > 0;
   }

   // is the point behind the viewer?
   public boolean visible(ThreeDPoint viewerPosition) {
      final ThreeDPoint forward = new ThreeDPoint(-0.05, 0, 0);
      return (forward.dotProduct(minus(viewerPosition.plus(forward))) >= 0);
   }

   // scalar product
   public ThreeDPoint scale(double d) {
      return new ThreeDPoint(x*d, y*d, z*d);
   }
   
   // interpolate linearly between this and the given point   
   public ThreeDPoint interpolate(ThreeDPoint t, Double a) {
      return new ThreeDPoint(a*x + (1 - a)*t.x, a*y + (1 - a)*t.y, a*z + (1 - a)*t.z);
   }
   
   // round to nearest integral point   
   public void round() {
      x = Math.rint(x);
      y = Math.rint(y);
      z = Math.rint(z);
   }

   public int hashCode() {
      return (int)(x + 3*y + 7*z);
   }
   
   public boolean equals(Object o) {
      try {
         ThreeDPoint op = (ThreeDPoint)o;
         return x == op.x && y == op.y && z == op.z;
      } catch (Exception e) {
         return false;
      }
   }
   
   public String toString() {
      return String.format("[%.0f %.0f %.0f]", x, y, z);
   }

}