package rubik;

import java.util.ArrayList;
import java.util.EmptyStackException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.Stack;

import rubik.Cube.Type;
import rubik.geometry.Matrix;
import rubik.geometry.ThreeDPoint;
import rubik.geometry.TwoDPoint;

/**
 * The state of the animation.
 */
public class State {
   
   final Clip click = new Clip("click.wav");
   final Clip tada = new Clip("tada.wav");
   
   final Random random = new Random();
   
   // for undoing
   final Stack<Move> history = new Stack<Move>();
   
   double rAngle = 0; // nonzero if a face is rotating
   Move rMove = Move.FL; // current move
   boolean solving = false; // true if we are solving the cube
   boolean undoing = false; // true if we are undoing last move
   boolean brake = false; // true if shift key is depressed
   double distanceVelocity = 0; // nonzero if we are zooming in or out
   
   // orientation matrix and helper matrices
   Matrix orientation = new Matrix();
   Matrix gamma = new Matrix();
   Matrix delta = new Matrix();
   
   // velocity of rotation
   TwoDPoint velocity = new TwoDPoint(0, 0);
   
   // distance of viewer from the cube
   double viewerDistance = 13;
   
   // position of viewer
   ThreeDPoint viewerPosition = new ThreeDPoint(viewerDistance, 0, 0);
   
   // position of light source - determines shadow effect
   static final ThreeDPoint BASE_LIGHT_SOURCE = new ThreeDPoint(10, 20, -30);
   
   static final int INIT_SPEED = 32;
   static final int MAX_SPEED = 64;
   static final int MIN_SPEED = 1;
   double rAngleDelta = setSpeed(INIT_SPEED);
   
   private static final Matrix permuteAxes = new Matrix(
            new ThreeDPoint(0,0,1),
            new ThreeDPoint(1,0,0),
            new ThreeDPoint(0,1,0));
   
   private static Matrix xMatrix(double angle) {
      return new Matrix(
               new ThreeDPoint(1, 0, 0),
               new ThreeDPoint(0, Math.cos(angle), Math.sin(angle)),
               new ThreeDPoint(0, -Math.sin(angle), Math.cos(angle)));
   }
   
   private static Matrix yMatrix(double angle) {
      return permuteAxes.mult(xMatrix(angle).mult(permuteAxes.transpose()));
   }
   
   private static Matrix zMatrix(double angle) {
      return permuteAxes.transpose().mult(xMatrix(angle).mult(permuteAxes));
   }
   
   static Matrix rMatrix(Plane plane, double angle) {
      switch (plane.ordinal()) {
         case 0: return xMatrix(angle);
         case 1: return xMatrix(angle).transpose();
         case 2: return yMatrix(angle);
         case 3: return yMatrix(angle).transpose();
         case 4: return zMatrix(angle);
         default: return zMatrix(angle).transpose();
      }
   }
   
   Matrix rMatrix = rMatrix(Plane.FRONT, 0);
   
   // is the given cubelet currently rotating?
   boolean rotating(Cube cube) {
      return rAngle != 0 && inRotationPlane(cube, rMove.getPlane());
   }
   
   // is the given cube in the given rotation plane?
   static boolean inRotationPlane(Cube cube, Plane plane) {
      return plane.normal.behind(cube.center);
   }
   
   // index planes by their normal vectors
   private static Map<ThreeDPoint, Plane> planes = new HashMap<ThreeDPoint, Plane>();

   // the six planes
   enum Plane {
      FRONT, BACK, RIGHT, LEFT, DOWN, UP;
      
      private ThreeDPoint normal;
      private Matrix matrix;
      
      Plane() {
         normal = Cube.normals[ordinal()];
         matrix = rMatrix(this, Math.PI/2);
         matrix.round();
         planes.put(normal, this);
      }
      
      // interpret plane name relative to given orientation
      Plane getRelative(Matrix m) {
         ThreeDPoint p = m.mult(normal);
         Plane closest = this;
         for (Plane q : Plane.values()) {
            if (p.cos(q.normal) > p.cos(closest.normal)) closest = q;
         }
         return closest;
      }

      Matrix getMatrix() {
         return matrix;
      }
      
      static Plane getPlane(ThreeDPoint normal) {
         return planes.get(normal);
      }
   }
   
   // index moves by their planes and whether clockwise (true) or counterclockwise (false)
   private static Map<Plane, Map<Boolean, Move>> moves = new HashMap<Plane, Map<Boolean, Move>>();

   // the moves
   enum Move {
      FR(Plane.FRONT, true), BR(Plane.BACK, true), RR(Plane.RIGHT, true),
      LR(Plane.LEFT, true), DR(Plane.DOWN, true), UR(Plane.UP, true),
      FL(Plane.FRONT, false), BL(Plane.BACK, false), RL(Plane.RIGHT, false),
      LL(Plane.LEFT, false), DL(Plane.DOWN, false), UL(Plane.UP, false);
      
      private Plane plane;
      private boolean inverse;
      private Matrix matrix;
      
      Move(Plane plane, boolean inverse) {
         this.plane = plane;
         this.inverse = inverse;
         matrix = inverse? plane.getMatrix().transpose() : plane.getMatrix();
         if (inverse) moves.put(plane, new HashMap<Boolean, Move>());
         moves.get(plane).put(inverse, this);
      }
      
      Matrix getMatrix() {
         return matrix;
      }
      
      Plane getPlane() {
         return plane;
      }

      static Move move(Plane plane, boolean inverse) {
         return moves.get(plane).get(inverse);
      }

      Move getRelative(Matrix orientation) {
         return move(plane.getRelative(orientation), inverse);
      }

      Move inverse() {
         return move(plane, !inverse);
      }
      
      // f^{-1} o this o f
      Move conjugate(CubeGroup f) {
         ThreeDPoint p = f.inverse().apply(plane.normal);
         return move(Plane.getPlane(p), inverse);
      }
      
      static List<Move> conjugateSequence(Iterable<Move> moves, CubeGroup f) {
         List<Move> clist = new ArrayList<Move>();
         for (Move m : moves) clist.add(m.conjugate(f));
         return clist;
      }
   }
   
   //update state for next display
   void update() {
      // update orientation matrix
      double theta = Math.atan2(velocity.y, velocity.x);
      double r = velocity.x * velocity.x + velocity.y * velocity.y;
      gamma.y.y = Math.cos(theta);
      gamma.y.z = -Math.sin(theta);
      gamma.z.y = Math.sin(theta);
      gamma.z.z = Math.cos(theta);
      delta.x.x = Math.cos(r);
      delta.x.y = -Math.sin(r);
      delta.y.x = Math.sin(r);
      delta.y.y = Math.cos(r);
      orientation = orientation.mult(gamma.mult(delta.mult(gamma.transpose())));

      // update viewer distance
      viewerDistance += distanceVelocity;
      viewerPosition.x = viewerDistance;
      
      // update rotation plane
      if (rAngle == 0) return;
      if (Math.abs(rAngle) >= Math.PI/2) {
         stopRotation();
      } else {
         rAngle += Math.signum(rAngle) * rAngleDelta;
         rMatrix = rMatrix(rMove.getPlane(), rAngle);
      }
   }
   
   // solve the cube
   int solve() {
      if (solving || rAngle > 0) return 0;
      solving = true;
      List<Move> solution = new Planner(this).solve();
      history.clear();
      for (int i = solution.size() - 1; i >= 0; i--) {
         history.push(solution.get(i));
      }
      if (history.isEmpty()) return 0;
      Move move = history.pop();
      startRotation(move, true);
      return solution.size();
   }

   // solve without animation
   int solveImmediately() {
      if (solving || rAngle > 0) return 0;
      List<Move> solution = new Planner(this).solve();
      history.clear();
      moves(solution);
      return solution.size();
   }

   // rotation is when we are rotating a plane
   void startRotation(Move move, boolean ignoreOrientation) {
      if (rAngle != 0) return; // already rotating
      rMove = ignoreOrientation? move : move.getRelative(orientation);
      rAngle = move.inverse? -rAngleDelta : rAngleDelta;
      rMatrix = rMatrix(rMove.getPlane(), rAngle);
   }

   void stopRotation() {
      move(rMove);
      rAngle = 0;
      rMatrix = Matrix.IDENTITY;
      click.play();
      if (solving) {
         String s = String.format("%d moves remaining", history.size());
         new MessageEvent(s).fire();
      }
      if (solved()) {
         history.clear();
         undoing = false;
         solving = false;
         tada.play();
      } else if (undoing) {
         undoing = false;
         solving = false;
      } else if (solving) {
         if (history.isEmpty()) solving = false;
         else startRotation(history.pop(), true);
      } else history.add(rMove.inverse());
   }
   
   // make one of the 12 moves
   void move(Move move) {
      Matrix m = move.getMatrix();
      CubeGroup cg = CubeGroup.elements.get(m);
      for (Cube c : Cube.cubes.values()) {
         c.oldPosition = c.position;
      }
      for (Cube c : Cube.cubes.values()) {
         if (inRotationPlane(c, move.getPlane())) {
            Cube source = Cube.cubes.get(m.transpose().mult(c.center));
            c.position = source.oldPosition.mult(cg.inverse());
         }
      }
   }
   
   // make a sequence of moves
   void moves(Iterable<Move> moves) {
      for (Move move : moves) {
         move(move);
         history.add(move.inverse());
      }
   }
   
   // undo last move
   void undo() throws EmptyStackException {
      if (solving) return;
      Move move = history.pop();
      undoing = true;
      startRotation(move, true);
   }
   
   // check if cube is solved
   boolean solved() {
      for (Cube c : Cube.cubes.values()) {
         if (c.type == Type.CENTER || c.type == Type.FACE) continue;
         if (!c.position.equals(CubeGroup.IDENTITY)) return false;
      }
      return true;
   }

   // reset cube to the solved state
   void reset() {
      for (Cube c : Cube.cubes.values()) {
         c.position = CubeGroup.elements.get(Matrix.IDENTITY);
      }
      history.clear();
      solving = false;
      undoing = false;
   }

   double setSpeed(int value) {
      rAngleDelta = Math.PI*value/1024;
      return rAngleDelta;
   }
   
   // get the Rubik group element associated with current state
   RubikGroup getGroup() {
      return new RubikGroup(this);
   }
   
   /*
    * Debugging tools
    */
   
   String debug() {
      RubikGroup rg = new RubikGroup(history);
      Set<Cube> s = new java.util.HashSet<Cube>();
      for (Cube c : Cube.cubes.values()) {
         if (c.type == Type.CORNER || c.type == Type.EDGE) s.add(c);
      }
      Set<Set<Cube>> orbits = rg.orbits(s);
      String st = "";
      for (Set<Cube> o : orbits) {
         st  += " " + o.size();
      }
      return st;

   }
   
   // solve cube on a random 20-move scramble
   String test() {
      reset();
      int next = 0, last = 0;
      for (int i = 0; i < 20; i++) {
         while (next == last) {
            next = random.nextInt(Move.values().length);
         }
         last = next;
         Move move = Move.values()[next];
         move(move);
         history.add(move.inverse());
      }
      int m = solveImmediately();
      return m + " moves";
   }
}
