package rubik;

import java.awt.Color;
import java.awt.Polygon;
import java.util.HashMap;
import java.util.Map;

import rubik.Cube.Type;
import rubik.geometry.ThreeDPoint;

/**
 * A Square is one of the six faces of a cubelet.
 */
public class Square {
   
   // index Squares by their centers
   static Map<ThreeDPoint, Square> elements = new HashMap<ThreeDPoint, Square>();
   
   Cube cube;
   ThreeDPoint center;
   ThreeDPoint normal; // points perpendicularly outward from the square
   Color color;
   
   static final ThreeDPoint[][] baseCorners = {
            { new ThreeDPoint(0, 1, 1), new ThreeDPoint(0, -1, 1),
              new ThreeDPoint(0, -1, -1), new ThreeDPoint(0, 1, -1) },
            { new ThreeDPoint(1, 0, 1), new ThreeDPoint(-1, 0, 1),
              new ThreeDPoint(-1, 0, -1), new ThreeDPoint(1, 0, -1) },
            { new ThreeDPoint(1, 1, 0), new ThreeDPoint(-1, 1, 0),
              new ThreeDPoint(-1, -1, 0), new ThreeDPoint(1, -1, 0) }
   };
   
   ThreeDPoint[] corners = new ThreeDPoint[4]; // the larger square
   ThreeDPoint[] subCorners = new ThreeDPoint[4]; // the smaller concentric square
   ThreeDPoint[] orientedCorners = new ThreeDPoint[4];
   
   Square(Cube cube, ThreeDPoint center, ThreeDPoint normal) {
      this.cube = cube;
      this.center = center;
      this.normal = normal;

      int cornersIndex;
      if (normal.x != 0) cornersIndex = 0;
      else if (normal.y != 0) cornersIndex = 1;
      else cornersIndex = 2;
      for (int i = 0; i < 4; i++) {
         corners[i] = center.plus(baseCorners[cornersIndex][i]);
         subCorners[i] = center.plus(baseCorners[cornersIndex][i].scale(.9));
      }
      elements.put(center, this);
      if (cube.type == Type.CENTER) {
         if (center.x != 0) BSPTreeNode.xRoot.insert(this);
         else if (center.y != 0) BSPTreeNode.yRoot.insert(this);
         else BSPTreeNode.zRoot.insert(this);
      } else {
         BSPTreeNode.xRoot.insert(this);
         BSPTreeNode.yRoot.insert(this);
         BSPTreeNode.zRoot.insert(this);
      }
   }
   
   // Returns a 2D polygon for screen display that is the projection of the
   // 3D internal model onto the screen. Position and appearance are determined
   // by the orientation matrix and other state parameters.
   // foreground parameter determines whether we are drawing the smaller
   // colored square or the larger black square.
   Polygon project(State state, boolean foreground) {
      
      Polygon p = new Polygon();

      for (int i = 0; i < 4; i++) {
         ThreeDPoint corner = foreground? subCorners[i] : corners[i];
         if (state.rotating(cube)) {
            corner = state.rMatrix.mult(corner);
         }
         orientedCorners[i] = state.orientation.transpose().mult(corner);
      }

      for (int i = 0; i < 4; i++) {
         if (orientedCorners[i].visible(state.viewerPosition)) {
            orientedCorners[i].project(state.viewerPosition).place(p);
         } else {
            ThreeDPoint s;
            if (orientedCorners[((i + 3) % 4)].visible(state.viewerPosition)) {
               s = truncate(orientedCorners[((i + 3) % 4)], orientedCorners[i], state.viewerPosition);
               s.project(state.viewerPosition).place(p);
            }
            if (orientedCorners[((i + 1) % 4)].visible(state.viewerPosition)) {
               s = truncate(orientedCorners[((i + 1) % 4)], orientedCorners[i], state.viewerPosition);
               s.project(state.viewerPosition).place(p);
            }
         }
      }
      return p;
   }

   ThreeDPoint truncate(ThreeDPoint s, ThreeDPoint t, ThreeDPoint vp) {
      ThreeDPoint forward = new ThreeDPoint(-0.05, 0, 0);
      vp = vp.plus(forward);
      double a = vp.minus(t).dotProduct(vp);
      double b = s.minus(t).dotProduct(vp);
      double alpha = (b == 0)? 0 : a / b;
      return s.interpolate(t, alpha);
   }

   // does the 3D point occur on the colored side?
   boolean behind(ThreeDPoint p) {
      return p.minus(center).dotProduct(normal) > 0;
   }

   void setColor(Color color) {
      this.color = color;
   }
}