package rubik;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import rubik.State.Move;
import rubik.geometry.Matrix;

/**
 * The Rubik group is the group of reachable states of the Rubik cube.
 * It has 8!*3^7 * 12!*2^11 * 2 = 173,008,013,097,959,424,000 elements.
 * Each element is represented by 20 cube group elements, one for
 * each corner and edge cubelet, however not all such states are
 * reachable.  
 */
public class RubikGroup {

   Map<Cube, CubeGroup> dest = new HashMap<Cube, CubeGroup>();
   
   static Map<Move, RubikGroup> generators = new HashMap<Move, RubikGroup>();
   static RubikGroup IDENTITY = new RubikGroup();
   
   // initialize generators
   static {
      for (Move move : Move.values()) {
         generators.put(move, new RubikGroup(move));
      }
   }
   
   // representative sequence of generators
   RubikGroup(Iterable<Move> moves) {
      this(); // start with identity
      for (Move move : moves) {
         compose(generators.get(move));
      }
   }
   
   // from current position of cubes
   RubikGroup(State state) {
      Iterable<Cube> cubes = Cube.cubes.values();
      for (Cube c : cubes) {
         if (c.type != Cube.Type.CORNER && c.type != Cube.Type.EDGE) continue;
         Cube source = c.position.apply(c);
         dest.put(source, c.position.inverse());
      }
      assert dest.size() == 20;
   }
   
   RubikGroup(Move move) {
      Matrix m = move.getMatrix();
      CubeGroup cg = CubeGroup.elements.get(m);
      for (Cube c : Cube.cubes.values()) {
         if (c.type != Cube.Type.CORNER && c.type != Cube.Type.EDGE) continue;
         dest.put(c, State.inRotationPlane(c, move.getPlane())? cg : CubeGroup.IDENTITY);
      }
      assert dest.size() == 20;
   }
   
   // group identity
   private RubikGroup() {
      for (Cube c : Cube.cubes.values()) {
         if (c.type != Cube.Type.CORNER && c.type != Cube.Type.EDGE) continue;
         dest.put(c, CubeGroup.IDENTITY);
      }
      assert dest.size() == 20;
   }
   
   // clone
   private RubikGroup(RubikGroup rg) {
      dest.putAll(rg.dest);
   }
   
   Cube apply(Cube c) {
      return dest.get(c).apply(c);
   }
   
   public int hashCode() {
      int hc = 11;
      for (CubeGroup cg : dest.values()) {
         hc += 7 * (hc++) * cg.hashCode();
      }
      return  hc;
   }
   
   public boolean equals(Object obj) {
      try {
         RubikGroup rg = (RubikGroup)obj;
         for (Cube c : dest.keySet()) {
            if (!dest.get(c).equals(rg.dest.get(c))) return false;
         }
         return true;
      } catch (Exception e) {
         return false;
      }
   }
   
   // compose this action with the given action
   // this is done first, followed by given action
   // destructive of this
   public void compose(RubikGroup rg) {
      for (Cube c : dest.keySet()) {
         CubeGroup cg1 = dest.get(c);
         CubeGroup cg2 = rg.dest.get(cg1.apply(c));
         dest.put(c, cg2.mult(cg1));
      }
   }

   // compose this action with the given move
   // this is done first, followed by given move
   // destructive of this
   public void compose(Move m) {
      compose(RubikGroup.generators.get(m));
   }

   public RubikGroup inverse() {
      RubikGroup rg = new RubikGroup();
      for (Cube c : dest.keySet()) {
         CubeGroup cg = dest.get(c);
         rg.dest.put(cg.apply(c), cg.inverse());
      }
      return rg;
   }
   
   // f o this o f^{-1}
   public RubikGroup conjugate(RubikGroup f) {
      RubikGroup fm = f.inverse();
      fm.compose(this);
      fm.compose(f);
      return fm;
   }

   public int order() {
      RubikGroup accum = new RubikGroup(this); // accumulated product
      int n;
      for (n = 1; !accum.equals(IDENTITY); n++) {
         accum.compose(this);
      }
      return n;
   }
   
   // check parity of this permutation on given set of cubes
   // false = even, true = odd
   // set must be closed under the given permutation
   boolean parity(Set<Cube> s) {
      boolean b = false;
      Set<Set<Cube>> orbits = orbits(s);
      for (Set<Cube> t : orbits) {
         if (t.size() % 2 == 0) b = !b;
      }
      return b; 
   }
   
   // find the orbits of this permutation on given set of cubes
   // set must be closed under the given permutation
   Set<Set<Cube>> orbits(Set<Cube> s) {
      Set<Set<Cube>> orbits = new HashSet<Set<Cube>>();
      while (!s.isEmpty()) {
         Set<Cube> t = new HashSet<Cube>();
         Cube c = s.iterator().next();
         s.remove(c);
         t.add(c);
         Cube d = apply(c);
         while (d != c) {
            assert s.contains(d);
            s.remove(d);
            t.add(d);
            d = apply(d);
         }
         orbits.add(t);
      }
      return orbits;
   }

}
