package rubik;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import rubik.State.Move;

/**
 * Solution planner
 * Plans the solution from any given starting position. 
 */
public class Planner {
   
   final State state;
   
   /*
    * These arrays of moves specify solutions to move a cubelet to its home position,
    * preserving cubelets already solved.
    */
   
   // edge cubelets whose home position is on top horizontal (z) plane
   static final Move[][] topEdgeSolutions = {
      // dest is on top face
      {},
      { Move.FL, Move.UR, Move.LL, Move.UL },
      { Move.RL, Move.UL, Move.RR, Move.UR },
      { Move.RL, Move.FL },
      { Move.LL, Move.UR, Move.LR, Move.UL },
      { Move.LR, Move.FR },
      { Move.BL, Move.UL, Move.UL, Move.BR, Move.UR, Move.UR },
      { Move.BL, Move.UL, Move.RL, Move.UR },
      
      // dest is on middle face
      { Move.FL },
      { Move.FR },
      { Move.UL, Move.RL, Move.UR },
      { Move.UL, Move.RR, Move.UR },
      { Move.UR, Move.LL, Move.UL },
      { Move.UR, Move.LR, Move.UL },
      { Move.UL, Move.UL, Move.BL, Move.UR, Move.UR },
      { Move.UL, Move.UL, Move.BR, Move.UR, Move.UR },
      
      // dest is on bottom face
      { Move.FL, Move.FL },
      { Move.DL, Move.FL, Move.FL },
      { Move.DR, Move.FL, Move.FL },
      { Move.DL, Move.DL, Move.FL, Move.FL },
      { Move.FL, Move.UL, Move.RR, Move.UR },
      { Move.DL, Move.FL, Move.UL, Move.RR, Move.UR },
      { Move.DR, Move.FL, Move.UL, Move.RR, Move.UR },
      { Move.DL, Move.DL, Move.FL, Move.UL, Move.RR, Move.UR }
   };

   // corner cubelets whose home position is on top horizontal (z) plane
   static final Move[][] topCornerSolutions = {
      // dest is on top face
      {},
      { Move.FR, Move.DL, Move.DL, Move.FL, Move.RL, Move.DL, Move.DL, Move.RR },
      { Move.RL, Move.DL, Move.DL, Move.RR, Move.FR, Move.DL, Move.DL, Move.FL },
      { Move.FR, Move.BL, Move.DL, Move.FL, Move.BR },
      { Move.RR, Move.DR, Move.RL, Move.FR, Move.DR, Move.DR, Move.FL },
      { Move.RR, Move.DR, Move.RL, Move.RL, Move.DL, Move.DL, Move.RR },
      { Move.RL, Move.LR, Move.DR, Move.RR, Move.LL },
      { Move.FL, Move.DL, Move.FR, Move.RL, Move.DL, Move.DL, Move.RR },
      { Move.FL, Move.DL, Move.FR, Move.FR, Move.DR, Move.DR, Move.FL },
      { Move.FR, Move.BR, Move.DL, Move.DL, Move.FL, Move.BL },
      { Move.LL, Move.DL, Move.LR, Move.FR, Move.DL, Move.FL },
      { Move.BR, Move.DR, Move.BL, Move.RL, Move.DR, Move.RR },

      // dest is on bottom face
      { Move.FR, Move.DL, Move.FL },
      { Move.FR, Move.DL, Move.DL, Move.FL },
      { Move.DL, Move.FR, Move.DL, Move.DL, Move.FL },
      { Move.DR, Move.FR, Move.DL, Move.FL },
      { Move.RL, Move.DR, Move.RR },
      { Move.RL, Move.DR, Move.DR, Move.RR },
      { Move.DR, Move.RL, Move.DR, Move.DR, Move.RR },
      { Move.DL, Move.RL, Move.DR, Move.RR },
      { Move.FR, Move.DL, Move.FL, Move.RL, Move.DL, Move.DL, Move.RR, Move.FR, Move.DL, Move.DL, Move.FL },
      { Move.FR, Move.DL, Move.DL, Move.FL, Move.RL, Move.DL, Move.DL, Move.RR, Move.FR, Move.DL, Move.DL, Move.FL },
      { Move.RL, Move.DR, Move.RR, Move.FR, Move.DL, Move.DL, Move.FL, Move.RL, Move.DL, Move.DL, Move.RR },
      { Move.DL, Move.RL, Move.DR, Move.RR, Move.FR, Move.DL, Move.DL, Move.FL, Move.RL, Move.DL, Move.DL, Move.RR }
   };
   
   // edge cubelets whose home position is in middle horizontal (z) plane
   static final Move[][] sideSolutions = {
      // dest is on bottom face
      { Move.RL, Move.DR, Move.RR, Move.DR, Move.FR, Move.DL, Move.FL },
      { Move.FR, Move.DL, Move.FL, Move.DL, Move.RL, Move.DR, Move.RR },
      { Move.DL, Move.RL, Move.DR, Move.RR, Move.DR, Move.FR, Move.DL, Move.FL },
      { Move.DL, Move.FR, Move.DL, Move.FL, Move.DL, Move.RL, Move.DR, Move.RR },
      { Move.DR, Move.RL, Move.DR, Move.RR, Move.DR, Move.FR, Move.DL, Move.FL },
      { Move.DR, Move.FR, Move.DL, Move.FL, Move.DL, Move.RL, Move.DR, Move.RR },
      { Move.DL, Move.DL, Move.RL, Move.DR, Move.RR, Move.DR, Move.FR, Move.DL, Move.FL },
      { Move.DL, Move.DL, Move.FR, Move.DL, Move.FL, Move.DL, Move.RL, Move.DR, Move.RR },
      
      // dest is on side face
      {},
      { Move.FR, Move.DR, Move.FL, Move.DL, Move.RL, Move.DL, Move.RR, Move.DL,
        Move.FR, Move.DL, Move.FL, Move.DL, Move.RL, Move.DR, Move.RR },
      { Move.FL, Move.DL, Move.FR, Move.DR, Move.LR, Move.DR, Move.LL, Move.DR,
        Move.FR, Move.DL, Move.FL, Move.DL, Move.RL, Move.DR, Move.RR },
      { Move.LR, Move.DR, Move.LL, Move.DL, Move.FL, Move.DL, Move.FR,
        Move.FR, Move.DL, Move.FL, Move.DL, Move.RL, Move.DR, Move.RR },
      { Move.LL, Move.DL, Move.LR, Move.DR, Move.BR, Move.DR, Move.BL, Move.DR, Move.DR,
        Move.FR, Move.DL, Move.FL, Move.DL, Move.RL, Move.DR, Move.RR },
      { Move.BR, Move.DR, Move.BL, Move.DL, Move.LL, Move.DL, Move.LR, Move.DR,
        Move.FR, Move.DL, Move.FL, Move.DL, Move.RL, Move.DR, Move.RR },
      { Move.BL, Move.DL, Move.BR, Move.DR, Move.RR, Move.DR, Move.RL, Move.DL,
        Move.FR, Move.DL, Move.FL, Move.DL, Move.RL, Move.DR, Move.RR },
      { Move.RR, Move.DR, Move.RL, Move.DL, Move.BL, Move.DL, Move.BR, Move.DR, Move.DR,
        Move.FR, Move.DL, Move.FL, Move.DL, Move.RL, Move.DR, Move.RR }
   };
   
   // edge cubelets whose home position is on bottom horizontal (z) plane
   static final Move[][] bottomEdgeSolutions = {
      {},
      { Move.RR, Move.DR, Move.FR, Move.DL, Move.FL, Move.RL },
      { Move.RR, Move.FR, Move.DR, Move.FL, Move.DL, Move.RL },
      { Move.LL, Move.DL, Move.FL, Move.DR, Move.FR, Move.LR },
      { Move.LL, Move.FL, Move.DL, Move.FR, Move.DR, Move.LR },
      { Move.FR, Move.DR, Move.LR, Move.DL, Move.LL, Move.FL, Move.RR, Move.DR, Move.FR, Move.DL, Move.FL, Move.RL },
      { Move.RR, Move.DR, Move.FR, Move.DL, Move.FL, Move.RL, Move.LL, Move.DL, Move.FL, Move.DR, Move.FR, Move.LR },
      { Move.BL, Move.LL, Move.DL, Move.LR, Move.DR, Move.BR, Move.RR, Move.DR, Move.FR, Move.DL, Move.FL, Move.RL },
   };
   
   static final List<Move> bottomCross = Arrays.asList(
      Move.RR, Move.DR, Move.FR, Move.DL, Move.FL, Move.RL, Move.LL, Move.FL, Move.DL,
      Move.FR, Move.DR, Move.LR, Move.RR, Move.DR, Move.FR, Move.DL, Move.FL, Move.RL
   );

   // corner cubelets whose home position is on bottom horizontal (z) plane
   static final List<Move> bottomTri = Arrays.asList(
      Move.FR, Move.BL, Move.DL, Move.BR, Move.DR, Move.FL, Move.DL, Move.BL, Move.DR, Move.BR
   );
   static final List<Move> bottomTriReflection = Arrays.asList(
      Move.LR, Move.DR, Move.LL, Move.DL, Move.RL, Move.DR, Move.LR, Move.DL, Move.LL, Move.RR
   );
   static final List<Move> bottomTriInverse = Arrays.asList(
      Move.BL, Move.DL, Move.BR, Move.DR, Move.FR, Move.DL, Move.BL, Move.DR, Move.BR, Move.FL
   );
   static final List<Move> bottomRotate = Arrays.asList(
      Move.FL, Move.UR, Move.UR, Move.FR, Move.RR, Move.UL, Move.UL, Move.RL, Move.DR,
      Move.RR, Move.UR, Move.UR, Move.RL, Move.FL, Move.UL, Move.UL, Move.FR, Move.DL
   );
   static final List<Move> bottomRotateInverse = Arrays.asList(
      Move.DR, Move.FL, Move.UR, Move.UR, Move.FR, Move.RR, Move.UL, Move.UL, Move.RL,
      Move.DL, Move.RR, Move.UR, Move.UR, Move.RL, Move.FL, Move.UL, Move.UL, Move.FR
   );
   
   static final CubeGroup left = CubeGroup.elements.get(Move.UL.getMatrix());
   static final CubeGroup right = left.inverse();
   static final CubeGroup opposite = left.mult(left);

   static final Map<CubeGroup, List<Move>> topEdgeMoves = new HashMap<CubeGroup, List<Move>>();
   static final Map<CubeGroup, List<Move>> topCornerMoves = new HashMap<CubeGroup, List<Move>>();
   static final Map<CubeGroup, List<Move>> sideMoves = new HashMap<CubeGroup, List<Move>>();
   static final Map<CubeGroup, List<Move>> bottomEdgeMoves = new HashMap<CubeGroup, List<Move>>();
   static final Map<CubeGroup, List<Move>> bottomCornerMoves = new HashMap<CubeGroup, List<Move>>();
   
   // reference cubelets for solving each phase
   static final Cube topEdgeSource = Cube.cubeAt(2, 0, -2);
   static final Cube topCornerSource = Cube.cubeAt(2, 2, -2);
   static final Cube sideSource = Cube.cubeAt(2, 2, 0);
   static final Cube bottomEdgeSource = Cube.cubeAt(2, 0, 2);
   static final Cube bottomCornerSource = Cube.cubeAt(2, 2, 2);
   static final Cube[] bottomEdges = {
      Cube.cubeAt(2, 0, 2), Cube.cubeAt(0, 2, 2), Cube.cubeAt(-2, 0, 2), Cube.cubeAt(0, -2, 2)
   };
   static final Cube[] bottomCorners = {
      Cube.cubeAt(2, 2, 2), Cube.cubeAt(-2, -2, 2), Cube.cubeAt(-2, 2, 2), Cube.cubeAt(2, -2, 2)
   };
   
   // index solution sequence by current position of the source
   // solution sequence restores source to its home position
   // others in same plane can be restored by conjugates of this sequence
   static {
      for (Move[] moveArray : topEdgeSolutions) {
         setup(topEdgeMoves, topEdgeSource, Arrays.asList(moveArray));
      }
      for (Move[] moveArray : topCornerSolutions) {
         setup(topCornerMoves, topCornerSource, Arrays.asList(moveArray));
      }
      for (Move[] moveArray : sideSolutions) {
         setup(sideMoves, sideSource, Arrays.asList(moveArray));
      }
      for (Move[] moveArray : bottomEdgeSolutions) {        
         setup(bottomEdgeMoves, bottomEdgeSource, Arrays.asList(moveArray));
      }
      // bottom corner solutions
      setup(bottomCornerMoves, bottomCornerSource, new ArrayList<Move>());
      setup(bottomCornerMoves, bottomCornerSource, bottomRotate);
      setup(bottomCornerMoves, bottomCornerSource, bottomRotateInverse);
      setup(bottomCornerMoves, bottomCornerSource, bottomTri);
      setup(bottomCornerMoves, bottomCornerSource, bottomTri, bottomRotate);
      setup(bottomCornerMoves, bottomCornerSource, bottomTri, bottomRotateInverse);
      setup(bottomCornerMoves, bottomCornerSource, bottomTriInverse);
      setup(bottomCornerMoves, bottomCornerSource, bottomTriInverse, bottomRotate);
      setup(bottomCornerMoves, bottomCornerSource, bottomTriInverse, bottomRotateInverse);
      setup(bottomCornerMoves, bottomCornerSource, bottomTriReflection);
      setup(bottomCornerMoves, bottomCornerSource, bottomTriReflection, bottomRotate);
      setup(bottomCornerMoves, bottomCornerSource, bottomTriReflection, bottomRotateInverse);
   }
   
   static void setup(Map<CubeGroup, List<Move>> options, Cube source, List<Move>... moves) {
      List<Move> sol = new ArrayList<Move>();
      for (List<Move> m : moves) sol.addAll(m);
      CubeGroup cg = new RubikGroup(sol).inverse().dest.get(source);
      options.put(cg, sol);
   }
   
   /**
    * Solution planner
    * This is the code that plans the solution.  The solution is built up in a List<Move> called solution.
    * The RubikGroup element rg holds the state that the cube would be in after executing the moves
    * currently in the solution; that is, it is the current state of the cube composed with
    * all the moves in solution.
    */
   RubikGroup rg;
   final List<Move> solution = new ArrayList<Move>();
   
   Planner(State state) {
      this.state = state;
   }
   
   // Solve a single cubelet in a phase.
   // The cube to solve is a conjugate of the source cube determined by f.
   void solveOne(Map<CubeGroup, List<Move>> phaseSolutions, CubeGroup f, Cube source) {
      CubeGroup g = rg.dest.get(f.apply(source));
      CubeGroup h = g.conjugate(f);
      List<Move> moves = phaseSolutions.get(h);
      moves = Move.conjugateSequence(moves, f.inverse());
      append(moves);   
   }
   
   // Solve all four cubelets in a plane.
   void solveFour(Map<CubeGroup, List<Move>> phaseSolutions, Cube source) {
      solveOne(phaseSolutions, CubeGroup.IDENTITY, source);
      solveOne(phaseSolutions, right, source);
      solveOne(phaseSolutions, opposite, source);
      solveOne(phaseSolutions, left, source);
   }
   
   /*
    * This is the actual method to solve the entire cube.
    * The algorithm is broken into several phases.
    */

   List<Move> solve() {
      rg = state.getGroup(); // start with current state of the cube
      solution.clear(); // no solution moves yet
      
      // solve top edges     
      solveFour(topEdgeMoves, topEdgeSource);
      // solve top corners
      solveFour(topCornerMoves, topCornerSource);
      // solve sides
      solveFour(sideMoves, sideSource);     
      
      // solve bottom edges
      // first check parity of edges on bottom face
      Set<Cube> s = asSet(bottomEdges);
      // if odd, make it even
      if (rg.parity(s)) append(Move.DL);
      solveOne(bottomEdgeMoves, CubeGroup.IDENTITY, bottomEdgeSource);
      solveOne(bottomEdgeMoves, opposite, bottomEdgeSource);
      // if two bottom sides are switched
      if (!rg.dest.get(Cube.cubeAt(0, 2, 2)).equals(CubeGroup.IDENTITY)) {
         append(bottomCross);
      }
      
      // solve bottom corners
      solveFour(bottomCornerMoves, bottomCornerSource);

      // delete extraneous moves
      simplify(solution);
      return solution;
   }
   
   /*
    * Helpers
    */
   
   // append a move to the solution list
   void append(Move move) {
      rg.compose(move);
      solution.add(move);
   }
   
   // append a list of moves to the solution list
   void append(List<Move> moves) {
      rg.compose(new RubikGroup(moves));
      solution.addAll(moves);
   }
   
   // simplify the solution -
   // collapse short substrings equivalent to the identity
   void simplify(List<Move> list) {
      int oldSize = list.size() + 1;
      while (list.size() < oldSize) {
         oldSize = list.size();
         for (int i = 0; i < list.size(); i++) {
            if (i < list.size() - 1 && list.get(i).equals(list.get(i + 1).inverse())) {
               list.remove(i + 1);
               list.remove(i);
               if (i > 0) i--;
            }
            if (i < list.size() - 2 && list.get(i).equals(list.get(i + 1)) && list.get(i).equals(list.get(i + 2))) {
               list.remove(i + 2);
               list.remove(i + 1);
               Move m = list.remove(i);
               list.add(i, m.inverse());
               i -= Math.min(i, 2);
            }
         }
      }
   }
   
   // generic method for converting an array to a Set
   <T> Set<T> asSet(T[] array) {
      Set<T> s = new HashSet<T>();
      for (T item : array) s.add(item);
      return s;
   }

}
