package rubik;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;

import javax.imageio.ImageIO;
import javax.swing.*;
import javax.swing.event.*;

import rubik.State.Move;
import rubik.geometry.ThreeDPoint;
import rubik.geometry.TwoDPoint;

public class GUI extends JFrame implements Runnable {
   
   Random random = new Random();
   
   Rubik rubik;
   State state;
   GUI gui;
   
   final int WINDOW_SIZE = 500;
   final int ORIGIN = WINDOW_SIZE/2;
   
   boolean fill = true;
   Image image;
   Image backgroundImage;
   Graphics2D buffer;
   TwoDPoint dragOrigin = null;
   
   final JPanel canvas = new JPanel() {
      public void paint(Graphics g) {
         synchronized (image) {
            g.drawImage(image, 0, 0, canvas);
         }
      }
   };
   
   final JLabel messageArea = new JLabel();
   
   GUI(Rubik rubik, State state) {
      super("Rubik's Cube");
      this.rubik = rubik;
      this.state = state;
      gui = this;
   }
   
   /**
    * This method initializes the gui. We must create the menu bar and menus,
    * create the message area, initialize the event handlers, and initialize the
    * display. Inform the animator when we are done.
    */
   public void run() {
      
      /*
       * Menu bar
       */
      
      JMenuBar menuBar = new JMenuBar();
      JMenu menu;
      JMenuItem menuItem;
      
      // File menu
      menu = new JMenu("File");
      final JFileChooser fileChooser = new JFileChooser();
      
      // File > Open
      menuItem = new JMenuItem("Open...");
      menuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            int ok = fileChooser.showOpenDialog(canvas);
            if (ok != JFileChooser.APPROVE_OPTION) return;
            File file = fileChooser.getSelectedFile();
            Scanner sc = null;
            try {
               sc = new Scanner(file);
            } catch (FileNotFoundException fnf) {
               JOptionPane.showMessageDialog(canvas, "File not found");
            }
            try {
               Cube.decode(sc);
               message("File successfully read");
            } catch (Exception exc) {
               JOptionPane.showMessageDialog(canvas, "Could not read input file");
            }
         }
      });
      menu.add(menuItem);
      
      // File > Save
      menuItem = new JMenuItem("Save...");
      menuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            //clearMessage();
            PrintStream out = null;
            int ok = fileChooser.showSaveDialog(canvas);
            if (ok != JFileChooser.APPROVE_OPTION) return;
            File file = fileChooser.getSelectedFile();
            if (file.exists()) {
               String msg = "File " + file.getAbsolutePath() + " exists; overwrite?";
               ok = JOptionPane.showConfirmDialog(canvas, msg, "", JOptionPane.YES_NO_OPTION);
               if (ok != JOptionPane.YES_OPTION) return;
            }
            try {
               out = new PrintStream(new FileOutputStream(file));
               out.println(Cube.encode());
               message("File successfully written");
            } catch (Exception exc) {
               JOptionPane.showMessageDialog(canvas, "Cannot write file " + file.getAbsolutePath());
            } finally {
               if (out != null) out.close();
            }
         }
      });
      menu.add(menuItem);
      
      menu.addSeparator();
      
      // File > Quit
      menuItem = new JMenuItem("Quit");
      menuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            System.exit(0);
         }
      });
      menu.add(menuItem);
      menuBar.add(menu);
      
      // Action menu
      menu = new JMenu("Action");
      
      // Action > Undo
      menuItem = new JMenuItem("Undo");
      menuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            try {
               state.undo();
               clearMessage();
            } catch (EmptyStackException ese) {
               message("Nothing to undo");
            }
         }
      });
      menu.add(menuItem);
      
      // Action > Scramble
      menuItem = new JMenuItem("Scramble...");
      final SpinnerNumberModel spinnerModel = new SpinnerNumberModel(3, 1, 99, 1);
      final JSpinner spinner = new JSpinner(spinnerModel);
      spinner.setPreferredSize(new Dimension(10, 27));
      //spinner.setFont(new Font("Helvetica", Font.PLAIN, 18));
      
      menuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            int result =
                     JOptionPane.showOptionDialog(gui, spinner, "Select number of moves",
                     JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE, null, null, null);
            if (result == JOptionPane.CLOSED_OPTION || result == JOptionPane.CANCEL_OPTION) return;
            int n = spinnerModel.getNumber().intValue();
            int next = 0, last = 0;
            for (int i = 0; i < n; i++) {
               while (next == last) {
                  next = random.nextInt(Move.values().length);
               }
               last = next;
               Move move = Move.values()[next];
               synchronized (state) {
                  state.move(move);
                  state.history.add(move.inverse());
               }
            }
            message(n + " random moves performed");
         }
      });
      menu.add(menuItem);
      
      // Action > Reset
      menuItem = new JMenuItem("Reset");
      menuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            state.reset();
            clearMessage();
         }
      });
      menu.add(menuItem);
      menuBar.add(menu);      
      
      // Action > Solve
      menuItem = new JMenuItem("Solve");
      menuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            message(state.solve() + " moves");
         }
      });
      menu.add(menuItem);
      menuBar.add(menu);      
      
      // Action > Solve Immediately
      menuItem = new JMenuItem("Solve Immediately");
      menuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            message(state.solveImmediately() + " moves");
         }
      });
      menu.add(menuItem);
      menuBar.add(menu);          
      
      // Action > Debug
      menuItem = new JMenuItem("Debug");
      menuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            message(state.debug());
         }
      });
      if (Rubik.DEBUG) menu.add(menuItem);
      menuBar.add(menu);          
      
      // Options menu
      menu = new JMenu("Options");
      
      // Option > Color
      menuItem = new JMenuItem("Color...");
      final ButtonGroup buttons = new ButtonGroup();
      
      class ColorPanel extends JRadioButton {
         Color color;

         public ColorPanel(int index) {
            setPreferredSize(new Dimension(40, 40));
            color = Cube.colors[index];     
            buttons.add(this);
            addActionListener(new ActionListener() {
               public void actionPerformed(ActionEvent e) {
                  buttons.setSelected(getModel(), true);
               }
            });
         }
         
         public void paint(Graphics g) {
            g.setColor(isSelected()? Color.BLACK : color);
            g.fillRect(0,  0, getWidth(), getHeight());
            if (isSelected()) {
               g.setColor(color);
               g.fillRect(3, 3, getWidth() - 6, getHeight() - 6);
            }
         }
      }
      
      class PreviewPanel extends JPanel {
         public PreviewPanel() {
            setPreferredSize(new Dimension(400, 50));
            // workaround for
            // http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=5029286
            setSize(getPreferredSize());
            setBorder(BorderFactory.createEmptyBorder(0,0,1,0));
         }
      }

      final PreviewPanel previewPanel = new PreviewPanel();
      final Map<ButtonModel, ColorPanel> buttonMap = new HashMap<ButtonModel, ColorPanel>();
      
      final JColorChooser colorChooser = new JColorChooser();
      colorChooser.getSelectionModel().addChangeListener(new ChangeListener() {
         public void stateChanged(ChangeEvent e) {
            ButtonModel buttonModel = buttons.getSelection();
            if (buttonModel == null) return;
            ColorPanel cp = buttonMap.get(buttonModel);
            cp.color = colorChooser.getColor();
         }
      });
      
      for (int i = 0; i < 6; i++) {
         ColorPanel cp = new ColorPanel(i);
         previewPanel.add(cp);
         buttonMap.put(cp.getModel(), cp);
      }
      colorChooser.setPreviewPanel(previewPanel);
      
      final JDialog colorDialog = JColorChooser.createDialog(canvas,
               "Choose colors", false, colorChooser,
               // ok listener
               new ActionListener() {
                  public void actionPerformed(ActionEvent arg0) {
                     for (int i = 0; i < 6; i++) {
                        ColorPanel cp = (ColorPanel)previewPanel.getComponent(i);
                        Cube.colors[i] = cp.color;
                     }
                     Cube.setAllColors();
                  }
               }, null);
      
      /*
       * Attempt to capture reset button in the color dialog. It would have been
       * nice if Java had provided a hook. As it is, we are forced to break all
       * kinds of abstraction barriers.
       */
      try {
         Container c = (Container)colorChooser.getParent().getComponent(1);
         JButton b = (JButton)c.getComponent(2);
         assert b.getText().equals("Reset");
         b.removeActionListener((b.getActionListeners()[0]));
         b.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent arg0) {
               for (int i = 0; i < 6; i++) {
                  ColorPanel cp = (ColorPanel)previewPanel.getComponent(i);
                  cp.color = Cube.colors[i];
                  previewPanel.repaint();
               }
            }
         });
      } catch (Exception e) {}

      menuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            colorDialog.setVisible(true);
         }
      });
      menu.add(menuItem);
      
      // Option > Speed
      menuItem = new JMenuItem("Speed...");
      final JSlider slider = new JSlider(JSlider.HORIZONTAL, State.MIN_SPEED, State.MAX_SPEED, State.INIT_SPEED);
      slider.addChangeListener(new ChangeListener() {
         public void stateChanged(ChangeEvent e) {
            state.setSpeed(slider.getValue());
         }
      });
      
      menuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            final JDialog speedDialog = new JDialog(gui, "Set speed of rotation", false);
            speedDialog.setResizable(false);
            speedDialog.setLayout(new FlowLayout(FlowLayout.CENTER));
            speedDialog.addWindowListener(new WindowAdapter() {
               public void windowClosing(WindowEvent event) {
                  speedDialog.dispose();
               }
            });
            JLabel label = new JLabel("Slow");
            speedDialog.add(label);
            speedDialog.add(slider);
            label = new JLabel("Fast");
            speedDialog.add(label);
            speedDialog.setLocationRelativeTo(gui);
            speedDialog.pack();
            speedDialog.setVisible(true);
         }
      });
      menu.add(menuItem);

      // Option > Mute
      final JCheckBoxMenuItem mute = new JCheckBoxMenuItem("Mute");
      mute.addItemListener(new ItemListener() {
         public void itemStateChanged(ItemEvent e) {
            state.click.setEnabled(!mute.isSelected());
            state.tada.setEnabled(!mute.isSelected());
         }
      });
      menu.add(mute);
      menuBar.add(menu);
      
      // Help menu
      menu = new JMenu("Help");
      
      // Help > Help...
      menuItem = new JMenuItem("Get Help...", KeyEvent.VK_H);
      menuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            final String[] helpMsg = {
                     "Drag with the mouse to rotate the cube.",
                     "Shift to brake.",
                     "Keys turn the faces:",
                     "s - front (closest to viewer)",
                     "e - back",
                     "w - top",
                     "a - left",
                     "d - right",
                     "x - bottom",
                     "Faces turn counterclockwise.",
                     "Shift to turn clockwise.",
                     "Arrow keys zoom in and out.",
                     "Menu items are self-explanatory."};
            JOptionPane.showMessageDialog(gui, helpMsg, "Rubik's Cube Help", JOptionPane.INFORMATION_MESSAGE);
         }
//      // Help dialog
//      class HelpDialog extends JDialog {
//         BufferedImage logo = null;
//         int width = 200, height = 200; // default in case read fails
//
//         HelpDialog() {
//            super(gui, "Help!", false);
//            try {
//               logo = ImageIO.read(ClassLoader.getSystemResourceAsStream("resources/images/munch.gif"));
//               width = logo.getWidth();
//               height = logo.getHeight();
//            } catch (Exception e) {
//               add(new JLabel("Help!", JLabel.CENTER));
//            }
//            setSize(width, height);
//            setLocationRelativeTo(gui);
//         }
//
//         // display help dialog
//         void display() {
//            setVisible(true);
//         }
//
//         public void paint(Graphics graphics) {
//            graphics.setColor(getBackground());
//            graphics.fillRect(0, 0, getSize().width, getSize().height);
//            if (logo != null) {
//               graphics.drawImage(logo, 0, 0, getBackground(), null);
//            }
//         }
//      }
//      
//      final HelpDialog helpDialog = new HelpDialog();     
//      menuItem.addActionListener(new ActionListener() {
//         public void actionPerformed(ActionEvent e) {
//            helpDialog.display();
//         }
      });
      menu.add(menuItem);
      
      menu.addSeparator();
      
      menuItem = new JMenuItem("About...", KeyEvent.VK_A);
      menuItem.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent e) {
            final String[] aboutMsg = {
                     "Rubik's Cube",
                     "CS2110 Fall 2011 Assignment 4",
                     "Version " + Rubik.VERSION,
                     "Java program \u00a9 2011 Cornell University\n\n",
                     "Rubik\u00ae and Rubik's Cube\u00ae are registered trademarks",
                     "of Seven Towns Limited, which is the exclusive worldwide",
                     "licensee of copyright for images of the Rubik's Cube",
                     "puzzle and the puzzle itself.\n\n",
                     "The use of Rubik's Cube in this context is deemed to be",
                     "fair use under the Copyright Act of 1976, 17 U.S.C. \u00a7107."};
            JOptionPane.showMessageDialog(gui, aboutMsg, "About CS2110 Rubik", JOptionPane.INFORMATION_MESSAGE);
         }
      });
      menu.add(menuItem);
      menuBar.add(menu);

      setJMenuBar(menuBar);
            
      /*
       * Mouse and key handlers
       */
      
      addMouseListener(new MouseAdapter() {
         public void mouseClicked(MouseEvent e) {
            if (e.getButton() == MouseEvent.BUTTON3) {
               fill = !fill;
            }
         }
         
         public void mousePressed(MouseEvent e) {
            if (e.getButton() == MouseEvent.BUTTON1) {
               state.velocity = TwoDPoint.ZERO;
               dragOrigin = mousePosition(e);
            }
         }
      });
    
      addMouseMotionListener(new MouseMotionAdapter() {
         public void mouseDragged(MouseEvent e) {
            final double scaleFactor = -0.002;
            if (state.brake) return;
            if (dragOrigin == null) {
               dragOrigin = mousePosition(e);
            } else {
               TwoDPoint pos = mousePosition(e);
               double dist = pos.distance(dragOrigin);
               if (dist < 10) return;
               state.velocity = pos.minus(dragOrigin).scalarMult(scaleFactor);
            }
         }
      });
      
      addKeyListener(new KeyAdapter() {
         public void keyPressed(KeyEvent e) {
            //System.out.println(e.getKeyCode());
            switch (e.getKeyCode()) {
            case 38: // uparrow
               state.distanceVelocity = 0.1;
               break;
            case 40: // downarrow
               state.distanceVelocity = -0.1;
               break;
            case 16: // shift
               state.brake = true;
               state.velocity = TwoDPoint.ZERO;
               break;
            case 72: // 'h'
               System.out.println(Cube.encode());
               break;
            case 65: // 'a'
               state.startRotation(e.isShiftDown()? Move.LR : Move.LL, false);
               break;
            case 68: // 'd'
               state.startRotation(e.isShiftDown()? Move.RR : Move.RL, false);
               break;
            case 69: // 'e'
               state.startRotation(e.isShiftDown()? Move.BR : Move.BL, false);
               break;
            case 83: // 's'
               state.startRotation(e.isShiftDown()? Move.FR : Move.FL, false);
               break;
            case 87: // 'w'
               state.startRotation(e.isShiftDown()? Move.UR : Move.UL, false);
               break;
            case 88: // 'x'
               state.startRotation(e.isShiftDown()? Move.DR : Move.DL, false);
            }
         }

         public void keyReleased(KeyEvent e) {
            switch (e.getKeyCode()) {
            case 38: // uparrow
            case 40: // downarrow
               state.distanceVelocity = 0;
               break;
            case 16: // shift
               state.brake = false;
            }
         }
      });
      
      /*
       * Custom message event handler
       */
      
      MessageEvent.addMessageEventListener(new MessageEventListener() {
         public void eventOccurred(MessageEvent event) {
            message(event.msg);
         }
      });
      
      
      /*
       * Initialize the display
       */
      
      // set main window parameters
      setDefaultCloseOperation(EXIT_ON_CLOSE);
      setResizable(false);
      setLocationRelativeTo(null); // center on screen
      setLocation(getLocation().x - ORIGIN, getLocation().y - ORIGIN);
      
      // where we draw stuff
      canvas.setPreferredSize(new Dimension(WINDOW_SIZE, WINDOW_SIZE));
      add(canvas, BorderLayout.CENTER);
      
      messageArea.setPreferredSize(new Dimension(WINDOW_SIZE, 20));
      add(messageArea, BorderLayout.SOUTH);

      pack();
      setVisible(true);
      
      // off-screen buffer
      image = createImage(WINDOW_SIZE, WINDOW_SIZE);
      buffer = (Graphics2D)image.getGraphics();
      buffer.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      buffer.setRenderingHint(RenderingHints.KEY_STROKE_CONTROL, RenderingHints.VALUE_STROKE_NORMALIZE);
      buffer.translate(ORIGIN, ORIGIN);
      
      backgroundImage = createImage(WINDOW_SIZE, WINDOW_SIZE);
      Graphics bg = backgroundImage.getGraphics();
      boolean daytime = (Calendar.getInstance().get(Calendar.HOUR_OF_DAY) + 6) % 24 >= 12;
      String fileName = daytime? "resources/images/clouds.jpg" : "resources/images/stars.jpg";
      try {
         backgroundImage = ImageIO.read(getClass().getClassLoader().getResourceAsStream(fileName));
      } catch (IOException ioe) {}
      bg.translate(ORIGIN, ORIGIN);
      
      // inform animator we are ready to roll      
      synchronized (rubik) {
         rubik.notify();
      }
   }
   
   void message(String msg) {
      messageArea.setText(" " + msg);
   }
   
   void clearMessage() {
      message("");
   }
   
   /*
    * These methods are called from rubik.animate() to display the image
    */
   
   private TwoDPoint mousePosition(MouseEvent e) {
      return new TwoDPoint(e.getX() - ORIGIN, e.getY() - ORIGIN);
   }
   
   void render(Square s) {
      // check whether we have anything to render
      Polygon pBack = s.project(state, false);
      Polygon pFront = s.project(state, true);
      if (pBack.npoints <= 2 || pFront.npoints <= 2) return;      
      
      // light intensity is cosine of angle of obliqueness of light source on surface
      // set color gradients accordingly
      ThreeDPoint lightSource = state.rotating(s.cube)?
               state.rMatrix.transpose().mult(state.orientation.mult(State.BASE_LIGHT_SOURCE)) :
               state.orientation.mult(State.BASE_LIGHT_SOURCE);
               
      // get color from home position of cube
      Color color = s.cube.position.apply(s).color; 
      float intensity = fill? (float)((lightSource.cos(s.normal) + 1) / 2) : 1;
      float r = intensity * color.getRed() / 256;
      float g = intensity * color.getGreen() / 256;
      float b = intensity * color.getBlue() / 256;      
      if (fill) {
         buffer.setColor(Color.BLACK);
         buffer.fillPolygon(pBack);
         buffer.setColor(new Color(r, g, b));
         buffer.fillPolygon(pFront);        
      } else {
         buffer.setColor(new Color(r, g, b));
         buffer.drawPolygon(pBack);
      }
   }
   
   void drawBackground() {
      buffer.drawImage(backgroundImage, -ORIGIN, -ORIGIN, canvas);
   }
   
}