package rubik;

import java.util.HashMap;
import java.util.Map;

import rubik.geometry.Matrix;
import rubik.geometry.ThreeDPoint;

/**
 * A representation of the group of orientation-preserving (det = 1) symmetries
 * of the cube. The group has 24 elements. Each element is specified by a 3x3
 * matrix with entries -1,0,1 with exactly one nonzero element in each row and
 * column.  The permutation of the nonzero elements is odd (resp., even) if
 * the number of -1's is odd (resp., even).
 */
public class CubeGroup {

   public static CubeGroup IDENTITY;

   private Matrix matrix;

   private static final int[][] evenReflections = {
      { 1, 1, 1 }, { 1, -1, -1 }, { -1, 1, -1 }, { -1, -1, 1 }
   };

   private static final int[][] oddReflections = {
      { -1, -1, -1 }, { -1, 1, 1 }, { 1, -1, 1 }, { 1, 1, -1 }
   };

   private static final int[][] evenPermutations = {
      { 0, 1, 2 }, { 1, 2, 0 }, { 2, 0, 1 }
   };
   
   private static final int[][] oddPermutations = {
      { 0, 2, 1 }, { 1, 0, 2 }, { 2, 1, 0 }
   };

   static final Map<Matrix, CubeGroup> elements = new HashMap<Matrix, CubeGroup>();

   private CubeGroup(int[] perm, int[] refl)  {
      final int[][] m = new int[3][3];
      for (int j = 0; j < 3; j++) {
         int k = perm[j];
         m[k][j] = refl[k];
         m[k][(j + 1) % 3] = 0;
         m[k][(j + 2) % 3] = 0;
      }
      matrix = new Matrix(m);
      elements.put(matrix, this);
   }

   // create the group elements
   static void create() {
      for (int i = 0; i < 12; i++) {
         new CubeGroup(evenPermutations[i/4], evenReflections[i%4]);
         new CubeGroup(oddPermutations[i/4], oddReflections[i%4]);
      }
      IDENTITY = elements.get(Matrix.IDENTITY);
   }
   
   CubeGroup mult(CubeGroup cg) {
      return elements.get(matrix.mult(cg.matrix));
   }
   
   CubeGroup inverse() {
      return elements.get(matrix.transpose());
   }
   
   ThreeDPoint apply(ThreeDPoint p) {
      return matrix.mult(p);
   }

   // apply this group element to the given Square to get a new Square
   Square apply(Square s) {
      return Square.elements.get(apply(s.center));
   }

   // apply this group element to the given Cube to get a new Cube
   Cube apply(Cube c) {
      return Cube.cubes.get(apply(c.center));
   }

   // f^{-1} o this o f
   CubeGroup conjugate(CubeGroup f) {
      return f.inverse().mult(this).mult(f);
   }
   
   public int hashCode() {
      return matrix.hashCode();
   }

   public boolean equals(Object obj) {
      try {
         CubeGroup cg = (CubeGroup)obj;
         return matrix.equals(cg.matrix);
      } catch (Exception e) {
         return false;
      }
   }
   
   public String toString() {
      return matrix.toString();
   }

}

