package rubik;

import java.util.HashMap;
import java.util.Map;

import rubik.State.Plane;

/**
 * BSP trees for rendering the cube.  We actually use 3 trees, depending on which 
 * plane is rotating.  This class implements the visitor pattern.
 */

class BSPTreeNode {
   
   // roots of the 3 BSP trees
   static final BSPTreeNode xRoot = new BSPTreeNode(); // traverse when rotating about x axis
   static final BSPTreeNode yRoot = new BSPTreeNode(); // traverse when rotating about y axis
   static final BSPTreeNode zRoot = new BSPTreeNode(); // traverse when rotating about z axis
   static final Map<Plane, BSPTreeNode> roots = new HashMap<Plane, BSPTreeNode>();

   static {
      roots.put(Plane.FRONT, xRoot);
      roots.put(Plane.BACK, xRoot);
      roots.put(Plane.LEFT, yRoot);
      roots.put(Plane.RIGHT, yRoot);
      roots.put(Plane.UP, zRoot);
      roots.put(Plane.DOWN, zRoot);
   }
   
   private Square data = null;
   private BSPTreeNode back = null;
   private BSPTreeNode front = null;

   public BSPTreeNode(Square s) {
      data = s;
   }

   public BSPTreeNode() {}

   public void insert(Square s) {
      if (data == null) data = s;
      else if (data.behind(s.center)) {
         if (front == null) front = new BSPTreeNode(s);
         else front.insert(s);
      } else if (back == null) back = new BSPTreeNode(s);
         else back.insert(s);
   }
   
   /*
    * Visitors traverse the BSP tree from back to front from the observer's
    * point of view and do something at each node. There is an accumulated
    * result accum that is passed along. The visitor's visit method may compute
    * a new accum based on the old accum and the data. We don't use it here, but
    * we left it in for illustration.
    */
   public <T> T enumerate(BSPVisitor<T> visitor, T accum) {
      if (data.behind(visitor.viewpoint(data))) {
         if (back != null) accum = back.enumerate(visitor, accum);
         accum = visitor.visit(data, accum);
         if (front != null) accum = front.enumerate(visitor, accum);
      } else {
         if (front != null) accum = front.enumerate(visitor, accum);
         accum = visitor.visit(data, accum);
         if (back != null) accum = back.enumerate(visitor, accum);
      }
      return accum;
   }

}