#!/usr/bin/env python
"""
CUGL Game Lobby Server

This module provides a websocket implementation of an ad-hoc game lobby. This lobby
is based on the Slikenet/Raknet game lobby used by the GDIAC game SweetSpace (2020),
which in turn was based on the Firebase game lobby used by Family Style (2019). This
conversion to websockets allows CUGL games to use standard STUN/TURN servers for 
NAT traversal instead of custom, hard-to-maintain solutions.

This game lobby is inspired by a traditional web RTC signaling server, like the ones 
provided by libdatachannel (by Paul-Louis Ageneau):
    
    https://github.com/paullouisageneau/libdatachannel
    
However, this server also supports game lobby messages on top of traditional signaling.
Such messages have type "lobby", rather than "offer", "answer", or "candidate".

To use this game sever, each game must first generate a UUID. This UUID should be 
globally unique, so as to prevent clashes with other clients. We recommend that you
generate a RFC 4122 conformant UUID, but this is not enforced.

Upon connection, the game receives a handshake from the game lobby. The game must 
specify its role as "host" or "client". A "host" is immediately given a room id; in 
this case connection is guaranteed to succeed. A "client" must also specify a room id. 
The client will attempt to join that room. A client may be denied for several reasons: 
no room with that id, the room is full, the API versions do not match, or the host has 
locked the room to new players. If a client succeeds, it is given the UUID of the room 
host. It can then use that UUID to set up a peer connection to the host using 
standard RTC signaling features. From that point on, all communication between the
game instances takes place through peer connections.

While game communication takes place through peer connections, all game instances 
maintain connections with the lobby for simple connection management. If a player is
disconnected, this information is communicated via the websocket to the remaining
game instances. In addition, while clients only create peer connections to the host,
all clients are notified of the existence of all other clients (in the same room) for
relay purposes.

Finally, this game lobby provides support for simple host migration. It a room host 
disconnects, the remaining clients are given the option to migrate to a new host. If
just one room client accepts this invitation, that client will become the new room host.
If multiple room clients accept, the first responder will be chosen.  If no room clients
accept, this game lobby will send a shutdown command, removing all clients from the room.

:author:  Walker M. White (wmw2)
:version: January 15, 2023
"""

# CUGL MIT License:
#   This software is provided 'as-is', without any express or implied
#   warranty.  In no event will the authors be held liable for any damages
#   arising from the use of this software.
#
#   Permission is granted to anyone to use this software for any purpose,
#   including commercial applications, and to alter it and redistribute it
#   freely, subject to the following restrictions:
#
#   1. The origin of this software must not be misrepresented; you must not
#      claim that you wrote the original software. If you use this software
#      in a product, an acknowledgment in the product documentation would be
#      appreciated but is not required.
#
#   2. Altered source versions must be plainly marked as such, and must not
#      be misrepresented as being the original software.
#
#   3. This notice may not be removed or altered from any source distribution.

import sys
import ssl
import json
import asyncio
import logging
import random
import string
import uuid
import traceback
import websockets
import websockets.exceptions


# CONNECTION SETTINGS
# The list of bindpoint addresses. An empty list will bind to all local interfaces
BINDPOINT = []
# The port to receive incoming connections
BINDPORT  = 8080
# A reference to the SSL certificate
SSL_CERT  = None
# The logging level for websocket information (does not include this module)
WEBSOCKET_LOG = logging.INFO
# The logging level for this module
CUGLLOBBY_LOG = logging.DEBUG


# ROOM STATES
# People can still join this room
ROOM_OPEN   = 0
# No one can join the room any more
ROOM_CLOSED = 1
# No one can join the room any more
ROOM_DELETED = 2
# We are migrating the room host
ROOM_MIGRATING = 3

class Room(object):
    """
    A class representing a game room.
    
    Our lobby organizes players into ad-hoc rooms. The first player joins as host, 
    receiving a room number. All other players join as room clients. A room has a
    maximum number of players, as specified by host. In addition, the rooms have an
    API version to make sure that all connected games are using the same software.
    
    Each room is identified by a unique string. The initializer method ensures that
    there are no identifier collisions. The identifier is guaranteed to be a hexadecimal
    integer with ``DIGITS`` characters.
    
    A room is simply a shared data structure. Rooms do not perform any communication
    on their own. However, they are readily used by asynchronous threads, which is 
    why we have locking features to ensure thread safety. However, all locking is 
    expected to be done externally with the ``lock`` attribute. No methods support
    locking by themselves.
    """
    # The number of (hexadecimal) digits for a room
    DIGITS = 4  # Guarantees classic 65536 connections 
    # Which room identifiers are in currently use
    INUSE = set()
    
    @property 
    def name(self):
        """
        The room name, chosen at creation.
        
        **Invariant**: Value is a string representation of a hexadecimal number with
        ``DIGITS`` characters.
        """
        return self._name
    
    @property 
    def host(self):
        """
        The current room host.
        
        **Invariant**: Value is UUID of a game connection
        """
        return self._host
    
    @host.setter
    def host(self, value):
        assert (type(value) == str), "{} is not a str".format(value)
        if value in self._clients:
            self._clients.remove(value)
        self._host = value
    
    @property 
    def clients(self):
        """
        The current set of room clients.
        
        **Invariant**: Value is set of game connections UUIDs.
        """
        return self._clients
    
    @property 
    def maxPlayers(self):
        """
        The maximum number of players allowed in this room
        
        **Invariant**: Value is an int > 0
        """
        return self._maxPlayers
    
    @property 
    def apiVersion(self):
        """
        The API version for this room
        
        **Invariant**: Value is an int > 0. All hosts and clients must agree on version.
        """
        return self._apiVersion
    
    @property 
    def state(self):
        """
        The current room state.
        
        **Invariant**: Value is one of ``ROOM_OPEN``, ``ROOM_CLOSED``, ``ROOM_DELETED``, 
        or ``ROOM_MIGRATING``
        """
        return self._state

    @state.setter 
    def state(self,value):
        options = [ROOM_OPEN, ROOM_CLOSED, ROOM_DELETED, ROOM_MIGRATING]
        assert value in options, "{} is not a valid state".format(value)
        self._state = value
    
    @property 
    def lock(self):
        """
        A mutex lock to guarantee thread safety
        
        **Invariant**: Value  is an instance of ``asyncio.Lock``
        """
        return self._room_lock
    
    def __init__(self,root,config):
        """
        Initializes a new room with the given configuration.
        
        The configuration should be a dictionary with keys 'id', 'maxPlayers', and
        'apiVersion'. The 'id' value should be the UUID of the room host. The room
        will be initialized with state ``ROOM_OPEN``.
        
        :param root: The UUID of the game lobby server
        :type root:  ``str``
        
        :param config: The JSON dictionary with the room settings
        :type root:  ``dict``
        """
        self._root = root
        self._host = config['id']
        self._maxPlayers = config['maxPlayers']
        self._apiVersion = config['apiVersion']
        self._room_lock = asyncio.Lock()
        
        self._name = None
        self._acquire()
        
        self._clients = set()
        self._prevs = ROOM_OPEN
        self._state = ROOM_OPEN
        self._attempts = 0

    def json(self):
        """
        Returns a JSON dictionary representing this room
        
        The dictionary will have keys 'room' (for the room name), 'host', and 'clients'.
        The list of clients will not include the host.
        
        :return: a JSON dictionary representing this room
        :rtype:  ``dict``
        """
        data = {}
        data['root'] = self._root
        data['room'] = self._name
        data['host'] = self._host
        players = []
        for item in self._clients:
            players.append(item)
        data['clients'] = players
        return data
    
    def __str__(self):
        """
        Returns a string representation of this room
        
        The string will be generated from the room JSON
        
        :return: a string representation of this room
        :rtype:  ``str``
        """
        return json.dumps(self.json())
    
    def dispose(self):
        """
        Disposes this room , recycling the number
        
        The room state will be set to ROOM_DELETED. This method will return a dictionary
        of messages to send to the host and clients, warning them of the room closure. 
        The dictionary keys will be connection UUIDs, and the values will be the strings 
        to submit.
        
        :return: a dictionary of messages to send to the host and clients
        :rtype:  ``dict``
        """
        response = {}
        message  = {'id': self._root, 'type':'lobby','category':'session', 'status':'shutdown'}
        data = json.dumps(message)
        if self.host:
            response[self.host] = data
        for c in self.clients:
            response[c] = data
        
        self._clients.clear()
        self._recycle()
        self._host = None
        self._state = ROOM_DELETED
        
        return response

    def join(self,uuid,api=None):
        """
        Adds the given UUID as a room client
        
        This method This method will return a dictionary of messages to send to the host 
        and clients, notifying them of the new player. The dictionary keys will be 
        connection UUIDs, and the values will be the strings to submit.
        
        If the player is rejected from this room, this dictionary will only contain a 
        message to the player, informing them of the room rejection. The following are
        all reasons for rejection:
        
        1. This room has been closed by the host, locking it to new players
        
        2. The room has hit the maximum player capacity
        
        3. The client API version does not match that of the room
        
        :param uuid: The UUID of the game connection to add
        :type uuid:  ``str``
        
        :param api: The API version to use (None to disable API checking)
        :type api:  ``int`` or ``None``
        
        :return: a dictionary of messages to send to the host and clients
        :rtype:  ``dict``
        """
        message = {'id': self._root, 'type':'lobby','category':'room-assign', 'status':'success'}
        response = {}
        if (self._state != ROOM_OPEN):
            message['status'] = 'denial'
            data = json.dumps(message)
            response = {uuid:data}
        elif (len(self._clients)+1 == self._maxPlayers):
            message['status'] = 'denial'
            data = json.dumps(message)
            response = {uuid:data}
        elif (not api is None and api != self._apiVersion):
            message['status'] = 'mismatch'
            data = json.dumps(message)
            response = {uuid:data}
        else:
            players = [self.host]
            players.extend(self.clients)
            message['host'] = self.host
            message['players'] = players
            message['room'] = self.name
            data = json.dumps(message)
            response[uuid] = data
            
            del message['host']
            del message['players']
            del message['room']
            message['category'] = 'player'
            message['status'] = 'connect'
            message['player'] = uuid
            data = json.dumps(message)
            
            response[self.host] = data
            for c in self.clients:
                response[c] = data
            
            self.clients.add(uuid)
        
        return response
    
    def leave(self,uuid):
        """
        Removes the given UUID from this room.  
        
        If the UUID is the host, host migration will be attempted. This method will 
        return a dictionary of messages to send to the host and clients, notifying them 
        of the dropped player. The dictionary keys will be connection UUIDs, and the 
        values will be the strings to submit.
        
        :param uuid: The UUID of the game connection to drop
        :type uuid:  ``str``
        
        :return: a dictionary of messages to send to the host and clients
        :rtype:  ``dict``
        """
        response = {}
        if self._state == ROOM_MIGRATING:
            return self.dispose()
        elif uuid == self.host and len(self._clients) == 0:
            return self.dispose()
        elif uuid == self.host:
            message  = {'id': self._root, 'type':'lobby','category':'migration', 'status':'start'}
            data = json.dumps(message)
            self._prevs = self._state
            self._state = ROOM_MIGRATING
            self._attempts = len(self._clients)
            self._host = None
        elif uuid in self._clients:
            message = {'id': self._root, 'type':'lobby','category':'player', 'status':'disconnect'}
            message['player'] = uuid
            data = json.dumps(message)
            response[self.host] = data
            self._clients.remove(uuid)
        else:
            return self.dispose()
        
        for c in self._clients:
            response[c] = data
        
        return response
    
    def close(self):
        """
        Closes this room to further players, starting the game
        
        This method will return a dictionary of messages to send to the host and clients, 
        notifying them of the room closure. The dictionary keys will be connection UUIDs, 
        and the values will be the strings to submit.
        
        :return: a dictionary of messages to send to the host and clients
        :rtype:  ``dict``
        """
        if (self._state != ROOM_OPEN):
            return {}
        
        response = {}
        self._state = ROOM_CLOSED
        
        message  = {'id': self._root, 'type':'lobby','category':'session', 'status':'start'}
        players = [self.host]
        players.extend(self.clients)
        message['room'] = self.name
        message['host'] = self.host
        message['players'] = players
        
        data = json.dumps(message)
        
        response[self.host] = data
        for c in self._clients:
            response[c] = data
        
        return response
    
    def migrate(self,uuid):
        """
        Migrates this room to have the given game connection as host.
        
        This method will return a dictionary of messages to send to the host and clients, 
        notifying them of the host migration. The dictionary keys will be connection UUIDs, 
        and the values will be the strings to submit.
        
        :param uuid: The UUID of the game connection to assign as host
        :type uuid:  ``str``
        
        :return: a dictionary of messages to send to the host and clients
        :rtype:  ``dict``
        """
        response = {}
        if self.host:
            return response
        
        self._host = uuid
        self._clients.remove(uuid)
        
        message = {'id': self._root, 'type':'lobby','category':'promotion', 'status':'confirmed'}
        message['host'] = uuid
        players = [uuid]
        for c in self.clients:
            players.append(c)
        message['players'] = players
        
        data1 = json.dumps(message)
        response[uuid] = data1
        
        message['category'] = 'migration'
        message['status'] = 'attempt'
        
        data2 = json.dumps(message)
        for c in self.clients:
            response[c] = data2
        
        return response
    
    def refuse(self,uuid):
        """
        Refuses a migration request
        
        This will deduct from the migration attempts (one for each client). If this drops
        to zero, all candidates have refused and so we create a shutdown event. In that
        case this method will return a dictionary of messages to send to the clients, 
        notifying them of the host migration. The dictionary keys will be connection 
        UUIDs, and the values will be the strings to submit.
        
        If the attempts do not reach 0, this method will return None.
        
        :param uuid: The UUID of the game connection to assign as host
        :type uuid:  ``str``
        
        :return: a dictionary of messages to send to the host and clients
        :rtype:  ``None`` or ``dict``
        """
        if not self._state == ROOM_MIGRATING:
            return None
        
        self._attempts -= 1
        if (self._attempts == 0):
            response = {}
            message  = {'id': self._root, 'type':'lobby','category':'session', 'status':'shutdown'}
            data = json.dumps(message)
            if self.host:
                response[self.host] = data
            for c in self.clients:
                response[c] = data
            
            return response
        
        return None
     
    def restore(self):
        """
        Restores the room to its pre-migration state
        """
        self._state = self._prevs
    
    def _recycle(self):
        """
        Recycles the current room name, making it available
        
        The room name is set to None.
        """
        self.INUSE.remove(self._name)
        self._name = None
    
    def _acquire(self):
        """
        Acquires a new room number.
        
        The room number will be a unique string of ``DIGITS`` characters. It is
        guaranteed to represent a hexadecimal number. If no more strings are 
        available (because all identifiers are in use), this will raise a 
        RuntimeError
        
        If the current name is not None, it is recycled.
        
        :raises: a ``RuntimeError`` if there are no more available rooms
        """
        if not self.name is None:
            self._recycle()
        
        idnum = random.randint(0,16**self.DIGITS)
        goup = idnum < 8**self.DIGITS
        self._name = hex(idnum)[2:].upper()
        self._name = '0'*(self.DIGITS-len(self._name))+self._name
        while self._name in self.INUSE and len(self._name) <= self.DIGITS:
            if goup:
                idum += 1
            else:
                idum += 1
            self._name = hex(idnum)[2:].upper()
            self._name = '0'*(self.DIGITS-len(self._name))+self._name
        
        if len(self._name) == self.DIGITS:
            self.INUSE.add(self._name)
        else:
            raise RuntimeError("No more rooms available")


class LobbyServer(object):
    """
    A class representing a game server.
    
    This is the central lobby server. As such, it is a collection of async methods
    for processing websocket connections. The `run` method executes the central loop.
    """
    @property
    def global_lock(self):
        """
        A mutex lock for this game lobby server
        
        For the most part, Python data structures are thread safe for "atomic" operations.
        However, there are some operations in this class that need to be atomic, which are
        not. This mutex lock is provided to guarantee atomicity
        
        **Invariant**: Value  is an instance of ``asyncio.Lock``
        """
        return self._lock
    
    def __init__(self):
        """
        Initializes a new game lobby server
        
        See the CONNECTION SETTINGS constants for how to configure this server.
        """
        # Initialize logging (keep websockets and lobby as separate streams)
        weblog = logging.getLogger('websockets')
        weblog.setLevel(WEBSOCKET_LOG)
        weblog.addHandler(logging.StreamHandler(sys.stdout))
        self._logger = logging.getLogger(__name__)
        self._logger.setLevel(CUGLLOBBY_LOG)
        self._logger.addHandler(logging.StreamHandler(sys.stdout))
        
        self._sockets = {}
        self._clients = {}
        self._rooms = {}
        self._lock = asyncio.Lock()
        
        self._host = None
        if BINDPOINT:
            if len(BINDPOINT) == 1:
                self._host = BINDPOINT[0]
            else:
                self._host = BINDPOINT
        self._port = BINDPORT
        self._ssl_cert = SSL_CERT
        self._active = False
        self._endpoint = str(self._host) + ':' + str(self._port) if self._host else ':' + str(self._port)
        self._rootid = str(uuid.uuid5(uuid.NAMESPACE_DNS,self._endpoint))
        
    async def run(self):
        """
        Create the webserver and listen for new connections
        """
        self._active = True
        self._logger.info('LOBBY: {} listening on {}'.format(self._rootid,self._endpoint))
        
        if (self._ssl_cert):
            self._ssl_context = ssl.SSLContext(ssl.PROTOCOL_TLS_SERVER)
            self._ssl_context.load_cert_chain(self._ssl_cert)
        else:
            self._ssl_context = None
        
        server = await websockets.serve(self.handle_websocket, self._host, int(self._port), ssl=self._ssl_context)
        await server.wait_closed()
    
    async def handle_websocket(self, websocket, path=None):
        """
        Handles connection by a single websocket
        
        This connection will remain active until disconnected
        
        :param websocket:The websocket connection
        :type websocket: ``websocket``
        
        :param path: The URL path on connection
        :type path:  ``str``
        """
        # WORKAROUND FOR WEBSOCKET VERSION
        if path is None:
            path = websocket.request.path
        
        client_id = None
        try:
            splitted = path.split('/')
            splitted.pop(0)
            client_id = splitted.pop(0)
            
            self._logger.info('LOBBY: client {} connected'.format(client_id))
            self._sockets[client_id] = websocket
            
            # Do the room handshake
            await self.on_negotiate(client_id,websocket)
            
            while True:
                data = await websocket.recv()
                self._logger.info('LOBBY: client {} << {}'.format(client_id, data))
                
                message = json.loads(data)
                type = message['type']
                if type == 'lobby':
                    await self.on_session(client_id,message,websocket)
                else:
                    await self.on_signal(client_id,message)
        except websockets.exceptions.WebSocketException as e1:
            if client_id:
                await self.on_disconnect(client_id)
        except Exception as e2:
            self._logger.debug(traceback.format_exc())
            self._logger.debug(e2)
        finally:
            # Cleanup with no wait
            async with self.global_lock:
                if uuid in self._sockets:
                    del self._sockets[uuid]
                if uuid in self._clients:
                    room = self._clients[uuid]
                    name = room.name
                    del self._clients[uuid]
                    room.leave(uuid)
                    if room.state == ROOM_DELETED and name in self._rooms:
                        del self._rooms[name]
                        
    
    async def on_negotiate(self,uuid,socket):
        """
        Performs the initial room assignment handshake
        
        :param uuid: The UUID of the game connection
        :type uuid:  ``str``
        
        :param socket: The game connection socket
        :type socket: ``websocket``
        """
        self._logger.info("NEGOTIATE: {}".format(uuid))
        try:
            
            # Do the room handshake
            message = {'id':self._rootid,'type' : 'lobby', 'category' : 'room-assign', 'status':'handshake' }
            data = json.dumps(message)
            await socket.send(data)
            
            # Get the response
            data = await socket.recv()
            response = json.loads(data)
            self._logger.info('NEGOTIATE: {} sent {}'.format(uuid,response))
            
            broadcast = []
            try:
                # Make sure the response is value
                if response['category'] != 'room-assign':
                    raise RuntimeError("NEGOTIATE: category '{}' is not valid".format(response['category']))
                if response['status'] != 'request':
                    raise RuntimeError("NEGOTIATE: status '{}' is not valid".format(response['status']))
                if uuid in self._clients:
                    raise RuntimeError('NEGOTIATE: {} is already in room {}'.format(uuid,self._clients[uuid].name))
                
                if response['host']:
                    room = Room(self._rootid,response)
                    message['category'] = 'room-assign'
                    message['status'] = 'success'
                    message['host'] = room.host
                    message['room'] = room.name
                    message['players'] = []
                    data = json.dumps(message)
                    
                    self._logger.info('NEGOTIATE: {} is assigned room {} as host'.format(response['id'],room.name))
                    self._rooms[room.name] = room
                    self._clients[uuid] = room
                    
                    broadcast.append(socket.send(data))
                else:
                    name = response['room']
                    notifies = None
                    if not name in self._rooms:
                        self._logger.info('NEGOTIATE: could not find room {}'.format(name))
                        message['category'] = 'room-assign'
                        message['status'] = 'invalid'
                        data = json.dumps(message)
                        await socket.send(data)
                    else:
                        room = self._rooms[name]
                        async with room.lock:
                            messages = room.join(uuid,response['apiVersion'])
                        if len(messages) > 1:
                            self._clients[uuid] = room
                            self._logger.info('NEGOTIATE: {} is assigned room {} as client'.format(response['id'],room.name))
                        
                        async with self.global_lock:
                            for sid in messages:
                                if sid in self._sockets:
                                    broadcast.append(self._sockets[sid].send(messages[sid]))
            except:
                message['category'] = 'failed'
                data = json.dumps(message)
                broadcast.append(socket.send(data))
                self._logger.info(traceback.format_exc())
                self._logger.info('NEGOTIATE: {} failed to get a room'.format(response['id']))
            
            # Respond
            if broadcast:
                await asyncio.gather(*broadcast)
        
        except Exception as e:
            self._logger.debug(traceback.format_exc())
            self._logger.debug(e)
    
    async def on_session(self, uuid, message,socket):
        """
        Responds to requests or responses from the game clients during a session.
        
        Examples include starting or shutting down a game. In addition, this method
        handles the results of host migration.
        
        :param uuid: The UUID of the game connection
        :type uuid:  ``str``
        
        :param uuid: The incoming JSON message
        :type uuid:  ``dict``
        
        :param socket: The game connection socket
        :type socket: ``websocket``
        """
        self._logger.info("SESSION: {}".format(uuid))
        category = message['category']
        status   = message['status']
        
        try:
            room = None
            async with self.global_lock:
                if uuid in self._clients:
                    room = self._clients[uuid]
            
            if room == None:
                raise RuntimeError("{} has no assigned room".format(uuid))
            
            broadcast = []
            if category == 'session':
                if status == 'request':
                    responses = []
                    async with room.lock:
                        responses = room.close()
                    
                    if uuid in responses:
                        self._logger.info("SESSION: {} >> {}".format(responses[uuid],uuid))
                    
                    async with self.global_lock:
                        for sid in responses:
                            if sid in self._sockets:
                                broadcast.append(self._sockets[sid].send(responses[sid]))
                elif status == "shutdown":
                    responses = []
                    async with room.lock:
                        responses = room.dispose()
                    
                    if uuid in responses:
                        self._logger.info("SESSION: {} >> {}".format(responses[uuid],uuid))
                    
                    async with self.global_lock:
                        for sid in responses:
                            if sid in self._sockets:
                                broadcast.append(self._sockets[sid].send(responses[sid]))
            
            elif category == 'promotion':
                if status == 'response':
                    answer = message['response']
                    
                    responses = []
                    if answer:
                        async with room.lock:
                            if room.host is None:
                                responses = room.migrate(uuid)
                        
                        if uuid in responses:
                            self._logger.info("SESSION: {} >> {}".format(responses[uuid],uuid))
                        
                        async with self.global_lock:
                            for sid in responses:
                                if sid in self._sockets:
                                    broadcast.append(self._sockets[sid].send(responses[sid]))
                    else:
                        async with room.lock:
                            if room.host is None:
                                responses = room.refuse(uuid)
                        
                        if responses:
                            self._logger.info("All candidates refused host migration")
                            async with self.global_lock:
                                for sid in responses:
                                    if sid in self._sockets:
                                        broadcast.append(self._sockets[sid].send(responses[sid]))
                
                elif status == 'complete':
                    response = {'id':self._rootid,'type' : 'lobby', 'category' : 'migration', 'status':'complete' }
                    data = json.dumps(response)
                    
                    addr = [uuid]
                    async with room.lock:
                        room.restore()
                        addr.extend(room.clients)
                    
                    if uuid in addr:
                        self._logger.info("SESSION: {} >> {}".format(response,uuid))
                    
                    async with self.global_lock:
                        for sid in addr:
                            if sid in self._sockets:
                                broadcast.append(self._sockets[sid].send(data))
            
            if broadcast:
                await asyncio.gather(*broadcast)
        
        except Exception as e:
            self._logger.debug(traceback.format_exc())
            self._logger.debug(e)
    
    async def on_signal(self, uuid, message):
        """
        Performs Web RTC signaling
        
        :param uuid: The UUID of the game connection
        :type uuid:  ``str``
        
        :param uuid: The incoming JSON message
        :type uuid:  ``dict``
        """
        # TODO: Add room security? 
        # Right now we can signal outside of a room.
        self._logger.info("SIGNAL: {}".format(uuid))
        try:
            destination_id = message['id']
            destination_websocket = self._sockets.get(destination_id)
            
            if destination_websocket:
                message['id'] = uuid
                data = json.dumps(message)
                self._logger.info('SIGNAL: {} >> client {}'.format(data, destination_id))
                await destination_websocket.send(data)
            else:
                self._logger.debug('SIGNAL: client {} not found'.format(destination_id))
        except Exception as e:
            self._logger.debug(traceback.format_exc())
            self._logger.debug(e)
    
    async def on_disconnect(self, uuid):
        """
        Disconnects a game connection
        
        This method attempts a sophisticated disconnection, where all of the other 
        clients in the room are notified. In addition, if the disconnecting UUID is
        the room host, this will initiate the host migration process.
        
        :param uuid: The UUID of the game connection to disconnect
        :type uuid:  ``str``
        """
        try:
            if uuid in self._sockets:
                del self._sockets[uuid]
            
            broadcast1 = []
            broadcast2 = []
            async with self.global_lock:
                room = None
                if uuid in self._clients:
                    room = self._clients[uuid]
                    name = room.name
                    del self._clients[uuid]
                
                    async with room.lock:
                        messages = room.leave(uuid)
                        if room.state == ROOM_DELETED and name in self._rooms:
                            del self._rooms[name]
                        migrate  = room.state == ROOM_MIGRATING
                    
                    if migrate:
                        self._logger.info('LOBBY: host migration underway for room {}'.format(name))
                        migrmssg = {'id':self._rootid,'type' : 'lobby', 'category' : 'promotion', 'status':'query' }
                        migrdata = json.dumps(migrmssg)
                    
                    for sid in messages:
                        if sid in self._sockets:
                            broadcast1.append(self._sockets[sid].send(messages[sid]))
                            if migrate:
                                broadcast2.append(self._sockets[sid].send(migrdata))
            
            # Update the others that we left
            if broadcast1:
                await asyncio.gather(*broadcast1)
                
            # Second round of broadcasts on migration
            if broadcast2:
                await asyncio.gather(*broadcast2)
            
            self._logger.info('LOBBY: client {} disconnected'.format(uuid))
        except Exception as e:
            self._logger.debug(traceback.format_exc())
            self._logger.debug(e)


# The main entry point
if __name__ == '__main__':
    lobby = LobbyServer()
    asyncio.run(lobby.run())
