/*
 * point.cpp
 *
 * This is the implementation of a classic point class in C++
 *
 * This example is going to involve a lot of memory allocation and deallocation. 
 * Therefore, we have added memory addresses to the print statements.
 *
 * Walker M. White
 * February 6, 2015
 */


#include "point.h"
#include <stdio.h>
#include <math.h>

/** 
 * Create a new point object
 *
 * Note the use of default parameters.
 *
 * @param x  The initial x-coordinate
 * @param y  The initial y-coordinate
 * @param z  The initial z-coordinate
 */
Point::Point(float x, float y, float z) {
	printf("Creating point (%4.2f, %4.2f, %4.2f) @address %p\n", x, y, z, this);
	this->x = x;
	this->y = y;
	this->z = z;
}

/** 
 * Create a new point object
 *
 * This is copy constructor.  If you want to return a point from a function
 * without allocating to the heap, you will need a copy constructor.  This
 * means that the point is copied.  But if the data is small enough (it is
 * just three floats) that may not be a big deal.
 *
 * @param p  The point to copy
 * @param y  The initial y-coordinate
 * @param z  The initial z-coordinate
 */
Point::Point(const Point& p) {
	printf("Copying point (%4.2f, %4.2f, %4.2f) @address %p\n", p.x, p.y, p.z, this);
	x = p.x; y = p.y; z = p.z;
}
	
/**
 * Delete a point object
 *
 * This is ONLY necessary if the object contains other array, objects, or 
 * assets that it must delete once it is deleted.
 */
Point::~Point() {
	printf("Deleting point (%4.2f, %4.2f, %4.2f) @address %p\n", x, y, z, this);
}
	 
/**
  * Returns the string representation of this point.
  *
  * The value returned is obtained from sstream.  Because we are returning
  * the value and it is not on the heap, this will COPY the contents of the 
  * stringstream to the next stack, essentially building the string twice.
  *
  * @return the string representation of this point.
  */
string Point::toString() {
	stringstream format;
	format << "(" << x << ", " << y << ", " << z << ")";
	return format.str();
}
	
/**
 * Returns the distance from this point to other
 *
 * Note the Point&.  If you do not do this, you are COPYING the point whenever 
 * you call this function.  We do not want to copy this point, so we pass by
 * reference.
 *
 * @param other  Reference to point to measure distance to
 *
 * @return the distance from this point to other
 */
float Point::dist(const Point& other) {
	float x2 = (x-other.x)*(x-other.x);
	float y2 = (y-other.y)*(y-other.y);
	float z2 = (z-other.z)*(z-other.z);
	return sqrt(x2+y2+z2);
}
