/*
 * strings.cpp
 * 
 * Remember how we said that you needed new for things that are variable in length? 
 * That also includes strings. Fortunately, you do not need a new for string literals
 * (the things in quotes).  But you do need to do something when you want to concatenate
 * strings together, as that creates new strings, which requires variable allocation.
 *
 * As you want to avoid allocating to the heap whenever possible, C and C++ have a lot
 * of tools that allow you to concatenate strings without explicitly allocating memory.
 * This code sample demonstrates these.
 *
 * Walker M. White
 * February 6, 2015
 */
#include <stdio.h>  // For printf
#include <iostream> // For cout
#include <sstream> // For stringstream

using namespace std;

/**
 * Shows off string concatenation with printf
 * 
 *
 * The function printf is the closest to CCLOG() provided by Cocos2D-x:
 *
 * http://www.cocos2d-x.org/wiki/How_to_use_CCLOG
 *
 * It uses C-style formatting rules to "substitute" values into a string literal.  
 * You should know how to use these for debugging.
 *
 * @param s  a string
 * @param x  an int
 * @param y  a float
 */
void test_printf(string s, int x, float y) {
	// The C method ONLY allows C-strings, not C++ strings.
	// Note that we have to convert s to a C-String
	printf("This test has \"%s\", %d, and %4.2f\n",s.c_str(),x,y); 
}

/**
 * Shows off string concatenation with cout
 * 
 * When writing to the console, most C++ programmers prefer count.  The syntax is
 * a little more convenient than printf.  However, it does not work only mobile
 * devices, unlike CCLOG.
 *
 * @param s  a string
 * @param x  an int
 * @param y  a float
 */
void test_cout(string s, int x, float y) {
	cout << "This test has \"" << s << "\", " << x << ", and " << y << endl;
}

/**
 * Shows off string concatenation with sstream
 * 
 * Cocos2d-X really wants you to use C-style string formatting.  However, this 
 * can be extremely awkward, particularly when you want to put put text in a 
 * a label for a score on screen.
 *
 * The solution is to combine the approaches.  A stringstream is a buffere that
 * you can write to like cout.  But instead of writing to the console, it produces
 * a string for you.  You can then plug that into code that wants a C-string directly.
 *
 * @param s  a string
 * @param x  an int
 * @param y  a float
 */
void test_buffer(string s, int x, float y) {
	stringstream ss; // Local variable. String is lost when function ends
	ss << "This test has \"" << s << "\", " << x << ", and " << y << endl;
	
	// Get the string (NOT the C-string)
	string str = ss.str();
	
	// Now convert to C-string and display
	printf("%s",str.c_str());
}


/**
 * Demonstrate string formatting
 *
 * You will find when you work with Cocos2d-x that is is helpful to know both of the
 * primary ways to construct strings: C-style and C++ style.  This function compares
 * the two approaches.
 */
int main() {
	test_printf("apple", 10, 3.14159265359);
	test_cout("apple", 10, 3.14159265359);
	test_buffer("apple", 10, 3.14159265359);
	
	return 0;
}
