/***********************************************************************/
/*                                                                     */
/*   perceptron_learn.c                                                */
/*                                                                     */
/*   Learning module of Perceptron, using SVM light components         */
/*                                                                     */
/*   Author: Filip Radlinski                                           */
/*   Date: 02.24.04                                                    */
/*                                                                     */
/***********************************************************************/

# include "svm_common.h"
# include "svm_io.h"

void print_help(char *);
void init_model(MODEL *m, DOC *w, int, int);
void init_w(DOC *w, int);

int main (int argc, char* argv[]) {  

  char *docfile, *modelfile;
  int tmp, iterations, w_pos, doc_pos, i, errors, iter;
  DOC *docs, w;
  MODEL *model;
  double *label;
  long totwords, totdoc;
  double eta=1, R, tmp_R, prediction;

  /* Get parameters */
  if (argc != 4) {
    print_help(argv[0]);
    exit(1);
  }

  docfile = argv[1];
  modelfile = argv[2];
  iterations = 1;
  if (argc == 4) {
    tmp = sscanf(argv[3], "%i", &iterations);
    if (tmp == 0) {
      fprintf(stderr, "Error: Can't parse number of iterations '%s'\n", argv[3]);
      exit(1);
    }
  }

  /* Load the documents */
  SVML_read_documents(docfile, &docs, &label, &totwords, &totdoc);

  init_w(&w, totwords);
  model = malloc(sizeof(MODEL));
  init_model(model, &w, totwords, totdoc);
  
  /* Algorithm from class:
     - Given S((x_1,y_1)..(x_n,y_n))
     - Init w=0, b=0, eta>0

     FOR k ITERATIONS
       FOR i = 1 TO n
         IF y_i(w . x_i) <= 0 THEN
           w = w + eta y_i x_i
           b = b + eta y_i R^2
         END IF
       END FOR
     END

     - Output: w_k
  */

  /* Run the algorithm */
  /* Measure R */
  R = 0;
  for(i=0; i<totdoc; i++) {
    tmp_R = sqrt(sprod_ssIO(docs[i].words, docs[i].words));
    if (tmp_R > R)
      R = tmp_R;
  }
  
  printf("%li Documents in train set.\n", totdoc);
  printf("Value of R is: %5.2f\n", R);

  /* Repeat */
  for(iter=0; iter<iterations; iter++) {
    
    /* For every document */
    for(i=0; i<totdoc; i++) {
      
      prediction = sprod_ssIO(docs[i].words, w.words) + model->b;
  
      /* If its misclassified */
      if (label[i] * prediction <= 0) {
	
	/* Update w */
	w_pos=0;
	doc_pos=0;
	while((w.words[w_pos].wnum != 0)&&
	      (docs[i].words[doc_pos].wnum != 0)) {
	  if (w.words[w_pos].wnum < docs[i].words[doc_pos].wnum) 
	    w_pos++;
	  else if (w.words[w_pos].wnum > docs[i].words[doc_pos].wnum)
	    doc_pos++;
	  else {
	    w.words[w_pos].weight += eta * label[i] * docs[i].words[doc_pos].weight;
	    doc_pos++;
	    w_pos++;
	  }
	}
	
	/* Update b */
	model->b = model->b + eta*label[i]*R*R;
	
      } /* End if */
      
    } /* End for loop over docs */

  } /* End loop over iterations */
  
  /* Print out the result */
  errors = 0;
  for(i=0; i<totdoc; i++) {
    prediction = sprod_ssIO(docs[i].words, w.words) + model->b;
    if (label[i] * prediction <= 0)
      errors++;
  }

  printf("Result has training error: %5.2f%%\n", 100.0*errors/totdoc);

  /* Save the model. SVM light file format stores -b */
  model->b = -model->b;
  SVML_write_model(modelfile, model, 0);

  return(0);
}

void print_help(char *prog_name)
{
  printf("\nperceptron_learn learner\n");
  printf("   usage: %s example_file model_file iterations\n\n", prog_name);
  printf("Arguments:\n");
  printf("         example_file -> file with training data.\n");
  printf("         model_file   -> file to store learned decision rule in.\n");
  printf("         iterations   -> number of iterations to run for.\n");
}

void init_model(MODEL *model, DOC *w, int totwords, int totdoc) {

  /* Initialize the model, most values simply aren't used. */
  model->sv_num = 2;
  model->b = 0;
  model->supvec = malloc(2*sizeof(DOC*));
  model->supvec[1] = w;
  model->alpha = malloc(2*sizeof(double));
  model->alpha[1] = 1;
  model->totwords = totwords;
  model->totdoc = totdoc;
  model->kernel_parm.kernel_type = 0;
  model->kernel_parm.poly_degree = 1;
  model->kernel_parm.rbf_gamma = 1;
  strcpy(model->kernel_parm.custom, "empty");

}

void init_w(DOC *w, int totwords) {

  int i;

  /* Initialize the separating hyperplane, w */
  w->docnum=0;
  w->queryid=0;
  w->costfactor=1;
  w->words = malloc(sizeof(WORD)*totwords+1);
  w->words[totwords].wnum = 0;
  w->words[totwords].weight = 0;
  
  for(i=0; i<totwords; i++) {
    w->words[i].wnum = i+1;
    w->words[i].weight = 0;
  }

}
