# include "svm_common.h"
# include "svm_io.h"

/*
 * svm_light defines a global variable verbosity, but if svm_io is called
 * from elsewhere, verbosity may not be defined 
 */
# ifndef VERBOSITY
long verbosity = 0;
# endif

void SVML_write_model(char *modelfile, MODEL * model, long binary) {

    if (binary == 1) {
	SVML_write_model_binary(modelfile, model);
    } else {
	SVML_write_model_ascii(modelfile, model);
    }

}

void SVML_write_model_ascii(char *modelfile, MODEL * model) {

    FILE *modelfl;
    long j, i;

    if ((modelfl = fopen(modelfile, "w")) == NULL) {
	perror(modelfile);
	exit(1);
    }
    fprintf(modelfl, "SVM-light Version %s\n", VERSION);
    fprintf(modelfl, "%ld # kernel type\n",
	    model->kernel_parm.kernel_type);
    fprintf(modelfl, "%ld # kernel parameter -d \n",
	    model->kernel_parm.poly_degree);
    fprintf(modelfl, "%.8g # kernel parameter -g \n",
	    model->kernel_parm.rbf_gamma);
    fprintf(modelfl, "%.8g # kernel parameter -s \n",
	    model->kernel_parm.coef_lin);
    fprintf(modelfl, "%.8g # kernel parameter -r \n",
	    model->kernel_parm.coef_const);
    fprintf(modelfl, "%s# kernel parameter -u \n",
	    model->kernel_parm.custom);
    fprintf(modelfl, "%ld # highest feature index \n", model->totwords);
    fprintf(modelfl, "%ld # number of training documents \n",
	    model->totdoc);

    fprintf(modelfl, "%ld # number of support vectors plus 1 \n",
	    model->sv_num);
    fprintf(modelfl,
	    "%.8g # negative of threshold b (i.e. -b), each following line is a SV (starting with alpha*y)\n",
	    model->b);

    for (i = 1; i < model->sv_num; i++) {
	fprintf(modelfl, "%.32g ", model->alpha[i]);
	for (j = 0; ((model->supvec[i])->words[j]).wnum; j++) {
	  if (((model->supvec[i])->words[j]).weight != 0) 
	    fprintf(modelfl, "%ld:%.8g ",
		    (long) ((model->supvec[i])->words[j]).wnum,
		    (double) (((model->supvec[i])->words[j]).weight));
	}
	fprintf(modelfl, "\n");
    }
    fclose(modelfl);

}

void SVML_write_model_binary(char *modelfile, MODEL * model) {

    FILE *modelfl;
    long j, i, *lengths;
    double *twonorms;

    if ((modelfl = fopen(modelfile, "w")) == NULL) {
	perror(modelfile);
	exit(1);
    }
    fprintf(modelfl, "SVM-light Version %s-binary\n", VERSION);

    /*
     * Kernel parameters 
     */
    fwrite(&model->kernel_parm, sizeof(KERNEL_PARM), 1, modelfl);

    /*
     * highest feature index 
     */
    fwrite(&model->totwords, sizeof(long), 1, modelfl);
    /*
     * total number of documents 
     */
    fwrite(&model->totdoc, sizeof(long), 1, modelfl);
    /*
     * Number of support vectors plus 1 
     */
    fwrite(&model->sv_num, sizeof(long), 1, modelfl);
    /*
     * Threshold b 
     */
    fwrite(&model->b, sizeof(double), 1, modelfl);

    /*
     * The alphas, one more than we need 
     */
    fwrite(model->alpha, sizeof(double), model->sv_num, modelfl);

    /*
     * Compute the lengths of the support vectors. Save the two norms 
     */
    lengths = my_mallocIO(sizeof(long) * (model->sv_num - 1));
    twonorms = my_mallocIO(sizeof(double) * (model->sv_num - 1));
    for (i = 1; i < model->sv_num; i++) {
	for (j = 0; ((model->supvec[i])->words[j]).wnum; j++);
	lengths[i - 1] = j + 1;	/* +1 to save the zeros too */
	twonorms[i - 1] = (model->supvec[i])->twonorm_sq;
    }

    /*
     * Write the lengths 
     */
    fwrite(lengths, sizeof(long), model->sv_num - 1, modelfl);
    /*
     * Write the two-norms 
     */
    fwrite(twonorms, sizeof(double), model->sv_num - 1, modelfl);
    /*
     * Write the vectors 
     */
    for (i = 1; i < model->sv_num; i++)
	fwrite((model->supvec[i])->words, sizeof(WORD), lengths[i - 1],
	       modelfl);

    fclose(modelfl);
}

long SVML_read_model(char *modelfile, MODEL * model) {

    long llsv, max_words_sv, max_sv, rv, i;
    char version_buffer[100];
    char binary_version[100];
    FILE *modelfl;

    rv = 0;

    /*
     * Open file 
     */
    if ((modelfl = fopen(modelfile, "r")) == NULL) {
	perror(modelfile);
	exit(1);
    }

    /*
     * Check version, work out if ascii or binary 
     */
    sprintf(binary_version, "%s-binary", VERSION);
    i = fscanf(modelfl, "SVM-light Version %s\n", version_buffer);

    if (i != 1) {
	perror
	    ("Model file does not appear to be an SVM-light model file!");
	exit(1);
    }

    if (strcmp(version_buffer, VERSION) == 0) {

	/*
	 * scan size of model file 
	 */
	nol_ll(modelfile, &max_sv, &max_words_sv, &llsv);
	max_words_sv += 2;
	llsv += 2;

	model->supvec = (DOC **) my_mallocIO(sizeof(DOC *) * max_sv);
	model->alpha = (double *) my_mallocIO(sizeof(double) * max_sv);

	SVML_read_model_ascii(modelfl, model, max_words_sv, llsv);

    } else if (strcmp(version_buffer, binary_version) == 0) {

	SVML_read_model_binary(modelfl, model);
	rv = 1;

    } else {
	perror
	    ("Version of model-file does not match version of svm_classify!");
	exit(1);
    }

    /*
     * Close file 
     */
    fclose(modelfl);

    return rv;
}

void
SVML_read_model_ascii(FILE * modelfl, MODEL * model, long max_words,
		      long ll) {
    long j, i;
    char *line;
    WORD *words;
    register long wpos;
    long wnum, pos;
    double weight;
    long numread;

    words = (WORD *) my_mallocIO(sizeof(WORD) * (max_words + 10));
    line = (char *) my_mallocIO(sizeof(char) * ll);

    fscanf(modelfl, "%ld%*[^\n]\n", &model->kernel_parm.kernel_type);
    fscanf(modelfl, "%ld%*[^\n]\n", &model->kernel_parm.poly_degree);
    fscanf(modelfl, "%lf%*[^\n]\n", &model->kernel_parm.rbf_gamma);
    fscanf(modelfl, "%lf%*[^\n]\n", &model->kernel_parm.coef_lin);
    fscanf(modelfl, "%lf%*[^\n]\n", &model->kernel_parm.coef_const);
    fscanf(modelfl, "%[^#]%*[^\n]\n", model->kernel_parm.custom);

    fscanf(modelfl, "%ld%*[^\n]\n", &model->totwords);
    fscanf(modelfl, "%ld%*[^\n]\n", &model->totdoc);
    fscanf(modelfl, "%ld%*[^\n]\n", &model->sv_num);
    fscanf(modelfl, "%lf%*[^\n]\n", &model->b);

    for (i = 1; i < model->sv_num; i++) {
	fgets(line, (int) ll, modelfl);
	pos = 0;
	wpos = 0;
	sscanf(line, "%lf", &model->alpha[i]);
	while (!space_or_null((int) line[++pos]));
	while (((numread =
		 sscanf(line + pos, "%ld:%lf", &wnum, &weight)) != EOF)
	       && (wpos < max_words)) {
	    if (numread != 2) {
		perror("Parsing error while reading model!");
		printf("LINE: %s\n", line);
	    }
	    while (!space_or_null((int) line[++pos]));
	    words[wpos].wnum = wnum;
	    words[wpos].weight = (FVAL) weight;
	    wpos++;
	}
	model->supvec[i] = (DOC *) my_mallocIO(sizeof(DOC));
	(model->supvec[i])->words =
	    (WORD *) my_mallocIO(sizeof(WORD) * (wpos + 1));
	for (j = 0; j < wpos; j++) {
	    (model->supvec[i])->words[j] = words[j];
	}
	((model->supvec[i])->words[wpos]).wnum = 0;
	(model->supvec[i])->twonorm_sq =
	    sprod_ssIO((model->supvec[i])->words,
		       (model->supvec[i])->words);
	(model->supvec[i])->docnum = -1;
    }
    free(line);
    free(words);

}

void SVML_read_model_binary(FILE * modelfl, MODEL * model) {

    double *twonorms;
    long i, *lengths;

    /*
     * Kernel parameters 
     */
    fread(&model->kernel_parm, sizeof(KERNEL_PARM), 1, modelfl);

    /*
     * Model parameters 
     */
    fread(&model->totwords, sizeof(long), 1, modelfl);
    fread(&model->totdoc, sizeof(long), 1, modelfl);
    fread(&model->sv_num, sizeof(long), 1, modelfl);
    fread(&model->b, sizeof(double), 1, modelfl);

    /*
     * Alphas, one more than we really need 
     */
    model->alpha = my_mallocIO(sizeof(double) * model->sv_num);
    fread(model->alpha, sizeof(double), model->sv_num, modelfl);

    /*
     * Lengths of support vectors 
     */
    lengths = my_mallocIO(sizeof(long) * model->sv_num - 1);
    fread(lengths, sizeof(long), model->sv_num - 1, modelfl);

    /*
     * Read the twonorms 
     */
    twonorms = my_mallocIO(sizeof(double) * model->sv_num - 1);
    fread(twonorms, sizeof(double), model->sv_num - 1, modelfl);

    /*
     * supvec is one longer than it really needs to be (starts at 1) 
     */
    model->supvec = (DOC **) my_mallocIO(sizeof(DOC *) * model->sv_num);

    /*
     * Read the vectors 
     */
    for (i = 1; i < model->sv_num; i++) {

	/*
	 * Put the doc here (supvec is an array of pointers) 
	 */
	model->supvec[i] = (DOC *) my_mallocIO(sizeof(DOC));

	/*
	 * Get the twonorm 
	 */
	(model->supvec[i])->twonorm_sq = twonorms[i - 1];
	(model->supvec[i])->docnum = -1;

	/*
	 * Get the words 
	 */
	(model->supvec[i])->words =
	    (WORD *) my_mallocIO(sizeof(WORD) * lengths[i - 1]);
	((model->supvec[i])->words)->wnum = 1;
	fread((model->supvec[i])->words, sizeof(WORD), lengths[i - 1],
	      modelfl);

    }				/* End of loop over support vectors */

    free(lengths);
    free(twonorms);

}

int SVML_documents_isbinary(char *filename) {

    FILE *f;
    char version[100];
    int i;

    f = fopen(filename, "r");
    if (f == NULL) {
	perror(filename);
	exit(1);
    }
    i = fscanf(f, "SVML Binary Version %s", version);
    fclose(f);

    /*
     * Text document files have no such string at the top, so version is
     * uninitialized. So we should skip the test. 
     */
    if (i != 1)
	return 0;

    /*
     * If its a binary file of the right version, we say its binary 
     */
    if (strcmp(version, VERSION) == 0)
	return 1;
    else
	return 0;

}

long SVML_read_documents(char *docfile, DOC ** docs, double **label,
			 long *totwords, long *totdoc) {

    long binary;

    binary = SVML_documents_isbinary(docfile);

    if (binary == 1) {
	if (verbosity >= 1)
	    printf("Reading binary document file.\n");
	SVML_read_documents_binary(docfile, docs, label, totwords, totdoc);
    } else {
	if (verbosity >= 1)
	    printf("Reading ascii document file.\n");
	SVML_read_documents_ascii(docfile, docs, label, totwords, totdoc);
    }

    return binary;
}

void SVML_read_documents_ascii(char *docfile, DOC ** docs, double **label,
			       long *totwords, long *totdoc) {

    long max_docs, max_words_doc, ll;
    char *line;
    DOC doc;
    long dnum = 0, wpos, i, dpos = 0, dneg = 0, dunlab = 0;
    double doc_label;
    FILE *docfl;

    if (verbosity >= 1) {
	printf("Scanning examples...");
	fflush(stdout);
    }

    nol_ll(docfile, &max_docs, &max_words_doc, &ll);	/* scan size of
							 * input file */
    max_words_doc += 2;
    ll += 2;
    max_docs += 2;

    if (verbosity >= 1) {
	printf("done\n");
	fflush(stdout);
    }

    *docs = (DOC *) my_mallocIO(sizeof(DOC) * max_docs);	/* feature 
								 * vectors 
								 */
    *label = (double *) my_mallocIO(sizeof(double) * max_docs);	/* target
								 * values */

    line = (char *) my_mallocIO(sizeof(char) * ll);

    if ((docfl = fopen(docfile, "r")) == NULL) {
	perror(docfile);
	exit(1);
    }

    doc.words = (WORD *) my_mallocIO(sizeof(WORD) * (max_words_doc + 10));
    dnum = 0;
    (*totwords) = 0;
    while ((!feof(docfl)) && fgets(line, (int) ll, docfl)) {
	if (line[0] == '#')
	    continue;		/* line contains comments */
	if (!parse_document(line, &doc, &doc_label, &wpos, max_words_doc)) {
	    printf("\nParsing error in line %ld!\n%s", dnum, line);
	    exit(1);
	}
	(*label)[dnum] = doc_label;
	/*
	 * printf("docnum=%ld: Class=%f ",dnum,doc_label); 
	 */
	if (doc_label > 0)
	    dpos++;
	if (doc_label < 0)
	    dneg++;
	if (doc_label == 0)
	    dunlab++;
	if ((wpos > 1) && ((doc.words[wpos - 2]).wnum > (*totwords)))
	    (*totwords) = (doc.words[wpos - 2]).wnum;
	if ((*totwords) > MAXFEATNUM) {
	    printf
		("\nMaximum feature number exceeds limit defined in MAXFEATNUM!\n");
	    printf("LINE: %s\n", line);
	    exit(1);
	}
	(*docs)[dnum].queryid = doc.queryid;
	(*docs)[dnum].costfactor = doc.costfactor;
	(*docs)[dnum].words = (WORD *) my_mallocIO(sizeof(WORD) * (wpos));
	(*docs)[dnum].docnum = dnum;
	for (i = 0; i < wpos; i++) {
	    (*docs)[dnum].words[i] = doc.words[i];
	    /*
	     * printf("%ld::%f
	     * ",(docs[dnum].words[i]).wnum,(docs[dnum].words[i]).weight); 
	     */

	}
	(*docs)[dnum].twonorm_sq = doc.twonorm_sq;
	/*
	 * printf("\nNorm=%f\n",docs[dnum].twonorm_sq); 
	 */
	dnum++;
	if (verbosity >= 1) {
	    if ((dnum % 100) == 0) {
		printf("%ld..", dnum);
		fflush(stdout);
	    }
	}
    }

    fclose(docfl);
    free(line);
    free(doc.words);
    (*totdoc) = dnum;

}

void SVML_read_documents_binary(char *docfile, DOC ** docs, double **label,
				long *totwords, long *totdoc) {

    FILE *f;
    char junk[200];
    long *nwords, *qids, i;
    double *costs;

    f = fopen(docfile, "r");
    if (f == NULL) {
	perror(docfile);
	exit(1);
    }

    /*
     * Remove first line 
     */
    fgets(junk, 200, f);

    fread(totdoc, sizeof(long), 1, f);

    /*
     * Allocate docs and labels 
     */
    *docs = my_mallocIO(sizeof(DOC) * (*totdoc));
    *label = my_mallocIO(sizeof(double) * (*totdoc));

    /*
     * Allocate space for words 
     */
    nwords = my_mallocIO(sizeof(long) * (*totdoc));
    fread(nwords, sizeof(long), *totdoc, f);
    for (i = 0; i < *totdoc; i++) {
	(*docs)[i].words = my_mallocIO(sizeof(WORD) * nwords[i]);
    }

    /*
     * Read the labels 
     */
    fread(*label, sizeof(double), *totdoc, f);

    /*
     * Read qids and costs 
     */
    qids = my_mallocIO(sizeof(long) * (*totdoc));
    costs = my_mallocIO(sizeof(double) * (*totdoc));
    fread(qids, sizeof(long), *totdoc, f);
    fread(costs, sizeof(double), *totdoc, f);
    for (i = 0; i < *totdoc; i++) {
	(*docs)[i].queryid = qids[i];
	(*docs)[i].costfactor = costs[i];
    }

    /*
     * Read words, initialize other stuff 
     */
    *totwords = 0;
    for (i = 0; i < *totdoc; i++) {
	fread((*docs)[i].words, sizeof(WORD), nwords[i], f);
	if ((*docs)[i].words[nwords[i] - 2].wnum > *totwords)
	    *totwords = (*docs)[i].words[nwords[i] - 2].wnum;
	(*docs)[i].docnum = i;
	(*docs)[i].twonorm_sq =
	    sprod_ssIO((*docs)[i].words, (*docs)[i].words);
    }

    fclose(f);

    free(nwords);
    free(costs);
    free(qids);

    return;
}

void SVML_write_documents(char *docfile, DOC * docs, double *labels,
			  long n_docs, long binary) {

    if (binary == 1)
	SVML_write_documents_binary(docfile, docs, labels, n_docs);
    else
	SVML_write_documents_ascii(docfile, docs, labels, n_docs);

    return;
}

void SVML_write_documents_ascii(char *docfile, DOC * docs, double *labels,
				long n_docs) {

    FILE *f;
    long i;
    WORD *w;

    f = fopen(docfile, "w");
    if (f == NULL) {
	perror(docfile);
	exit(1);
    }

    for (i = 0; i < n_docs; i++) {
	if ((labels[i] == 1) || (labels[i] == -1) || (labels[i] == 0))
	    fprintf(f, "%+.0f", labels[i]);
	else
	    fprintf(f, "%.8f", labels[i]);

	if (docs[i].queryid != 0)
	    fprintf(f, " qid:%li", docs[i].queryid);

	if (docs[i].costfactor != 1)
	    fprintf(f, " cost:%e", docs[i].costfactor);

	w = docs[i].words;
	while (w->wnum != 0) {
	    fprintf(f, " %li:%.6e", w->wnum, w->weight);
	    w++;
	}
	fprintf(f, "\n");
    }

    fclose(f);

    return;
}

void SVML_write_documents_binary(char *docfile, DOC * docs, double *labels,
				 long n_docs) {

    FILE *f;
    long i, *nwords, *qids;
    double *costs;
    WORD *w;

    f = fopen(docfile, "w");
    if (f == NULL) {
	perror(docfile);
	exit(1);
    }

    fprintf(f, "SVML Binary Version %s\n", VERSION);

    /*
     * Number of documents 
     */
    fwrite(&n_docs, sizeof(long), 1, f);

    /*
     * Count number of words in each doc 
     */
    nwords = my_mallocIO(sizeof(long) * n_docs);
    for (i = 0; i < n_docs; i++) {
	nwords[i] = 1;		/* Save the null word at the end */
	w = docs[i].words;
	while (w->wnum != 0) {
	    w++;
	    nwords[i]++;
	}
    }

    /*
     * Write the lengths 
     */
    fwrite(nwords, sizeof(long), n_docs, f);

    /*
     * Write the labels 
     */
    fwrite(labels, sizeof(double), n_docs, f);

    /*
     * Write the qids and costs 
     */
    qids = my_mallocIO(sizeof(long) * n_docs);
    costs = my_mallocIO(sizeof(double) * n_docs);
    for (i = 0; i < n_docs; i++) {
	qids[i] = docs[i].queryid;
	costs[i] = docs[i].costfactor;
    }
    fwrite(qids, sizeof(long), n_docs, f);
    fwrite(costs, sizeof(double), n_docs, f);

    /*
     * Write the words 
     */
    for (i = 0; i < n_docs; i++)
	fwrite(docs[i].words, sizeof(WORD), nwords[i], f);

    fclose(f);

    free(nwords);
    free(costs);
    free(qids);

    return;
}

int parse_document(char *line, DOC * doc, double *label,
		   long int *numwords, long int max_words_doc) {
  
    register long wpos, pos;
    long wnum;
    double weight;
    int numread;
    char featurepair[1000], junk[1000];

    doc->queryid = 0;
    doc->costfactor = 1;

    pos = 0;
    while (line[pos]) {		/* cut off comments */
	if (line[pos] == '#') {
	    line[pos] = 0;
	} else {
	    pos++;
	}
    }
    wpos = 0;
    /*
     * check, that line starts with target value or zero, but not with
     * feature pair 
     */
    if (sscanf(line, "%s", featurepair) == EOF)
	return (0);
    pos = 0;
    while ((featurepair[pos] != ':') && featurepair[pos])
	pos++;
    if (featurepair[pos] == ':') {
	perror("Line must start with label or 0!!!\n");
	printf("LINE: %s\n", line);
	exit(1);
    }
    /*
     * read the target value 
     */
    if (sscanf(line, "%lf", label) == EOF)
	return (0);
    pos = 0;
    while (space_or_null((int) line[pos]))
	pos++;
    while ((!space_or_null((int) line[pos])) && line[pos])
	pos++;
    while (((numread = sscanf(line + pos, "%s", featurepair)) != EOF) &&
	   (wpos < max_words_doc)) {
	/*
	 * printf("%s\n",featurepair); 
	 */
	while (space_or_null((int) line[pos]))
	    pos++;
	while ((!space_or_null((int) line[pos])) && line[pos])
	    pos++;
	if (sscanf(featurepair, "qid:%ld%s", &wnum, junk) == 1) {
	    /*
	     * it is the query id 
	     */
	    doc->queryid = (long) wnum;
	} else if (sscanf(featurepair, "cost:%lf%s", &weight, junk) == 1) {
	    /*
	     * it is the example-dependent cost factor 
	     */
	    doc->costfactor = (double) weight;
	} else if (sscanf(featurepair, "%ld:%lf%s", &wnum, &weight, junk)
		   == 2) {
	    /*
	     * it is a regular feature 
	     */
	    if (wnum <= 0) {
		perror
		    ("Feature numbers must be larger or equal to 1!!!\n");
		printf("LINE: %s\n", line);
		exit(1);
	    }
	    if ((wpos > 0) && ((doc->words[wpos - 1]).wnum >= wnum)) {
		perror("Features must be in increasing order!!!\n");
		printf("Got %li then %li\n", doc->words[wpos - 1].wnum,
		       wnum);
		printf("LINE: %s\n", line);
		exit(1);
	    }
	    (doc->words[wpos]).wnum = wnum;
	    (doc->words[wpos]).weight = (FVAL) weight;
	    wpos++;
	} else {
	    perror("Cannot parse feature/value pair!!!\n");
	    printf("'%s' in LINE: %s\n", featurepair, line);
	    exit(1);
	}
    }
    (doc->words[wpos]).wnum = 0;
    (*numwords) = wpos + 1;
    doc->docnum = -1;
    doc->twonorm_sq = sprod_ssIO(doc->words, doc->words);
    return (1);
}

/*
 * We need svm_io.o to be a standalone .o file, so we need to include
 * these 
 */

void *my_mallocIO(size_t size) {
  
    void *ptr;
    ptr = (void *) malloc(size);
    if (!ptr) {
	perror("Out of memory!\n");
	exit(1);
    }
    return (ptr);
}

int space_or_null(int c) {
  
    if (c == 0)
	return 1;
    return isspace(c);
}

void nol_ll(char *file, long int *nol, long int *wol, long int *ll) {
/*
 * Grep through file and count number of lines, maximum number of spaces
 * per line, and longest line. 
 */
  
    FILE *fl;
    int ic;
    char c;
    long current_length, current_wol;

    if ((fl = fopen(file, "r")) == NULL) {
	perror(file);
	exit(1);
    }
    current_length = 0;
    current_wol = 0;
    (*ll) = 0;
    (*nol) = 1;
    (*wol) = 0;
    while ((ic = getc(fl)) != EOF) {
	c = (char) ic;
	current_length++;
	if (space_or_null((int) c)) {
	    current_wol++;
	}
	if (c == '\n') {
	    (*nol)++;
	    if (current_length > (*ll)) {
		(*ll) = current_length;
	    }
	    if (current_wol > (*wol)) {
		(*wol) = current_wol;
	    }
	    current_length = 0;
	    current_wol = 0;
	}
    }
    fclose(fl);
}

double sprod_ssIO(WORD * a, WORD * b) {
/*
 * compute the inner product of two sparse vectors 
 */

    register CFLOAT sum = 0;
    register WORD *ai, *bj;
    ai = a;
    bj = b;
    while (ai->wnum && bj->wnum) {
	if (ai->wnum > bj->wnum) {
	    bj++;
	} else if (ai->wnum < bj->wnum) {
	    ai++;
	} else {
	    sum += (CFLOAT) (ai->weight) * (CFLOAT) (bj->weight);
	    ai++;
	    bj++;
	}
    }
    return ((double) sum);
}
