#include <cassert>
#include <cmath>
#include <iostream>
#include <opencv/cv.h>
#include <opencv/highgui.h>
#include <opencv/cxcore.h>
#include "Features.h"

// This is the entry point for all panorama generation.  The output image will
// be allocated by your code and in particular should be allocated from a call
// to compositeImages.  This function will also depend on ransacHomography
// in order to compute a best homography for the pair of images.  You should
// use computeFeatures and matchFeatures when necessary.
IplImage * constructPanorama ( IplImage *     img1,
                               IplImage *     img2,
                               int            featureType,
                               int matchType
#ifdef                                        Q_WS_MAEMO_5
                               ,
                               Progressable * thread
#endif
                               )
{
  assert ( img1->depth == IPL_DEPTH_32F );
  assert ( img1->nChannels == 3 );
  assert ( img2->depth == IPL_DEPTH_32F );
  assert ( img2->nChannels == 3 );
  // @@@ TODO Project 2b
  assert ( 0 ); // Remove this when ready
}

// img1 and img2 are color images that you want to make into a panorama by
// applying the homography, h, to img2.  This function needs to determine the
// size of the output image and allocate the memory for it.
IplImage * compositeImages ( IplImage *     img1,
                             IplImage *     img2,
                             CvMat *        h
#ifdef                                      Q_WS_MAEMO_5
                             ,
                             Progressable * thread
#endif
                             )
{
  assert ( img1->depth == IPL_DEPTH_32F );
  assert ( img1->nChannels == 3 );
  assert ( img2->depth == IPL_DEPTH_32F );
  assert ( img2->nChannels == 3 );

  // @@@ TODO Project 2b
  assert ( 0 ); // Remove this when ready
}

// Compute features of an image.
bool computeFeatures ( IplImage *     image,
                       FeatureSet &   features,
                       int featureType
#ifdef \
                       Q_WS_MAEMO_5
                       ,
                       Progressable * thread
#endif
                       )
{
  assert ( image->depth == IPL_DEPTH_32F );
  assert ( image->nChannels == 3 );
  // Instead of calling dummyComputeFeatures, write your own
  // feature computation routines and call them here.
  switch ( featureType )
  {
  case 1:
    dummyComputeFeatures ( image, features
#ifdef Q_WS_MAEMO_5
                           , thread
#endif
                           );
    break;
  case 2:
    ComputeMOPSFeatures ( image, features
#ifdef Q_WS_MAEMO_5
                          , thread
#endif
                          );
    break;
  case 3:
    ComputeSURFFeatures ( image, features
#ifdef Q_WS_MAEMO_5
                          , thread
#endif
                          );
    break;
  default:
    return false;
  }

  // This is just to make sure the IDs are assigned in order, because
  // the ID gets used to index into the feature array.
  for ( size_t i = 0; i < features.size (); ++i )
  {
    features [i].id = i + 1;
  }

  return true;
}

void ComputeSURFFeatures ( IplImage * image, FeatureSet & features
#ifdef Q_WS_MAEMO_5
                           , Progressable * thread
#endif
                           )
{
  cv::SURF surf;

  std::vector<cv::KeyPoint> keypoints;
  std::vector<float> descriptors;
  IplImage * temp = cvCreateImage ( cvGetSize ( image ), IPL_DEPTH_8U, 3 );
  cvConvertScale ( image, temp, 255, 0 );
  IplImage * temp2 = cvCreateImage ( cvGetSize ( image ), IPL_DEPTH_8U, 1 );
  cvCvtColor ( temp, temp2, CV_RGB2GRAY );
  cvReleaseImage ( &temp );
  surf ( temp2, cv::Mat (), keypoints, descriptors );
  cvReleaseImage ( &temp );
  features.clear ();
  features.reserve ( keypoints.size () );
  for ( size_t i = 0; i < keypoints.size (); ++i )
  {
#ifdef Q_WS_MAEMO_5
    if ( thread && i % ( keypoints.size () / 100 ) == 0 )
    {
      thread->emitProgressUpdate ( 100 * i / keypoints.size () );
    }
#endif
    Feature f;
    f.x = keypoints[i].pt.x;
    f.y = keypoints[i].pt.y;
    f.type = 3;
    f.id = 0;
    f.angleRadians = keypoints[i].angle;
    f.data.reserve ( surf.descriptorSize () );
    for ( int j = 0; j < surf.descriptorSize (); ++j )
    {
      f.data.push_back ( descriptors [i * surf.descriptorSize () + j] );
    }
    features.push_back ( f );
  }
}

// Perform a query on the database.  This simply runs matchFeatures on
// each image in the database, and returns the feature set of the best
// matching image.
bool performQuery ( const FeatureSet &     f,
                    const ImageDatabase &  db,
                    size_t &               bestIndex,
                    vector<FeatureMatch> & bestMatches,
                    float &
                    bestScore,
                    int matchType
#ifdef                                     Q_WS_MAEMO_5
                    ,
                    Progressable *         thread
#endif
                    )
{
  bestScore = 1e100;
  bestIndex = 0;


  for ( size_t i = 0; i < db.size (); ++i )
  {
    vector<FeatureMatch> tempMatches;
    float tempScore = 0;
    if ( !matchFeatures ( f, db [i].features, tempMatches, tempScore,
                          matchType
#ifdef Q_WS_MAEMO_5
                          , thread
#endif
                          ) )
    {
      return false;
    }

    if ( tempScore < bestScore )
    {
      bestIndex = i;
      bestScore = tempScore;
      bestMatches = tempMatches;
    }
  }
  return true;
}

// Match one feature set with another.
bool matchFeatures ( const FeatureSet &     f1,
                     const FeatureSet &     f2,
                     vector<FeatureMatch> & matches,
                     float &                totalScore,
                     int matchType
#ifdef                                      Q_WS_MAEMO_5
                     ,
                     Progressable *         thread
#endif
                     )
{
  // We have given you the ssd matching function, you must write your own
  // feature matching function for the ratio test.

  std::cout << std::endl << "Matching features......." << std::endl;

  switch ( matchType )
  {
  case 1:
    ssdMatchFeatures ( f1, f2, matches, totalScore
#ifdef Q_WS_MAEMO_5
                       , thread
#endif
                       );
    return true;
  case 2:
    ratioMatchFeatures ( f1, f2, matches, totalScore
#ifdef Q_WS_MAEMO_5
                         , thread
#endif
                         );
    return true;
  default:
    return false;
  }
}

// Evaluate a match using a ground truth homography.  This computes the
// average SSD distance between the matched feature points and
// the actual transformed positions.
float evaluateMatch ( const FeatureSet &           f1,
                      const FeatureSet &           f2,
                      const vector<FeatureMatch> & matches,
                      float                        h [9] )
{
  float d = 0;
  int n = 0;

  float xNew;
  float yNew;

  size_t num_matches = matches.size ();

  for ( size_t i = 0; i < num_matches; ++i )
  {
    int id1 = matches [i].id1;
    int id2 = matches [i].id2;
    assert ( id1 );
    assert ( id2 );
    applyHomography ( f1 [id1 - 1].x, f1 [id1 - 1].y, xNew, yNew, h );
    d +=
      sqrt ( pow ( xNew - f2 [id2 - 1].x,
                   2 ) + pow ( yNew - f2 [id2 - 1].y, 2 ) );
    ++n;
  }

  return d / n;
}

void addRocData ( const FeatureSet &           f1,
                  const FeatureSet &           f2,
                  const vector<FeatureMatch> & matches,
                  float                        h [9],
                  vector<bool> &
                  isMatch,
                  float                        threshold,
                  float &
                  maxD )
{
  float d = 0;

  maxD = 0;
  float xNew;
  float yNew;

  size_t num_matches = matches.size ();

  for ( size_t i = 0; i < num_matches; ++i )
  {
    int id1 = matches [i].id1;
    int id2 = matches [i].id2;
    assert ( id1 );
    assert ( id2 );
    applyHomography ( f1 [id1 - 1].x, f1 [id1 - 1].y, xNew, yNew, h );

    // Ignore unmatched points.  There might be a better way to
    // handle this.
    d =
      sqrt ( pow ( xNew - f2 [id2 - 1].x,
                   2 ) + pow ( yNew - f2 [id2 - 1].y, 2 ) );
    if ( d <= threshold )
    {
      isMatch.push_back ( 1 );
    }
    else
    {
      isMatch.push_back ( 0 );
    }

    if ( matches [i].score > maxD )
    {
      maxD = matches [i].score;
    }
  }
}

vector<ROCPoint> computeRocCurve ( vector<FeatureMatch> & matches,
                                   vector<bool> &         isMatch,
                                   vector<float> &        thresholds )
{
  vector<ROCPoint> dataPoints;

  for ( size_t i = 0; i < thresholds.size (); ++i )
  {
    int tp = 0;
    int actualCorrect = 0;
    int fp = 0;
    int actualError = 0;
    int total = 0;

    size_t num_matches = matches.size ();
    for ( size_t j = 0; j < num_matches; ++j )
    {
      if ( isMatch [j] )
      {
        actualCorrect++;
        if ( matches [j].score < thresholds [i] )
        {
          tp++;
        }
      }
      else
      {
        actualError++;
        if ( matches [j].score < thresholds [i] )
        {
          fp++;
        }
      }
      total++;
    }

    ROCPoint newPoint;
    newPoint.trueRate = ( float( tp ) / actualCorrect );
    newPoint.falseRate = ( float( fp ) / actualError );

    dataPoints.push_back ( newPoint );
  }

  return dataPoints;
}


// Compute silly example features.  This doesn't do anything
// meaningful.
void dummyComputeFeatures ( IplImage * image, FeatureSet & features
#ifdef Q_WS_MAEMO_5
                            , Progressable * thread
#endif
                            )
{
  assert ( image->depth == IPL_DEPTH_32F );
  assert ( image->nChannels == 3 );

  Feature f;
#ifdef Q_WS_MAEMO_5
  int total_size = image->height * image->width;
#endif
  int count = 0;
  for ( int y = 0; y < image->height; ++y )
  {
    for ( int x = 0; x < image->width; ++x )
    {
// @@@ Nothing to do here, but take note of how you can notify the UI on the
// phone to display the status on a progress bar.  Values range from 0 - 100%
#ifdef Q_WS_MAEMO_5
      if ( thread && count % ( total_size / 100 ) == 0 )
      {
        thread->emitProgressUpdate ( 100 * count / total_size );
      }
#endif
      ++count;
      float r = CV_IMAGE_ELEM ( image, float, y, 3 * x );
      float g = CV_IMAGE_ELEM ( image, float, y, 3 * x + 1 );
      float b = CV_IMAGE_ELEM ( image, float, y, 3 * x + 2 );
      if ( ( int )( 255 * ( r + g + b ) + 0.5 ) % 100 == 1 )
      {
        // If the pixel satisfies this meaningless criterion,
        // make it a feature.

        f.type = 1;
        f.id += 1;
        assert ( f.id );
        f.x = x;
        f.y = y;

        f.data.resize ( 1 );
        f.data [0] = r + g + b;

        features.push_back ( f );
      }
    }
  }
}

void ComputeMOPSFeatures ( IplImage * image, FeatureSet & features
#ifdef Q_WS_MAEMO_5
                           , Progressable * thread
#endif
                           )
{
  assert ( image->depth == IPL_DEPTH_32F );
  assert ( image->nChannels == 3 );

  //Create grayscale image used for MOPS detection
  IplImage * grayImage =
    cvCreateImage ( cvGetSize ( image ), IPL_DEPTH_32F, 1 );
  cvCvtColor ( image, grayImage, CV_RGB2GRAY );

  //Create image to store MOPS values
  IplImage * harrisImage =
    cvCreateImage ( cvGetSize ( image ), IPL_DEPTH_32F, 1 );

  //Create image to store local maximum harris values as 255, other pixels 0
  IplImage * harrisMaxImage =
    cvCreateImage ( cvGetSize ( image ), IPL_DEPTH_32F, 1 );

  //TO DO--------------------------------------------------------------------
  //function puts harris values at each pixel position in harrisImage and
  // orientations at each pixel position in orientationImage
  IplImage * orientationImage =
    cvCreateImage ( cvGetSize ( image ), IPL_DEPTH_32F, 1 );
  //@@@PROJECT 2b: Fill in computeMOPSValues if you want to use MOPS
  computeMOPSValues ( grayImage, harrisImage, orientationImage );
  cvSaveImage ( "harris.png", harrisImage );

  //TO DO---------------------------------------------------------------------
  //Loop through harrisValues and find the best features in a local 3x3 maximum
  //compute the feature descriptor
  //@@@PROJECT 2b: Fill in computeLocalMaxima if you want to use MOPS
  computeLocalMaxima ( harrisImage, harrisMaxImage );
  cvSaveImage ( "harrisMax.png", harrisMaxImage );

  //TO DO--------------------------------------------------------------------
  //Loop through feature points in harrisMaxImage and create feature descriptor
  //for each point above a threshold
  //@@@PROJECT 2b: Copy your project 2a code here if you want to use MOPS


  cvReleaseImage ( &orientationImage );
  cvReleaseImage ( &harrisImage );
  cvReleaseImage ( &grayImage );
  cvReleaseImage ( &harrisMaxImage );
}

//TO DO---------------------------------------------------------------------
//Loop through the image to compute the harris corner values as described in
//class
// srcImage:  grayscale of original image
// harrisImage:  populate the harris values per pixel in this image
// orientationImage: populate the orientation in this image
void computeMOPSValues ( IplImage * srcImage,
                         IplImage * harrisImage,
                         IplImage * orientationImage )
{
  assert ( srcImage->depth == IPL_DEPTH_32F );
  assert ( harrisImage->depth == IPL_DEPTH_32F );
  assert ( orientationImage->depth == IPL_DEPTH_32F );
  assert ( srcImage->nChannels == 1 );
  assert ( harrisImage->nChannels == 1 );
  assert ( orientationImage->nChannels == 1 );

  //@@@PROJECT 2b: Copy your project 2a code here if you want to use MOPS
}

//Loop through the image to compute the harris corner values as described in
//class
// srcImage:  image with MOPS values
// destImage: Assign 1 to local maximum in 3x3 window, 0 otherwise
void computeLocalMaxima ( IplImage * srcImage, IplImage * destImage )
{
  assert ( srcImage->depth == IPL_DEPTH_32F );
  assert ( destImage->depth == IPL_DEPTH_32F );
  assert ( srcImage->nChannels == 1 );
  assert ( destImage->nChannels == 1 );

  //@@@PROJECT 2b: Copy your project 2a code here if you want to use MOPS
}


// Perform simple feature matching.  This just uses the SSD
// distance between two feature vectors, and matches a feature in the
// first image with the closest feature in the second image.  It can
// match multiple features in the first image to the same feature in
// the second image.
void ssdMatchFeatures ( const FeatureSet &     f1,
                        const FeatureSet &     f2,
                        vector<FeatureMatch> & matches,
                        float &                totalScore
#ifdef                                         Q_WS_MAEMO_5
                        ,
                        Progressable *         thread
#endif
                        )
{
  // @@@ You can leave is as-is since you will probably want to use the ratio
  // test anyway.  It is okay if you want to replace this with FLANN code such
  // as what we provide in the ratio test function.
  totalScore = 0;
#ifdef Q_WS_MAEMO_5
  int count = 0;
  int totalCount = f1.size ();
#endif
  for ( FeatureSet::const_iterator i1 = f1.begin (); i1 != f1.end (); ++i1 )
  {
#ifdef Q_WS_MAEMO_5
    if ( thread && count % ( totalCount / 100 ) == 0 )
    {
      thread->emitProgressUpdate ( 100 * count / totalCount );
    }
    ++count;
#endif
    size_t bestMatch = 1;
    float bestScore = 1e100;
    for ( FeatureSet::const_iterator i2 = f2.begin (); i2 != f2.end (); ++i2 )
    {
      float score = 0;
      assert ( i1->data.size () == i2->data.size () );
      for ( size_t i = 0; i < i1->data.size (); ++i )
      {
        float diff = i1->data[i] - i2->data[i];
        score += diff * diff;
      }
      if ( score < bestScore )
      {
        bestMatch = i2->id;
        bestScore = score;
      }
    }
    FeatureMatch m;
    m.id1 = i1->id;
    m.id2 = f2[bestMatch - 1].id;
    m.score = bestScore;
    totalScore += m.score;
    matches.push_back ( m );
  }
}

// This just uses the ratio of the SSD distance of the two best matches
// and matches a feature in the first image with the closest feature in the
// second image.
// It can match multiple features in the first image to the same feature in
// the second image.  (See class notes for more information)
void ratioMatchFeatures ( const FeatureSet &     f1,
                          const FeatureSet &     f2,
                          vector<FeatureMatch> & matches,
                          float &                totalScore
#ifdef                                           Q_WS_MAEMO_5
                          ,
                          Progressable *         thread
#endif
                          )
{
  // @@@ We are improving the speed of your matching code using the
  // Fast Library for Approximate Nearest Neighbors (FLANN)
  // We've done the work to populate the index.  Now you need to
  // look up how to query it and how to use the results.  Feel free
  // to use it for SSD match as well.
  cv::Mat features ( f2.size (), f2[0].data.size (), CV_32F );

  for ( size_t i = 0; i < f2.size (); ++i )
  {
    for ( size_t j = 0; j < f2[i].data.size (); ++j )
    {
      features.at<float>( ( int )i, ( int )j ) = f2[i].data[j];
    }
  }
  cv::flann::Index::Index flannIndex ( features, cv::flann::KDTreeIndexParams () );
  // End FLANN index population

  totalScore = 0;
#ifdef Q_WS_MAEMO_5
  int count = 0;
  int totalCount = f1.size ();
#endif
  for ( FeatureSet::const_iterator i1 = f1.begin (); i1 != f1.end (); ++i1 )
  {
#ifdef Q_WS_MAEMO_5
    if ( thread && count % ( totalCount / 100 ) == 0 )
    {
      thread->emitProgressUpdate ( 100 * count / totalCount );
    }
    ++count;
#endif
    // @@@ Find out how to query a cv::flann::Index::Index
  }
}

// RANSAC as described in lecture.  The result is a 3x3 homography matrix that
// computeHomography produced.  computeHomography expects a preallocated 3x3
// matrix of type CV_32F.  Just a tip, if you want to calculate an exact
// homography for 4 point correspondences, it's easy to pass computeHomography
// the full f1 and f2 vectors and just construct a temporary FeatureMatch
// vector with four elements from the full matches vector.
CvMat * ransacHomography ( const std::vector<Feature> &      f1,
                           const std::vector<Feature> &      f2,
                           const std::vector<FeatureMatch> & matches,
                           int                               numRounds,
                           int inlierThreshold
#ifdef                                                       Q_WS_MAEMO_5
                           ,
                           Progressable *                    thread
#endif
                           )
{
  // @@@ TODO Project 2b
  assert ( 0 ); // Remove when ready
}

// The resulting matrix is a 3x3 homography matrix.  You may find cvSolve
// (CV_SVD option) useful for solving the least squares problem.  The memory
// for this matrix is allocated by this caller.
CvMat * computeHomography ( const std::vector<Feature> &      f1,
                            const std::vector<Feature> &      f2,
                            const std::vector<FeatureMatch> & matches,
                            CvMat *                           h )
{
  // @@@ TODO Project 2b
  assert ( 0 ); // Remove when ready
}

//If you find a need for a byte array from your floating point images,
//here's some code you may use.
void convertToByteImage ( IplImage * floatImage, IplImage * byteImage )
{
  assert ( floatImage->depth == IPL_DEPTH_32F );
  assert ( byteImage->depth == IPL_DEPTH_32F );
  cvConvertScale ( floatImage, byteImage, 255, 0 );
}

// Transform point by homography.
void applyHomography ( float   x,
                       float   y,
                       float & xNew,
                       float & yNew,
                       float   h [9] )
{
  float d = h[6] * x + h[7] * y + h[8];

  xNew = ( h[0] * x + h[1] * y + h[2] ) / d;
  yNew = ( h[3] * x + h[4] * y + h[5] ) / d;
}

// Compute AUC given a ROC curve
float computeAUC ( vector<ROCPoint> & results )
{
  float auc = 0;
  float xdiff, ydiff;

  for ( size_t i = 1; i < results.size (); ++i )
  {
    xdiff = ( results[i].falseRate - results[i - 1].falseRate );
    ydiff = ( results[i].trueRate - results[i - 1].trueRate );
    auc = auc + xdiff * results[i - 1].trueRate + xdiff * ydiff / 2;
  }
  return auc;
}
