#include <cassert>
#include <cmath>
#include <iostream>
#include <opencv/cv.h>
#include <opencv/highgui.h>
#include "Features.h"

// Compute features of an image.
bool computeFeatures ( IplImage *   image,
                       FeatureSet & features,
                       int          featureType )
{
  assert ( image->depth == IPL_DEPTH_32F );
  assert ( image->nChannels == 3 );

  // Instead of calling dummyComputeFeatures, write your own
  // feature computation routines and call them here.
  switch ( featureType )
  {
  case 1:
    dummyComputeFeatures ( image, features );
    break;
  case 2:
    ComputeMOPSFeatures ( image, features );
    break;
  default:
    return false;
  }

  // This is just to make sure the IDs are assigned in order, because
  // the ID gets used to index into the feature array.
  for ( size_t i = 0; i < features.size (); ++i )
  {
    features [i].id = i + 1;
  }

  return true;
}

// Perform a query on the database.  This simply runs matchFeatures on
// each image in the database, and returns the feature set of the best
// matching image.
bool performQuery ( const FeatureSet &     f,
                    const ImageDatabase &  db,
                    size_t &                  bestIndex,
                    vector<FeatureMatch> & bestMatches,
                    float &                bestScore,
                    int                    matchType )
{
  // Here's a nice low number.
  bestScore = 1e100;
  bestIndex = 0;

  vector<FeatureMatch> tempMatches;
  float tempScore;

  for ( size_t i = 0; i < db.size (); ++i )
  {
    if ( !matchFeatures ( f, db [i].features, tempMatches, tempScore,
                          matchType ) )
    {
      return false;
    }

    if ( tempScore < bestScore )
    {
      bestIndex = i;
      bestScore = tempScore;
      bestMatches = tempMatches;
    }
  }
  return true;
}

// Match one feature set with another.
bool matchFeatures ( const FeatureSet &     f1,
                     const FeatureSet &     f2,
                     vector<FeatureMatch> & matches,
                     float &                totalScore,
                     int                    matchType )
{
  // We have given you the ssd matching function, you must write your own
  // feature matching function for the ratio test.

  std::cout << std::endl << "Matching features......." << std::endl;

  switch ( matchType )
  {
  case 1:
    ssdMatchFeatures ( f1, f2, matches, totalScore );
    return true;
  case 2:
    ratioMatchFeatures ( f1, f2, matches, totalScore );
    return true;
  default:
    return false;
  }
}

// Evaluate a match using a ground truth homography.  This computes the
// average SSD distance between the matched feature points and
// the actual transformed positions.
float evaluateMatch ( const FeatureSet &           f1,
                      const FeatureSet &           f2,
                      const vector<FeatureMatch> & matches,
                      float                        h [9] )
{
  float d = 0;
  int n = 0;

  float xNew;
  float yNew;

  size_t num_matches = matches.size ();

  for ( size_t i = 0; i < num_matches; ++i )
  {
    int id1 = matches [i].id1;
    int id2 = matches [i].id2;
    applyHomography ( f1 [id1 - 1].x, f1 [id1 - 1].y, xNew, yNew, h );
    d +=
      sqrt ( pow ( xNew - f2 [id2 - 1].x,
                   2 ) + pow ( yNew - f2 [id2 - 1].y, 2 ) );
    ++n;
  }

  return d / n;
}

void addRocData ( const FeatureSet &           f1,
                  const FeatureSet &           f2,
                  const vector<FeatureMatch> & matches,
                  float                        h [9],
                  vector<bool> &               isMatch,
                  float                        threshold,
                  float &                      maxD )
{
  float d = 0;

  float xNew;
  float yNew;

  size_t num_matches = matches.size ();

  for ( size_t i = 0; i < num_matches; ++i )
  {
    int id1 = matches [i].id1;
    int id2 = matches [i].id2;
    applyHomography ( f1 [id1 - 1].x, f1 [id1 - 1].y, xNew, yNew, h );

    // Ignore unmatched points.  There might be a better way to
    // handle this.
    d =
      sqrt ( pow ( xNew - f2 [id2 - 1].x,
                   2 ) + pow ( yNew - f2 [id2 - 1].y, 2 ) );
    if ( d <= threshold )
    {
      isMatch.push_back ( 1 );
    }
    else
    {
      isMatch.push_back ( 0 );
    }

    if ( matches [i].score > maxD )
    {
      maxD = matches [i].score;
    }
  }
}

vector<ROCPoint> computeRocCurve ( vector<FeatureMatch> & matches,
                                   vector<bool> &         isMatch,
                                   vector<float> &        thresholds )
{
  vector<ROCPoint> dataPoints;

  for ( size_t i = 0; i < thresholds.size (); ++i )
  {
    int tp = 0;
    int actualCorrect = 0;
    int fp = 0;
    int actualError = 0;
    int total = 0;

    size_t num_matches = matches.size ();
    for ( size_t j = 0; j < num_matches; ++j )
    {
      if ( isMatch [j] )
      {
        actualCorrect++;
        if ( matches [j].score < thresholds [i] )
        {
          tp++;
        }
      }
      else
      {
        actualError++;
        if ( matches [j].score < thresholds [i] )
        {
          fp++;
        }
      }
      total++;
    }

    ROCPoint newPoint;
    newPoint.trueRate = ( float( tp ) / actualCorrect );
    newPoint.falseRate = ( float( fp ) / actualError );

    dataPoints.push_back ( newPoint );
  }

  return dataPoints;
}


// Compute silly example features.  This doesn't do anything
// meaningful.
void dummyComputeFeatures ( IplImage * image, FeatureSet & features )
{
  assert ( image->depth == IPL_DEPTH_32F );
  assert ( image->nChannels == 3 );

  Feature f;

  for ( int y = 0; y < image->height; ++y )
  {
    for ( int x = 0; x < image->width; ++x )
    {
      float r = CV_IMAGE_ELEM ( image, float, y, 3 * x );
      float g = CV_IMAGE_ELEM ( image, float, y, 3 * x + 1 );
      float b = CV_IMAGE_ELEM ( image, float, y, 3 * x + 2 );

      if ( ( int )( 255 * ( r + g + b ) + 0.5 ) % 100 == 1 )
      {
        // If the pixel satisfies this meaningless criterion,
        // make it a feature.

        f.type = 1;
        f.id += 1;
        f.x = x;
        f.y = y;

        f.data.resize ( 1 );
        f.data [0] = r + g + b;

        features.push_back ( f );
      }
    }
  }
}

void ComputeMOPSFeatures ( IplImage * image, FeatureSet & features )
{
  assert ( image->depth == IPL_DEPTH_32F );
  assert ( image->nChannels == 3 );

  //Create grayscale image used for MOPS detection
  IplImage * grayImage =
    cvCreateImage ( cvGetSize ( image ), IPL_DEPTH_32F, 1 );
  cvCvtColor ( image, grayImage, CV_RGB2GRAY );

  //Create image to store MOPS values
  IplImage * harrisImage =
    cvCreateImage ( cvGetSize ( image ), IPL_DEPTH_32F, 1 );

  //Create image to store local maximum harris values as 255, other pixels 0
  IplImage * harrisMaxImage =
    cvCreateImage ( cvGetSize ( image ), IPL_DEPTH_8U, 1 );

  //TO DO--------------------------------------------------------------------
  //function puts harris values at each pixel position in harrisImage and
  // orientations at each pixel position in orientationImage
  IplImage * orientationImage =
    cvCreateImage ( cvGetSize ( image ), IPL_DEPTH_32F, 1 );
  //@@@PROJECT 2: Fill in computeMOPSValues
  computeMOPSValues ( grayImage, harrisImage, orientationImage );
  cvSaveImage ( "harris.png", harrisImage );

  //TO DO---------------------------------------------------------------------
  //Loop through harrisValues and find the best features in a local 3x3 maximum
  //compute the feature descriptor
  //@@@PROJECT 2: Fill in computeLocalMaxima
  computeLocalMaxima ( harrisImage, harrisMaxImage );
  cvSaveImage ( "harrisMax.png", harrisMaxImage );

  //TO DO--------------------------------------------------------------------
  //Loop through feature points in harrisMaxImage and create feature descriptor
  //for each point above a threshold
  //@@@PROJECT 2: Fill in this code here

  cvReleaseImage ( &orientationImage );
  cvReleaseImage ( &harrisImage );
  cvReleaseImage ( &grayImage );
  cvReleaseImage ( &harrisMaxImage );
}



//TO DO---------------------------------------------------------------------
//Loop through the image to compute the harris corner values as described in
//class
// srcImage:  grayscale of original image
// harrisImage:  populate the harris values per pixel in this image
// orientationImage: populate the orientation in this image
void computeMOPSValues ( IplImage * srcImage,
                         IplImage * harrisImage,
                         IplImage * orientationImage )
{
  assert ( srcImage->depth == IPL_DEPTH_32F );
  assert ( harrisImage->depth == IPL_DEPTH_32F );
  assert ( orientationImage->depth == IPL_DEPTH_32F );
  assert ( srcImage->nChannels == 1 );
  assert ( harrisImage->nChannels == 1 );
  assert ( orientationImage->nChannels == 1 );

  //@@@PROJECT 2: Your code here
}



//TO DO---------------------------------------------------------------------
//Loop through the image to compute the harris corner values as described in
//class
// srcImage:  image with MOPS values
// destImage: Assign 1 to local maximum in 3x3 window, 0 otherwise
void computeLocalMaxima ( IplImage * srcImage, IplImage * destImage )
{
  assert ( srcImage->depth == IPL_DEPTH_32F );
  assert ( destImage->depth == IPL_DEPTH_8U );
  assert ( srcImage->nChannels == 1 );
  assert ( destImage->nChannels == 1 );

  //@@@PROJECT 2: Your code here
}

float distanceSSD ( const std::vector<float> & d1, 
                    const std::vector<float> & d2 )
{
  //@@@PROJECT 2: Your code here
  return 0;
}

// Perform simple feature matching.  This just uses the SSD
// distance between two feature vectors, and matches a feature in the
// first image with the closest feature in the second image.  It can
// match multiple features in the first image to the same feature in
// the second image.
void ssdMatchFeatures ( const FeatureSet &     f1,
                        const FeatureSet &     f2,
                        vector<FeatureMatch> & matches,
                        float &                totalScore )
{
  int m = f1.size();
  int n = f2.size();
  matches.resize(m);
  totalScore = 0;
  double d;
  double dBest;
  int idBest;
  for (int i = 0; i < m; i++) 
  {
    dBest = 1e100;
    idBest = 0;
    for (int j=0; j<n; j++) 
    {
      d = distanceSSD(f1[i].data, f2[j].data);
      if (d < dBest) 
      {
        dBest = d;
        idBest = f2[j].id;
      }
    }
    matches[i].id1 = f1[i].id;
    matches[i].id2 = idBest;
    matches[i].score = dBest;
    totalScore += matches[i].score;
  }
}

//TODO: Write this function to perform ratio feature matching.
// This just uses the ratio of the SSD distance of the two best matches
// and matches a feature in the first image with the closest feature in the
// second image.
// It can match multiple features in the first image to the same feature in
// the second image.  (See class notes for more information)
void ratioMatchFeatures ( const FeatureSet &     f1,
                          const FeatureSet &     f2,
                          vector<FeatureMatch> & matches,
                          float &                totalScore )
{
  //@@@PROJECT 2: Your code here
}

//If you find a need for a byte array from your floating point images,
//here's some code you may use.
void convertToByteImage ( IplImage * floatImage, IplImage * byteImage )
{
  assert ( floatImage->depth == IPL_DEPTH_32F );
  assert ( byteImage->depth == IPL_DEPTH_8U );
  cvConvertScale ( floatImage, byteImage, 255, 0 );
}

// Transform point by homography.
void applyHomography ( float   x,
                       float   y,
                       float & xNew,
                       float & yNew,
                       float   h [9] )
{
  float d = h[6] * x + h[7] * y + h[8];

  xNew = ( h[0] * x + h[1] * y + h[2] ) / d;
  yNew = ( h[3] * x + h[4] * y + h[5] ) / d;
}

// Compute AUC given a ROC curve
float computeAUC ( vector<ROCPoint> & results )
{
  float auc = 0;
  float xdiff, ydiff;

  for ( size_t i = 1; i < results.size (); ++i )
  {
    xdiff = ( results[i].falseRate - results[i - 1].falseRate );
    ydiff = ( results[i].trueRate - results[i - 1].trueRate );
    auc = auc + xdiff * results[i - 1].trueRate + xdiff * ydiff / 2;
  }
  return auc;
}
