#include <opencv/cv.h>
#include <opencv/highgui.h>
#include "Filters.h"
#ifndef __WIN__
#include <unistd.h>
#else
#include "getopt.h"
#endif
#include <map>
#include <string>
#include <iostream>

void
print_usage ()
{
  std::cout << "Usage: cs4670 -f <filter> -i <input file> [-o output file] [-p parameter (if applicable)]" << std::endl;
  std::cout << "Filters:" << std::endl;
  std::cout << "  -gaussian" << std::endl;
  std::cout << "  -sharpen" << std::endl;
  std::cout << "  -sobel_x" << std::endl;
  std::cout << "  -sobel_y" << std::endl;
  std::cout << "  -median (requires parameter k for (2k+1)x(2k+1) filter)" << std::endl;
}

int
main (int argc, char ** argv)
{
  FilterData filter;
  bool filterSpecified = false;
  std::string inputFile;
  std::string outputFile;
  std::string filterName;
  char c;
  int p = 0;
  if (argc == 1)
  {
    print_usage ();
    exit (0);
  }
  while ((c = getopt (argc, argv, "hp:f:i:o:")) != -1)
  {
    switch (c)
    {
      case 'h':
        print_usage ();
        exit (0);
        break;
      case 'p':
        p = atoi (optarg);
        break;
      case 'f':
        filter = Filters::GetInstance().GetFilter (optarg);
        filterName = optarg;
        if (!filter.ImageFilter)
        {
          std::cout << "Filter " << optarg << " not found." << std::endl;
          exit (1);
        }
        else
        {
          filterSpecified = true;
        }
        break;
      case 'i':
        inputFile = optarg;
        break;
      case 'o':
        outputFile = optarg;
        break;
      default:
        std::cout << "Unknown parameter " << c << std::endl;
        break;
    }
  }
  if (!filterSpecified)
  {
    std::cout << "Filter unspecified." << std::endl; 
    exit (2);
  }
  if (inputFile == "")
  {
    std::cout << "Input file unspecified." << std::endl;
    exit (3);
  }

  CvCapture * capture = cvCreateFileCapture (inputFile.c_str());
  if (!capture)
  {
    std::cout << "Could not open file " << inputFile << "." << std::endl;
    exit (4);
  }

  IplImage * img = cvQueryFrame (capture);

  IplImage * floating = cvCreateImage (cvSize (img->width, img->height), IPL_DEPTH_32F, 3);
  cvConvertScale (img, floating, 1/255., 0);

  IplImage * filtered;

  switch (filter.ImageFormat)
  {
    case YUV24:
      filtered = cvCreateImage (cvSize (floating->width, floating->height), IPL_DEPTH_32F, 3);
      cvCvtColor (floating, filtered, CV_BGR2YCrCb);
      break;
    case BGR24:
      filtered = cvCloneImage (floating);
      break;
    case Gray:
      filtered = cvCreateImage (cvSize (floating->width, floating->height), IPL_DEPTH_32F, 1);
      cvCvtColor (floating, filtered, CV_BGR2GRAY);
      break;
  }

  cvNamedWindow ("original", CV_WINDOW_AUTOSIZE);
  cvShowImage ("original", img);

  filter.ImageFilter (filtered, p);
  
  if (filter.ImageFormat == YUV24)
  {
    cvCvtColor (filtered, filtered, CV_YCrCb2BGR);
  }

  cvNamedWindow (filterName.c_str(), CV_WINDOW_AUTOSIZE);
  cvShowImage (filterName.c_str(), filtered);
 
  if (outputFile != "")
  {
    cvConvertScale (filtered, filtered, 255, 0);
    cvSaveImage (outputFile.c_str(), filtered);
  }

  cvWaitKey (0);

  cvDestroyWindow ("original");
  cvDestroyWindow (filterName.c_str());
  
  cvReleaseCapture (&capture);
  cvReleaseImage (&filtered);
  cvReleaseImage (&floating);
}
