package resample;

import resample.image.Color;
import resample.image.Image;

/**
 * This class is the base class for discrete filters, which are used to apply filtering
 * operations to images (without changing their size).  Discrete filters are defined on
 * an integer grid, so the filter's radius is an integer, and it can be evaluated only for
 * integer-valued arguments.  Subclasses should override getSupportRadius and evaluate to
 * implement particular filters.
 * 
 * @author srm
 */
public abstract class DiscreteFilter extends FilterFunction {
	
	/**
	 * Returns the support radius of this function.
	 * 
	 * @return the support radius of the function
	 */
	public int getSupportRadius() { return 0; }
	
	/**
	 * Returns the value of the function at x. The value x should always be in
	 * the range [-r, r], where r is the result of getSupportRadius(). Because
	 * our filters are all separable, this method takes just one argument, and
	 * the value of a filter f at the point (i,j) is
	 *    f.evaluate(i) * f.evaluate(j).
	 * 
	 * @param x the requested location
	 * @return the value of the function
	 */
	public double evaluate(int x) { return 0; }
	
	/**
	 * This function applies the filter to an image.  It is implemented in terms
	 * of the methods getSupportRadius and evaluate, so it should work for any
	 * filter that overrides those two methods correctly.  The dimensions of the
	 * input and output images must be the same.
	 * 
	 * @param input the input image
	 * @param output the output image
	 */
	public void filterImage(Image input, Image output) {
		
		// Get the output image size
		int width = output.getWidth();
		int height = output.getHeight();
		
		// This just copies the input to the output
		Color pixelColor = new Color();
		for (int y = 0; y < height; y++)
			for (int x = 0; x < width; x++) {
				input.getPixelColor(pixelColor, x, y);
				output.setPixelColor(pixelColor, x, y);
			}
	}
	
	/**
	 * Does the same job as filterImage, but takes advantage of separability.
	 * 
	 * @param input the input image
	 * @param output the output image
	 */
	public void filterImageSeparable(Image input, Image output) {
		filterImage(input, output);
	}
	
}
