package resample;

import resample.continuous.NearestNeighborFilter;
import resample.image.Image;

/**
 * This class is the base class for continuous filters, which are used to
 * resample images. Continuous filters are defined for all real numbers, so the
 * filter's radius is a real number, and it can be evaluated for any real
 * number. Subclasses should override getSupportRadius and evaluate to implement
 * particular filters.
 * 
 * @author srm
 */
public abstract class ContinuousFilter extends FilterFunction {
	
	/**
	 * Returns the support radius of this function.
	 * 
	 * @return the support of the function
	 */
	public double getSupportRadius() { return 0; }
	
	/**
	 * Returns the value of the function at x. The value x should always be in
	 * the range [-r, r], where r is the result of getSupportRadius(). Because
	 * our filters are all separable, this method takes just one argument, and
	 * the value of a filter f at the point (x,y) is
	 *    f.evaluate(x) * f.evaluate(y).
	 * 
	 * @param x the requested location
	 * @return the value of the function
	 */
	public double evaluate(double x) { return 0; }
	
	/**
	 * Resamples input to output using the filter function. For most filters it
	 * can be implemented in terms of the methods evaluate and getSupportRadius,
	 * but for the box filter it is probably easier to override this method as a
	 * special case.  The dimensions of the output image determine the desired
	 * size of the image.
	 * 
	 * @param input the input image
	 * @param output the output image
	 */
	public void resampleImage(Image input, Image output) {
		NearestNeighborFilter.quickResample(input, output);
	}
	
	/**
	 * Does the same job as resampleImage, but takes advantage of separability.
	 * 
	 * @param input the input image
	 * @param output the output image 
	 */
	public void resampleImageSeparable(Image input, Image output) {
		resampleImage(input, output);
	}

}
