/*
 * Created on Oct 25, 2005
 * Copyright 2005 Program of Computer Grpahics, Cornell University
 */
package modeler.shape;

import java.awt.Graphics2D;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.text.DecimalFormat;
import java.util.ArrayList;

import javax.vecmath.Point2f;
import javax.vecmath.Vector2f;


/**
 * An implementation of a Bezier spline.
 * @author arbree
 * Oct 25, 2005
 * BezierSpline.java
 * Copyright 2005 Program of Computer Graphics, Cornell University
 */
public class BezierSpline {

  /** Different criteria for refinement */
  private static final int CRITERION_FLAT = 0;
  private static final int CRITERION_DISTANCE = 1;
  
  //Tolerance for flatness when selecting a point on the spline
  protected static final float PICK_TOLERANCE = 0.02f;
  
  /** The neighbor splines in the curve */
  protected BezierSpline left = null;
  protected BezierSpline right = null;
  
  /** The control points of this spline */
  protected Point2f p0;
  protected Point2f p1;
  protected Point2f p2;
  protected Point2f p3;
  
  /** Flags for the left or right end of the spline being a corner */
  protected boolean leftCorner = true;
  protected boolean rightCorner= true;
  
  /** Storage space for tangents */
  protected transient final Vector2f t0 = new Vector2f();
  protected transient final Vector2f t1 = new Vector2f();
  
  /** Minimum and maximum t values for this spline */
  protected float minT = 0;
  protected float maxT = 1;
  
  /**
   * Required for IO
   */
  public BezierSpline() { }
  
  /**
   * Divides this segment into two children segments.  Returns
   * the lefthand segment.  Should make the left segment point to the
   * right and the right to the left, create the new intermediate point
   * at t (make sure to share the point object between the two splines, 
   * and set the appropriate min and max t values for the new splines.
   * 
   * @param t
   * @return
   */
  private BezierSpline divide(float t) {

    return null;
    
  }
  
  /**
   * Method that backs getPointsBySpacing() and getPointsByFlatness().  Should return a list of 
   * points, tangents and tvalues (optionally the last two if outTangents or outTValues is null) such that
   * the list of points satisfies criterion and tolerance.
   * 
   * @param outPoints
   * @param outTangents
   * @param tolerance
   * @param criterion
   */
  protected void getPoints(ArrayList outPoints, ArrayList outTangents, ArrayList outTValues, float tolerance, int criterion) {

    //TODO: Implement this method
    
  }
  
  ////////////////////////////////////////////////////////////////////////////////////////////////////
  // Other spline methods
  ////////////////////////////////////////////////////////////////////////////////////////////////////
  
  /**
   * Construct a spline with certain control points.  THE CONTROL POINTS ARE
   * NOT COPIED SO CHNAGES TO THE POINTS WILL CHANGE THE SPLINE!
   * @param inP0
   * @param inP1
   * @param inP2
   * @param inP3
   */
  protected BezierSpline(Point2f inP0, Point2f inP1, Point2f inP2, Point2f inP3) {
    
    p0 = inP0;
    p1 = inP1;
    p2 = inP2;
    p3 = inP3;
    
  }
  
  /**
   * Creates a spline with the given control points
   * @param inP0
   * @param inP1
   * @param inP2
   * @param inP3
   * @return
   */
  public static final BezierSpline createSpline(Point2f inP0, Point2f inP1, Point2f inP2, Point2f inP3) {
    
    BezierSpline newSpline = new BezierSpline(inP0, inP1, inP2, inP3);
    HeadSpline head = newSpline.new HeadSpline();
    head.right = newSpline;
    newSpline.left = head;
    return head;
    
  }
  
  /**
   * Split this spline at t
   * @param t
   */
  public void refine(float t) {
    
    BezierSpline newL = divide(t);
    BezierSpline newR = newL.right;
    
    //Add the spline pair to the list
    left.right = newL;
    newL.left = left;
    if(!isRightEnd())
      right.left = newR;
    newR.right = right;
    
    //Set all the corner flags
    if(isLeftEnd())
      newL.leftCorner = true;
    else newL.leftCorner = left.rightCorner;
    if(isRightEnd())
      newR.rightCorner = true;
    else newR.rightCorner = right.leftCorner;
    newL.rightCorner = newR.leftCorner = false;
    
  }
  
  /**
   * Returns a list of points such that the distance between any two points is less
   * than tolerance
   * @param outPoints
   * @param outTangents
   * @param tolerance
   */
  public void getPointsByDistance(ArrayList outPoints, ArrayList outTangents, ArrayList outTValues, float tolerance) {
    getPoints(outPoints, outTangents, outTValues, tolerance, CRITERION_DISTANCE);
  }
  
  /**
   * Get a set of points such that the spline is never greater than tolerance away from
   * the piecewise linear function defined by those points.
   * @param outPoints
   * @param outTangents
   * @param tolerance
   */
  public void getPointsByFlatness(ArrayList outPoints, ArrayList outTangents, ArrayList outTValues, float tolerance) {
    getPoints(outPoints, outTangents, outTValues, tolerance, CRITERION_FLAT);
  }
  
  /**
   * Returns the handle points of this spline
   * @param outPoints
   * @param outTangents
   * @param tolerance
   */
  public void getHandles(ArrayList outPoints, ArrayList outCorners) {

    //Add our points
    outPoints.add(p0);
    outPoints.add(p1);
    outPoints.add(p2);

    //Add our corners
    if(outCorners != null) {
      outCorners.add(new Boolean(leftCorner));
      outCorners.add(new Boolean(true));
      outCorners.add(new Boolean(true));
    }
    
    //Add our right neighbors points or the last point if we are the end
    if(!isRightEnd())
      right.getHandles(outPoints, outCorners);
    else {
      outPoints.add(p3);
      if(outCorners != null)
        outCorners.add(new Boolean(rightCorner));
    }
  }
  
  /**
   * Returns the index of inPoint in this objects control points.  Returns -1
   * if this point is not a control point for this spline
   * @param inPoint
   * @return
   */
  private int getControlPointIndexOf(Point2f inPoint) {
    
    if(inPoint == p0)
      return 0;
    if(inPoint == p1)
      return 1;
    if(inPoint == p2)
      return 2;
    if(inPoint == p3)
      return 3;
    return -1;
    
  }
  
  /**
   * Returns the left most spline containing this control point.  Note:
   * referential equality is used.
   * @param inPoint
   */
  public BezierSpline getSplineForPoint(Point2f inPoint) {
    
    if(getControlPointIndexOf(inPoint) >= 0)
      return this;
    if(isRightEnd())
      return null;
    return right.getSplineForPoint(inPoint);
    
  }

  /**
   * Updates a control point for this spline to a new value
   * @param oldPoint
   * @param newPoint
   */
  public void updateControlPoint(Point2f oldPoint, Point2f newPoint) {
    
    int idx = getControlPointIndexOf(oldPoint);

    // Move selected point
    Vector2f d = new Vector2f();
    Vector2f v = new Vector2f();
    if (idx == 0) {
      d.sub(p1, p0);
      if(!isLeftEnd())
        v.sub(left.p2, p0);
      p0.set(newPoint);
      if(!leftCorner) {
        p1.add(newPoint, d);
        if(!isLeftEnd()) { 
          left.p2.add(newPoint, v);
        }
      }
    }
    else if (idx == 1) {
      p1.set(newPoint);
      if (!isLeftEnd() && !leftCorner) {
        d.sub(newPoint, p0);
        v.sub(left.p2, p0);
        d.scale(v.length() / d.length());
        left.p2.sub(p0, d);
      }
    }
    else if (idx == 2) {
      p2.set(newPoint);
      if (!isRightEnd() && !rightCorner) {
        d.sub(newPoint, p3);
        v.sub(right.p1, p3);
        d.scale(v.length() / d.length());
        right.p1.sub(p3, d);
      }
    }
    else if (idx == 3) {
      d.sub(p2, p3);
      if(!isRightEnd())
        v.sub(right.p1, p3);
      p3.set(newPoint);
      if (!rightCorner) {
        p2.add(newPoint, d);
        if (!isRightEnd()) {
          right.p1.add(newPoint, v);
        }
      }
    }
  }
  
  /**
   * Adds a control point on the spline at the point of projection of inPoint
   * to the spline.  If inPoint is not sufficiently close to the spline,
   * no point is added.
   * @param inPoint
   * @return
   */
  public boolean addControlPointNear(Point2f inPoint) {
    
    float ans = getTForProjectedPoint(inPoint, PICK_TOLERANCE);
    if(ans > 0) {
      
      //Derive a local t from the global one returned
      float localT = (ans - minT)/(maxT - minT);
      refine(localT);
      return true;
    }
    else if(!isRightEnd())
      return right.addControlPointNear(inPoint);
    return false;
    
  }
  
  /**
   * Toggle the corner control for the supplied point
   * @param inPoint
   * @return
   */
  public void toggleCorner(Point2f inPoint) {
    
    int idx = getControlPointIndexOf(inPoint);
    if(idx == 0 && !isLeftEnd())
      left.rightCorner = leftCorner = !leftCorner;
    else if(idx == 3 && !isRightEnd())
      right.leftCorner = rightCorner = !rightCorner;
    
  }
  
  /**
   * Returns a t value for the entire spline from a local t
   * @param t
   * @return
   */
  private float getSplineTFromLocalT(float t) {
    
    return (maxT - minT)*t + minT;
    
  }
  
  /**
   * Recursive helper for above
   * @param inPoint
   * @return
   */
  private float getTForProjectedPoint(Point2f inPoint, float tolerance) {
    
    //Determine if the picked point is accurate
    Vector2f u = new Vector2f(p3.y - p0.y, p0.x - p3.x);
    u.scale(1.0f / u.length());
    Vector2f d1 = new Vector2f(p1);
    d1.sub(p0);
    Vector2f d2 = new Vector2f(p2);
    d2.sub(p0);
    boolean test = Math.abs(d1.dot(u)) < tolerance && Math.abs(d2.dot(u)) < tolerance;
    
    //If we are locally flat, find the projection of inPoint to a linear approximation of ourselves
    if(test) {
      
      Vector2f splineVec = new Vector2f();
      Vector2f pointVec = new Vector2f();
      splineVec.sub(p3, p0);
      pointVec.sub(inPoint, p0);
      float dot = splineVec.dot(pointVec);
      if(dot > 0) {
        float t = splineVec.lengthSquared();
        dot /= t;
        splineVec.scale(dot);                   //Now the projection of pointVec onto splineVec
        t = (float) Math.sqrt(splineVec.lengthSquared() / t);
        if(t < 1.0) {                           //Make sure we don't project past the end
          pointVec.sub(splineVec);
          if(pointVec.lengthSquared() < tolerance*tolerance) {   // Check that the point is close enough to the spline
            return getSplineTFromLocalT(t);
          }
        }
      }
      return -1;
      
    }
    
    //if we are not flat
    BezierSpline left = divide(0.5f);
    float ans = left.getTForProjectedPoint(inPoint, tolerance);
    if(ans > 0)
      return ans;
    ans = left.right.getTForProjectedPoint(inPoint, tolerance);
    if(ans > 0)
      return ans;
    return -1;
    
  }
  
  /**
   * Removes the control point from the spline.  Only interpolated
   * control points can be removed.
   * @param inPoint
   */
  public boolean removeControlPoint(Point2f inPoint) {
    
    //If the spline is only one unit long, no points can be removed
    if(isLeftEnd() && isRightEnd())
      return false;
    
    int idx = getControlPointIndexOf(inPoint);
    if(idx == 0) {
      if(!isLeftEnd()) {
        left.p2 = this.p2;
        left.p3 = this.p3;
        left.rightCorner = this.rightCorner;
        left.maxT = this.maxT;
      } else {
        right.minT = 0;
        right.leftCorner = true;
      }
      left.right = this.right;
      right.left = this.left;
      return true;
    } else if(idx == 3) {
      if(!isRightEnd()) {
        right.p0 = this.p0;
        right.p1 = this.p1;
        right.left = this.left;
        right.leftCorner = this.leftCorner;
        right.minT = this.minT;
      } else {
        left.maxT = 1.0f;
        left.rightCorner = true;
      }
      left.right = this.right;
      return true;
    }
    return false;
  }
  
  /**
   * Returns true if this spline is the leftmost spline in the curve
   * @return
   */
  public boolean isLeftEnd() {
    
    return left instanceof HeadSpline;
    
  }
  
  /**
   * Returns true if this spline is the rightmost spline in the curve
   * @return
   */
  public boolean isRightEnd() {
    
    return right == null;
    
  }
  
  /**
   * Prints this spline
   */
  public void print() {
   
    System.out.println("Spline: "+this);
    System.out.println("  Lf: "+left);
    System.out.println("  P0: "+p0);
    System.out.println("  P1: "+p1);
    System.out.println("  P2: "+p2);
    System.out.println("  P3: "+p3);
    System.out.println("  Rt: "+right);
    if(!isRightEnd())
      right.print();
    
  }
  
  /**
   * Print information about this spline at (x,y) in the g2d
   * @param g2d
   * @param i 
   */
  public void print(Graphics2D g2d, int x, int y, int i) {
    
    DecimalFormat df = new DecimalFormat("#0.00");
    String p0S = (p0 == null ? "" : "("+df.format(p0.x)+","+df.format(p0.y)+")");
    String p1S = (p1 == null ? "" : "("+df.format(p1.x)+","+df.format(p1.y)+")");
    String p2S = (p2 == null ? "" : "("+df.format(p2.x)+","+df.format(p2.y)+")");
    String p3S = (p3 == null ? "" : "("+df.format(p3.x)+","+df.format(p3.y)+")");
    
    g2d.drawString("t0: "+df.format(minT), 5, y);
    g2d.drawString("t1: "+df.format(maxT), 100, y);
    y+= i;
    g2d.drawString("P0: "+p0S, 5, y);
    g2d.drawString("P1: "+p1S, 100, y);
    y+= i;
    g2d.drawString("P2: "+p2S, 5, y);
    g2d.drawString("P3: "+p3S, 100, y);
    
  }
  
  /**
   * @param in
   * @throws IOException 
   */
  public void readData(ObjectInputStream in) throws IOException {

    p0 = new Point2f();
    p1 = new Point2f();
    p2 = new Point2f();
    if(!(left instanceof HeadSpline))
      left.p3 = p0;
    p0.x = in.readFloat();
    p0.y = in.readFloat();
    p1.x = in.readFloat();
    p1.y = in.readFloat();
    p2.x = in.readFloat();
    p2.y = in.readFloat();
    leftCorner = in.readBoolean();
    rightCorner = in.readBoolean();
    minT = in.readFloat();
    maxT = in.readFloat();
    boolean isRightEnd = in.readBoolean();
    if(isRightEnd) {
      p3 = new Point2f();
      p3.x = in.readFloat();
      p3.y = in.readFloat();
    }
    else {
      right = new BezierSpline();
      right.left = this;
      right.readData(in);
    }
    
  }

  /**
   * @param out
   * @throws IOException 
   */
  public void writeData(ObjectOutputStream out) throws IOException {

    out.writeFloat(p0.x);
    out.writeFloat(p0.y);
    out.writeFloat(p1.x);
    out.writeFloat(p1.y);
    out.writeFloat(p2.x);
    out.writeFloat(p2.y);
    out.writeBoolean(leftCorner);
    out.writeBoolean(rightCorner);
    out.writeFloat(minT);
    out.writeFloat(maxT);
    out.writeBoolean(isRightEnd());
    if(isRightEnd()) {
      out.writeFloat(p3.x);
      out.writeFloat(p3.y);
    }
    else right.writeData(out);
    
  }
  
  /**
   * Private dummy spline that forms the head of the node list. 
   * @author arbree
   * Oct 26, 2005
   * BezierSpline.java
   * Copyright 2005 Program of Computer Graphics, Cornell University
   */
  private class HeadSpline extends BezierSpline {

    /** Create a head spline */
    protected HeadSpline() {
      super(null, null, null, null);
      minT = -1;
      maxT = -1;
    }
    
    /**
     * @see modeler.shape.BezierSpline#getHandles(java.util.ArrayList)
     */
    public void getHandles(ArrayList outPoints, ArrayList outCorners) {
      right.getHandles(outPoints, outCorners);
    }

    /**
     * @see modeler.shape.BezierSpline#getPoints(java.util.ArrayList, java.util.ArrayList, float, int)
     */
    protected void getPoints(ArrayList outPoints, ArrayList outTangents, ArrayList tValues, float tolerance, int criterion) {
      right.getPoints(outPoints, outTangents, tValues, tolerance, criterion);
    }
    
    /**
     * @see modeler.shape.BezierSpline#addControlPointNear(javax.vecmath.Point2f)
     */
    public boolean addControlPointNear(Point2f inPoint) {

      return right.addControlPointNear(inPoint);
      
    }
  
    /**
     * Print information about this spline at (x,y) in the g2d
     * @param g2d
     * @param i 
     */
    public void print(Graphics2D g2d, int x, int y, int i) {
      
      g2d.drawString("No spline selected.", 5, y);
      
    }

    public void readData(ObjectInputStream in) throws IOException {

      right.readData(in);
      
    }

    public void writeData(ObjectOutputStream out) throws IOException {

      right.writeData(out);
      
    }
  }
}
