package modeler;

import java.io.BufferedOutputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;

import javax.swing.Icon;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.vecmath.Matrix4f;

import modeler.shape.Shape;
import net.java.games.jogl.GL;
import net.java.games.jogl.GLU;

public abstract class Treeable extends DefaultMutableTreeNode {

  /** The name of this node */
  protected String name;
    
  /** The parent of this node * */
  protected Treeable parent;

  /** The list of children for this node */
  protected ArrayList children = new ArrayList();

  /**
   * Empty constructor required for I/O
   */
  protected Treeable() {
    
  }
  
  /**
   * Sets the name of this node
   * 
   * @param name
   */
  public void setName(String name) {
    
    this.name = name;
    
  }

  /**
   * Render this node.  The current model view matrix is supplied.  Default just called 
   * render on the children of this node
   * @param gl
   * @param currMat
   */
  public int render(GL gl, GLU glu, Matrix4f currMat) {
    
    int outCount = 0;
    for (int ctr = 0; ctr < getChildCount(); ctr++)
      outCount += ((Treeable) getChildAt(ctr)).render(gl, glu, new Matrix4f(currMat));
    return outCount;
    
  }
  
  /**
   * Export this file to data that can be read by the ray tracer
   * @param fileName
   * @throws FileNotFoundException 
   * @throws FileNotFoundException 
   */
  public static void export(String fileName, Treeable root) throws FileNotFoundException {
    
    //First export the camera, lights and materials
    PrintWriter pw = new PrintWriter(new BufferedOutputStream(new FileOutputStream(fileName)));
    pw.println("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>");
    pw.println("<scene>");
    HashMap matMap = new HashMap();
    root.exportMainData(matMap, pw);
    pw.println("  <surface type=\"ray1.surface.Model\">");
    pw.println("     <data>"+fileName+".mesh</data>");
    pw.println("  </surface>");
    pw.println("  <image>512 512</image>");
    pw.println("</scene>");
    pw.close();
    
    //Export the meshes
    PrintWriter mw = new PrintWriter(new BufferedOutputStream(new FileOutputStream(fileName+".mesh")));
    Matrix4f mat = new Matrix4f();
    mat.setIdentity();
    root.exportMeshData(matMap, mw, mat);
    mw.close();
    
    
  }
  
  /**
   * Exports this nodes data for the main xml output file
   */
  protected void exportMainData(HashMap materialsMap, PrintWriter pw) {
    
    for (Iterator iter = children.iterator(); iter.hasNext();) {
      Treeable currNode = (Treeable) iter.next();
      currNode.exportMainData(materialsMap, pw);
    }
  }
  
  /**
   * Exports this nodes mesh data
   */
  protected void exportMeshData(HashMap materialsMap, PrintWriter pw, Matrix4f mat) {
    
    for (Iterator iter = children.iterator(); iter.hasNext();) {
      Treeable currNode = (Treeable) iter.next();
      currNode.exportMeshData(materialsMap, pw, new Matrix4f(mat));
    }
  }
  
  /**
   * Writes the treeable out to the data stream
   * @param out
   * @throws IOException 
   */
  public void save(ObjectOutputStream out) throws IOException {
    
    //Write the class of this treeable
    out.writeObject(this.getClass().getName());

    //Write this treeables name
    out.writeObject(name);
    
    //Write that nodes data
    saveData(out);
    
    //Write the children
    out.writeInt(children.size());
    for (Iterator iter = children.iterator(); iter.hasNext();) {
      Treeable currChild = (Treeable) iter.next();
      currChild.save(out);
    }
  }
  
  /**
   * Reads this treeable from the data stream
   * @param in
   * @throws ClassNotFoundException 
   * @throws IOException 
   */
  public static Treeable read(ObjectInputStream in) throws IOException, ClassNotFoundException {
    
    //Constructs an instance of the class to be read
    String className = (String) in.readObject();
    Class currClass = Class.forName(className);
    if(!Treeable.class.isAssignableFrom(currClass))
      throw new Error("modeler.Treeable.read(): Read a non-treeable class: "+currClass);
    Treeable currNode;
    try {
      currNode = (Treeable) currClass.newInstance();
    }
    catch (Exception e) {
      e.printStackTrace();
      throw new Error("modeler.Treeable.read(): Instantiating new treeable node.");
    }
    
    //Reads the data
    currNode.name = (String) in.readObject();
    currNode.readData(in);
    
    //Adds its children
    int numChildren = in.readInt();
    for(int i = 0; i < numChildren; i++)
      currNode.add(read(in));
    
    return currNode;
    
  }
  
  /**
   * Writes this object to the stream.
   * @param out
   * @throws IOException 
   */
  protected abstract void saveData(ObjectOutputStream out) throws IOException;
  
  /**
   * Writes this objec to the stream
   * @param in
   * @throws IOException 
   */
  protected abstract void readData(ObjectInputStream in) throws IOException;
  
  /**
   * Rebuilds all meshes
   */
  public void rebuild() {
    if(this instanceof Shape) {
      ((Shape) this).mesh = null;
      return;
    }
    for (Iterator iter = children.iterator(); iter.hasNext();) {
      Treeable currChild = (Treeable) iter.next();
      currChild.rebuild();
    }
  }
  
  /**
   * @see java.lang.Object#toString()
   */
  public String toString() {
    
    return name;
    
  }

  /**
   * Gets the icon representing this node
   * 
   * @return
   */
  public abstract Icon getIcon();

  /**
   * Gets the attribute panel for this node
   * 
   * @param mainFrame
   * @return
   */
  public abstract ItemAttributePanel getAttributePanel(MainFrame mainFrame);

  // //////////////////////////////////////////////////////////////////////////////////////////////////
  // Methods for tree node interface
  // //////////////////////////////////////////////////////////////////////////////////////////////////

  /**
   * @see javax.swing.tree.TreeNode#children()
   */
  public Enumeration children() {

    return new TreeEnumerator(children.iterator());

  }

  /**
   * @see javax.swing.tree.TreeNode#getChildAt(int)
   */
  public TreeNode getChildAt(int childIndex) {

    return (TreeNode) children.get(childIndex);

  }

  /**
   * @see javax.swing.tree.TreeNode#getChildCount()
   */
  public int getChildCount() {

    return children.size();

  }

  /**
   * @see javax.swing.tree.TreeNode#getIndex(javax.swing.tree.TreeNode)
   */
  public int getIndex(TreeNode node) {

    return children.indexOf(node);

  }

  /**
   * @see javax.swing.tree.TreeNode#getParent()
   */
  public TreeNode getParent() {

    return parent;

  }

  /**
   * Provides a enumerators for a Treeable object
   * 
   * @author arbree Oct 20, 2005 Treeable.java Copyright 2005 Program of
   *         Computer Graphics, Cornell University
   */
  private class TreeEnumerator implements Enumeration {

    /** The iterator backing this enumerator */
    protected Iterator iter;

    /**
     * Constructor takes an iterator
     * 
     * @param inIter
     */
    protected TreeEnumerator(Iterator inIter) {

      iter = inIter;

    }

    /**
     * @see java.util.Enumeration#hasMoreElements()
     */
    public boolean hasMoreElements() {

      return iter.hasNext();

    }

    /**
     * @see java.util.Enumeration#nextElement()
     */
    public Object nextElement() {

      return iter.next();

    }
  }

  /**
   * @see javax.swing.tree.MutableTreeNode#insert(javax.swing.tree.MutableTreeNode,
   *      int)
   */
  public void insert(MutableTreeNode child, int index) {

    child.setParent(this);
    if (index >= children.size())
      children.add(child);
    else
      children.add(children.set(index, child));

  }

  /**
   * @see javax.swing.tree.MutableTreeNode#remove(int)
   */
  public void remove(int index) {

    children.remove(index);

  }

  /**
   * @see javax.swing.tree.MutableTreeNode#remove(javax.swing.tree.MutableTreeNode)
   */
  public void remove(MutableTreeNode node) {

    children.remove(children.indexOf(node));

  }

  /**
   * @see javax.swing.tree.MutableTreeNode#removeFromParent()
   */
  public void removeFromParent() {

    parent.remove(this);
    parent = null;

  }

  /**
   * @see javax.swing.tree.MutableTreeNode#setParent(javax.swing.tree.MutableTreeNode)
   */
  public void setParent(MutableTreeNode newParent) {

    parent = (Treeable) newParent;

  }

  /**
   * @see javax.swing.tree.MutableTreeNode#setUserObject(java.lang.Object)
   */
  public void setUserObject(Object object) {

    name = (String) object;

  }
}
