package modeler;

import java.awt.GridLayout;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.PrintWriter;
import java.text.DecimalFormat;
import java.util.HashMap;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.vecmath.AxisAngle4f;
import javax.vecmath.Matrix3f;
import javax.vecmath.Matrix4f;
import javax.vecmath.Vector3f;

import net.java.games.jogl.GL;
import net.java.games.jogl.GLU;

/**
 * @author andru Transformation provides an interface to a 3D transformation
 *         matrix. It allows independent access to the translational,
 *         rotational, and scaling properties of the matrix. By convention,
 *         rotations are performed around the axes in X,Y,Z order.
 */

public class Transformation extends Treeable {
  
  private static final long serialVersionUID = 3256437006337718072L;
  
  // Individual Translation, Rotation, and Scaling components of this transformation
  protected final Vector3f R = new Vector3f(); // Stored in degrees
  protected final Vector3f S = new Vector3f();
  protected final Vector3f T = new Vector3f();

  //Constants defining the coordinate axies
  public static final Vector3f xAxis = new Vector3f(1, 0, 0);
  public static final Vector3f yAxis = new Vector3f(0, 1, 0);
  public static final Vector3f zAxis = new Vector3f(0, 0, 1);

  protected transient Icon icon;
  private transient AttributePanel attribs;
  
  /**
   * Required for IO
   */
  public Transformation() {}
  
  /**
   * Constructor
   * @param name
   */
  public Transformation(String name) {

    this.name = name;
    S.set(1, 1, 1);
  }

  /**
   * Copy constructor
   * @param copy
   */
  public Transformation(Transformation copy) {

    this.R.set(copy.R);
    this.S.set(copy.S);
    this.T.set(copy.T);
  }
  
  /**
   * @see modeler.Treeable#getIcon()
   */
  public Icon getIcon() {

    if(icon == null)
      icon = new ImageIcon("./icons/transform.gif");
    return icon;
  }

  /**
   * Converts the given point v from local space to world space 
   */
  public void toWorld(Vector3f v, Vector3f outv) {
	  Vector3f result = new Vector3f();
	  transform(v, result);
	  if (parent == null) {
		  outv.set(result);
	  } else {
		  ((Transformation)parent).toWorld(result, outv);
	  }
  }
  
  /**
   * Transforms the given point by this transformation
   */
  Matrix3f tempMatrix = new Matrix3f();
  public void transform(Vector3f v, Vector3f outv) {
	  outv.set(v);
	  // Scale
	  outv.set(outv.x * S.x, outv.y * S.y, outv.z * S.z);
	  
	  // Rotate
	  tempMatrix.rotX((float)Math.toRadians(R.x));
	  tempMatrix.transform(outv);
	  tempMatrix.rotY((float)Math.toRadians(R.y));
	  tempMatrix.transform(outv);
	  tempMatrix.rotZ((float)Math.toRadians(R.z));
	  tempMatrix.transform(outv);
	  
	  // Translate
	  outv.add(T);
  }
  
  /**
   * Sets the translation vector
   * @param translation
   */
  public void setTranslate(Vector3f translation) {

    T.set(translation);
  }

  /**
   * Sets the scale vector
   * @param scaling
   */
  public void setScale(Vector3f scaling) {

    S.set(scaling);
  }

  /**
   * Sets the rotation angle in degrees
   * @param rotation
   */
  public void setRotate(Vector3f rotation) {

    R.set(rotation);
  }

  /**
   * Gets the translation vector
   * @return
   */
  public Vector3f getTranslate() {

    return new Vector3f(T);
  }

  /**
   * Gets the scale vector
   * @return
   */
  public Vector3f getScale() {

    return new Vector3f(S);
  }

  /**
   * Gets the rotation vector in degrees
   * @return
   */
  public Vector3f getRotate() {

    return new Vector3f(R);
  }

  /**
   * Refreshs the state of the attribute panel
   *
   */
  public void refreshAttribs() {

    if (attribs != null)
      attribs.refresh();
  }

  /**
   * @see modeler.Treeable#getAttributePanel(modeler.MainFrame)
   */
  public ItemAttributePanel getAttributePanel(MainFrame mainFrame) {

    if (attribs == null) {
      attribs = new AttributePanel(mainFrame, this);
    }
    return attribs;
  }

  /**
   * Concatenates this transform with the currentMatrix
   * @param currMat
   */
  private void addTransform(Matrix4f currMat) {
    
    Matrix4f workMat = new Matrix4f();
    
    workMat.setIdentity();
    workMat.setTranslation(T);
    currMat.mul(workMat);
    
    workMat.setIdentity();
    workMat.set(new AxisAngle4f(zAxis, (float) Math.toRadians(R.z)));
    currMat.mul(workMat);
    
    workMat.setIdentity();
    workMat.set(new AxisAngle4f(yAxis, (float) Math.toRadians(R.y)));
    currMat.mul(workMat);
    
    workMat.setIdentity();
    workMat.set(new AxisAngle4f(xAxis, (float) Math.toRadians(R.x)));
    currMat.mul(workMat);

    workMat.setIdentity();
    workMat.setElement(0, 0, S.x);
    workMat.setElement(1, 1, S.y);
    workMat.setElement(2, 2, S.z);
    currMat.mul(workMat);
  }
  
  /**
   * @see modeler.Treeable#render(net.java.games.jogl.GL, javax.vecmath.Matrix4f)
   */
  public int render(GL gl, GLU glu, Matrix4f currMat) {
    
    addTransform(currMat);
    return super.render(gl, glu, currMat);
    
  }
  
  /**
   * @see modeler.Treeable#readData(java.io.ObjectInputStream)
   */
  protected void readData(ObjectInputStream in) throws IOException {

    R.x = in.readFloat();
    R.y = in.readFloat();
    R.z = in.readFloat();
    S.x = in.readFloat();
    S.y = in.readFloat();
    S.z = in.readFloat();
    T.x = in.readFloat();
    T.y = in.readFloat();
    T.z = in.readFloat();
    
  }

  /**
   * @see modeler.Treeable#saveData(java.io.ObjectOutputStream)
   */
  protected void saveData(ObjectOutputStream out) throws IOException {

    out.writeFloat(R.x);
    out.writeFloat(R.y);
    out.writeFloat(R.z);
    out.writeFloat(S.x);
    out.writeFloat(S.y);
    out.writeFloat(S.z);
    out.writeFloat(T.x);
    out.writeFloat(T.y);
    out.writeFloat(T.z);
    
  }

  /**
   * @see modeler.Treeable#exportMeshData(java.util.HashMap, java.io.PrintWriter, javax.vecmath.Matrix4f)
   */
  protected void exportMeshData(HashMap materialsMap, PrintWriter pw, Matrix4f mat) {

    addTransform(mat);
    super.exportMeshData(materialsMap, pw, mat);
    
  }

  /**
   * @see javax.swing.tree.TreeNode#getAllowsChildren()
   */
  public boolean getAllowsChildren() {

    return true;

  }

  /**
   * @see javax.swing.tree.TreeNode#isLeaf()
   */
  public boolean isLeaf() {

    return false;

  }
  
  /**
   * The attribute panel for this node
   * @author arbree
   * Oct 21, 2005
   * Transformation.java
   * Copyright 2005 Program of Computer Graphics, Cornell University
   */
  private class AttributePanel extends ItemAttributePanel implements DocumentListener {

    private static final long serialVersionUID = 3256437006337718072L;
    
    //Text fields for each of the components
    protected JFormattedTextField Tx;
    protected JFormattedTextField Ty;
    protected JFormattedTextField Tz;
    protected JFormattedTextField Rx;
    protected JFormattedTextField Ry;
    protected JFormattedTextField Rz;
    protected JFormattedTextField Sx;
    protected JFormattedTextField Sy;
    protected JFormattedTextField Sz;
    boolean refreshing = false;

    /**
     * Inherited constructor
     * @param mf
     */
    protected AttributePanel(MainFrame mf, Transformation trans) {

      super(mf, trans);
      
    }
    
    /**
     * Builds the attribute panel
     * @param mf
     */
    public void initialize() {

      Tx = new JFormattedTextField(new DecimalFormat("#0.0#"));
      Ty = new JFormattedTextField(new DecimalFormat("#0.0#"));
      Tz = new JFormattedTextField(new DecimalFormat("#0.0#"));
      Rx = new JFormattedTextField(new DecimalFormat("#0.0#"));
      Ry = new JFormattedTextField(new DecimalFormat("#0.0#"));
      Rz = new JFormattedTextField(new DecimalFormat("#0.0#"));
      Sx = new JFormattedTextField(new DecimalFormat("#0.0#"));
      Sy = new JFormattedTextField(new DecimalFormat("#0.0#"));
      Sz = new JFormattedTextField(new DecimalFormat("#0.0#"));
      
      JPanel outPanel = new JPanel();
      outPanel.setLayout(new GridLayout(4, 4));

      outPanel.add(new JLabel(""));
      JLabel lbl = new JLabel("X");
      lbl.setHorizontalAlignment(SwingConstants.CENTER);
      outPanel.add(lbl);
      lbl = new JLabel("Y");
      lbl.setHorizontalAlignment(SwingConstants.CENTER);
      outPanel.add(lbl);
      lbl = new JLabel("Z");
      lbl.setHorizontalAlignment(SwingConstants.CENTER);
      outPanel.add(lbl);

      lbl = new JLabel("T");
      lbl.setHorizontalAlignment(SwingConstants.CENTER);
      outPanel.add(lbl);
      outPanel.add(Tx);
      outPanel.add(Ty);
      outPanel.add(Tz);
      Tx.getDocument().addDocumentListener(this);
      Ty.getDocument().addDocumentListener(this);
      Tz.getDocument().addDocumentListener(this);

      lbl = new JLabel("R");
      lbl.setHorizontalAlignment(SwingConstants.CENTER);
      outPanel.add(lbl);
      outPanel.add(Rx);
      outPanel.add(Ry);
      outPanel.add(Rz);
      Rx.getDocument().addDocumentListener(this);
      Ry.getDocument().addDocumentListener(this);
      Rz.getDocument().addDocumentListener(this);

      lbl = new JLabel("S");
      lbl.setHorizontalAlignment(SwingConstants.CENTER);
      outPanel.add(lbl);
      outPanel.add(Sx);
      outPanel.add(Sy);
      outPanel.add(Sz);
      Sx.getDocument().addDocumentListener(this);
      Sy.getDocument().addDocumentListener(this);
      Sz.getDocument().addDocumentListener(this);
      
      this.setLayout(new GridLayout(2,1));
      this.add(outPanel);
    }

    /**
     * Sets the display fields to the current values of the transformations
     */
    public void refresh() {

      refreshing = true;
      Tx.setValue(new Float(T.x));
      Ty.setValue(new Float(T.y));
      Tz.setValue(new Float(T.z));
      Rx.setValue(new Float(R.x));
      Ry.setValue(new Float(R.y));
      Rz.setValue(new Float(R.z));
      Sx.setValue(new Float(S.x));
      Sy.setValue(new Float(S.y));
      Sz.setValue(new Float(S.z));
      refreshing = false;
    }

    /**
     * Updates the stored data
     */
    private void update() {

      if(refreshing)
        return;
      try {
        T.x = Float.parseFloat(Tx.getText());
        T.y = Float.parseFloat(Ty.getText());
        T.z = Float.parseFloat(Tz.getText());
        R.x = Float.parseFloat(Rx.getText());
        R.y = Float.parseFloat(Ry.getText());
        R.z = Float.parseFloat(Rz.getText());
        S.x = Float.parseFloat(Sx.getText());
        S.y = Float.parseFloat(Sy.getText());
        S.z = Float.parseFloat(Sz.getText());
        mf.refresh();
      }
      catch (NumberFormatException e) {
        // Do nothing just skip those updates for invalid data
      }

    }

    /**
     * @see javax.swing.event.DocumentListener#changedUpdate(javax.swing.event.DocumentEvent)
     */
    public void changedUpdate(DocumentEvent e) {

      update();
    }

    /**
     * @see javax.swing.event.DocumentListener#insertUpdate(javax.swing.event.DocumentEvent)
     */
    public void insertUpdate(DocumentEvent e) {

      update();
    }

    /**
     * @see javax.swing.event.DocumentListener#removeUpdate(javax.swing.event.DocumentEvent)
     */
    public void removeUpdate(DocumentEvent e) {

      update();
    }
  }
}
