package modeler;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.util.ArrayList;

import javax.swing.JPanel;
import javax.vecmath.Point2f;
import javax.vecmath.Tuple2f;
import javax.vecmath.Vector2f;

import modeler.shape.BezierRotation;
import modeler.shape.BezierSpline;
import modeler.shape.Shape;

public class SplineEditingPanel extends JPanel implements MouseListener, MouseMotionListener {

  private static final long serialVersionUID = 3258129163356287028L;
  
  // Some constants for window layout
  protected static final int LINES = 5;
  protected static final int DIVS = 5;

  protected static final double HANDLE_SIZE_SMOOTH = 3;
  protected static final double HANDLE_SIZE_CORNER = 2.5;

  // Distance thresholds for clicking on things and snapping to the center line
  protected static final double CLICK_PIX = 4;
  protected static final double SNAP_PIX = 4;

  protected static final Color gridColorDark = new Color(0.8f, 0.0f, 0.0f);
  protected static final Color gridColorMed = new Color(0.5f, 0.5f, 0.5f);
  protected static final Color gridColorLight = new Color(0.8f, 0.8f, 0.8f);
  protected static final Color splineColor = new Color(0, 0, 0);
  protected static final Color polygonColor = new Color(0.0f, 0.0f, 0.8f, 0.5f);
  protected static final Color handleColor = new Color(0, 0, 0);

  //The rendering mode
  public static final int RENDER_OFF = 0;
  public static final int RENDER_DISTANCE = 1;
  public static final int RENDER_FLAT = 2;

  // Are we currently showing the debug display, and if so which?
  protected int debugDisplay = RENDER_OFF;

  // Segment and point currently being edited
  protected BezierSpline selectedSpline;
  protected Point2f selectedPoint;

  // protected double distanceThreshold = 0.5;
  protected float scale = 100;
  protected float xShift = 250;
  protected float yShift = 250;

  //The space to place vertex data when rendering
  private final int[] iYPoints = new int[4];
  private final int[] iXPoints = new int[4];
  private final double[] dXPoints = new double[4];
  private final double[] dYPoints = new double[4];

  //Current mouse data
  private Point2f lastMousePoint = new Point2f();
  private Point2f mouseDelta = new Point2f();
  
  protected BezierRotation rotation;
  protected BezierSpline bezier;
  protected MainFrame mf;
  protected Dimension size = new Dimension();

  /**
   * Create a panel of the given size
   * @param d
   */
  public SplineEditingPanel(Dimension d, MainFrame mf) {

    this.mf = mf;
    addMouseListener(this);
    addMouseMotionListener(this);

    size.setSize(d);
    setSize(d);
    setPreferredSize(d);
    setMaximumSize(d);
    setMinimumSize(d);
  }

  /**
   * Set the bezier rotation for this panel
   * @param newRotation
   */
  public void setBezier(BezierRotation newRotation) {
    rotation = newRotation;
    bezier = newRotation.getSpline();
  }

  /**
   * Draws a GL line
   * @param g
   * @param x1
   * @param y1
   * @param x2
   * @param y2
   */
  public void drawLine(Graphics g, double x1, double y1, double x2, double y2) {

    g.drawLine((int) Math.round(xShift + x1 * scale), (int) Math.round(yShift - y1 * scale), (int) Math.round(xShift + x2 * scale), (int) Math.round(yShift - y2 * scale));
  }

  /**
   * Draws a GL filled rectangle
   * @param g
   * @param x
   * @param y
   * @param w
   * @param h
   */
  public void fillRect(Graphics g, double x, double y, double w, double h) {

    g.fillRect((int) Math.round(xShift + x * scale), (int) Math.round(yShift - y * scale - h * scale), (int) Math.round(w * scale), (int) Math.round(h * scale));
  }

  /**
   * Draws a GL filled polygon
   * @param g
   * @param xPoints
   * @param yPoints
   */
  public void fillPoly(Graphics g, double[] xPoints, double[] yPoints) {

    for (int ctr = 0; ctr < 4; ctr++) {
      iXPoints[ctr] = (int) Math.round(xShift + xPoints[ctr] * scale);
      iYPoints[ctr] = (int) Math.round(yShift - yPoints[ctr] * scale);
    }
    g.fillPolygon(iXPoints, iYPoints, 4);
  }

  /**
   * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
   */
  public void paintComponent(Graphics g) {

    super.paintComponent(g);
    Graphics2D g2d = null;
    if(g instanceof Graphics2D)
      g2d = (Graphics2D) g;
    if(g2d != null) {
      g2d.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
      g2d.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
    }
    g.setColor(Color.white);
    g.fillRect(0, 0, getWidth(), getHeight());

    // First draw a grid for the background
    g.setColor(gridColorLight);
    for (float a = 0; a < LINES; a += 1.0 / DIVS) {
      drawLine(g, a, -LINES, a, LINES);
      drawLine(g, -a, -LINES, -a, LINES);
      drawLine(g, -LINES, a, LINES, a);
      drawLine(g, -LINES, -a, LINES, -a);
    }

    g.setColor(gridColorMed);
    for (int a = 0; a < LINES; a++) {
      drawLine(g, a, -LINES, a, LINES);
      drawLine(g, -a, -LINES, -a, LINES);
      drawLine(g, -LINES, a, LINES, a);
      drawLine(g, -LINES, -a, LINES, -a);
    }

    g.setColor(gridColorDark);
    drawLine(g, 0, -LINES, 0, LINES);
    drawLine(g, -LINES, 0, LINES, 0);

    if (bezier == null)
      return;

    // Draw the spline curve
    ArrayList points = new ArrayList();
    bezier.getPointsByDistance(points, null, null, Shape.TOLERANCE);
    g.setColor(splineColor);
    for (int i = 0; i < points.size() - 1; i++) {
      Point2f p = (Point2f) points.get(i + 0);
      Point2f q = (Point2f) points.get(i + 1);
      drawLine(g, p.x, p.y, q.x, q.y);
    }
    

    // Third, draw the handles
    points.clear();
    ArrayList corners = new ArrayList();
    bezier.getHandles(points, corners);
    g.setColor(handleColor);
    for (int i = 0; i < points.size() - 1; i++) {
      Point2f p = (Point2f) points.get(i + 0);
      Point2f q = (Point2f) points.get(i + 1);
      drawHandle(g, p, ((Boolean) corners.get(i)).booleanValue());
      drawLine(g, p.x, p.y, q.x, q.y);
    }
    Point2f last = (Point2f) points.get(points.size() - 1);
    drawHandle(g, last, ((Boolean) corners.get(points.size() - 1)).booleanValue());
    
    // Finally, draw debugging information
    int subDivCount = 0;
    if (debugDisplay != RENDER_OFF) {

      //Get the debugging data to draw
      points.clear();
      ArrayList tangents = (debugDisplay == RENDER_FLAT) ? new ArrayList() : null;
      if (debugDisplay == RENDER_DISTANCE) {
        bezier.getPointsByDistance(points, tangents, null, Shape.TOLERANCE);
      }
      else {
        bezier.getPointsByFlatness(points, tangents, null, Shape.TOLERANCE);
      }
      subDivCount = points.size() - 1;

      //Draw the debugging data
      g.setColor(handleColor);
      Vector2f t = new Vector2f();
      for (int i = 0; i < points.size(); i++) {
        Point2f p = (Point2f) points.get(i);
        if (tangents != null) {
          t.set((Vector2f) tangents.get(i));
          t.normalize();
          t.scale(0.2f);
          t.set(-t.y, t.x);
          drawLine(g, p.x, p.y, p.x + t.x, p.y + t.y);
        }
        else {
          drawHandle(g, p, true);
        }
      }
    }
    
    //Print some useful information
    if(g2d != null) {

      Font font = Font.decode("Arial-BOLD-12");
      g2d.setFont(font);
      g2d.setColor(Color.BLUE);
      if (debugDisplay != RENDER_OFF) {
        g2d.drawString("Subdivisions: "+subDivCount, 5, size.height-55);
      }
      if(selectedSpline != null)
        selectedSpline.print(g2d, 5, size.height-40, 15);
      else bezier.print(g2d, 5, size.height-40, 15);
      
    }
  }

  /**
   * Draw the handle objects for the spline controls
   * @param g
   * @param p
   * @param smooth
   */
  protected void drawHandle(Graphics g, Point2f p, boolean smooth) {

    if (smooth) {
      double xOff = HANDLE_SIZE_SMOOTH / scale;
      double yOff = HANDLE_SIZE_SMOOTH / scale;
      dXPoints[0] = p.x + xOff;
      dXPoints[1] = p.x;
      dXPoints[2] = p.x - xOff;
      dXPoints[3] = p.x;

      dYPoints[0] = p.y;
      dYPoints[1] = p.y + yOff;
      dYPoints[2] = p.y;
      dYPoints[3] = p.y - yOff;

      fillPoly(g, dXPoints, dYPoints);
    }
    else {
      double xOff = HANDLE_SIZE_CORNER / scale;
      double yOff = HANDLE_SIZE_CORNER / scale;
      fillRect(g, p.x - xOff, p.y - yOff, 2 * xOff, 2 * yOff);
    }
  }

  /**
   * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
   */
  public void mousePressed(MouseEvent e) {

    Point2f mousePoint = new Point2f(e.getX(), e.getY());
    windowToWorld(mousePoint);
    selPoint(mousePoint);
    lastMousePoint.set(e.getX(), e.getY());
    windowToViewport(lastMousePoint);
    
  }

  /**
   * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
   */
  public void mouseReleased(MouseEvent e) {

    // Deselect any selected point
    selectedSpline = null;
    selectedPoint = null;
    rotation.mesh = null;
    mf.refresh();
    
  }

  /**
   * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
   */
  public void mouseClicked(MouseEvent e) {

    // Vector smooth = bezier.getSmooth();
    Point2f mousePoint = new Point2f(e.getX(), e.getY());
    windowToWorld(mousePoint);

    if (e.isShiftDown()) {
      
      // First try to delete a selected hanlde
      selPoint(mousePoint);
      if(selectedSpline != null && selectedSpline.removeControlPoint(selectedPoint)) {
        selectedSpline = null;
        selectedPoint = null;
        repaint();
        return;
      }

      //Try and split a segment
      if(bezier.addControlPointNear(mousePoint)) {
        selectedSpline = null;
        selectedPoint = null;
        repaint();
        return;
      }
    }
    
    else { // Try to toggle a joint between smooth and corner
      selPoint(mousePoint);
      if(selectedSpline != null)
        selectedSpline.toggleCorner(selectedPoint);
      repaint();
    }
  }

  /**
   * Sets the current mode
   * @param mode
   */
  public void setDebugMode(int mode) {

    debugDisplay = mode;
    repaint();
  }

  /**
   * Getter
   * @return
   */
  protected double getClickThreshold() {

    return CLICK_PIX / scale;
  }

  /**
   * Getter
   * @return
   */
  protected double getSnapThreshold() {

    return SNAP_PIX / scale;
  }

  /**
   * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
   */
  public void mouseDragged(MouseEvent e) {

    mouseDelta.set(e.getX(), e.getY());
    windowToViewport(mouseDelta);
    mouseDelta.sub(lastMousePoint);

    Point2f mousePoint = new Point2f(e.getX(), e.getY());
    windowToWorld(mousePoint);

    // Move selected point
    if(selectedSpline != null) {
      
      if(Math.abs(mousePoint.x) < getSnapThreshold())
        mousePoint.x = 0;
      if(Math.abs(mousePoint.y) < getSnapThreshold())
        mousePoint.y = 0;
      selectedSpline.updateControlPoint(selectedPoint, mousePoint);
      
    }
    else if ((e.getModifiersEx() & MouseEvent.CTRL_DOWN_MASK) == MouseEvent.CTRL_DOWN_MASK) {
      xShift += mouseDelta.x * getWidth() / 2;
      yShift -= mouseDelta.y * getHeight() / 2;
    }
    else if (e.isShiftDown()) {
      float a = (float) Math.pow(1.01, mouseDelta.y * getHeight() / 2);
      scale *= a;
      xShift = a * (xShift - 250) + 250;
      yShift = a * (yShift - 250) + 250;
    }

    lastMousePoint.set(e.getX(), e.getY());
    windowToViewport(lastMousePoint);
    repaint();
    
  }

  /**
   * Check to see if any of the control points are close to _mousePoint_, and if
   * so set the fields _selSeg_ and _selPt_ to indicate which one.
   * 
   * @param mousePoint The point against which all control points are compared.
   */
  protected void selPoint(Point2f mousePoint) {

    ArrayList points = new ArrayList();
    bezier.getHandles(points, null);
    double threshold = getClickThreshold();
    threshold *= threshold;
    for (int i = 0; i < points.size(); i++) {
      Point2f currPoint = (Point2f) points.get(i);
      if(currPoint.distanceSquared(mousePoint) < threshold) {
        this.selectedPoint = currPoint;
        selectedSpline = bezier.getSplineForPoint(currPoint);
        return;
      }
    }
  }

  /**
   * Converts coordinates
   * @param p
   */
  protected void windowToWorld(Point2f p) {

    p.x = (p.x - xShift) / scale;
    p.y = (yShift - p.y) / scale;
    
  }

  /**
   * Converts coordinates
   * @param p
   */
  protected void windowToViewport(Tuple2f p) {

    int w = getWidth();
    int h = getHeight();
    p.set((2 * p.x - w) / w, (2 * (h - p.y - 1) - h) / h);
  }

  ////////////////////////////////////////////////////////////////////////////////////////////////////
  // Unused interface methods
  ////////////////////////////////////////////////////////////////////////////////////////////////////
  
  public void mouseEntered(MouseEvent e) {}
  public void mouseExited(MouseEvent e) {}
  public void mouseMoved(MouseEvent e) {}

}