/*
 * Created on Sep 11, 2005 Copyright 2005 Program of Computer Grpahics, Cornell
 * University
 */
package resample.function;

import resample.image.Color;
import resample.image.Image;

/**
 * Default filter operations uses point sampling to resample and filter the images.
 * You will implement the same methods in FilterOperations overriding this default
 * behavior with the answer code.
 * 
 * @author arbree Sep 11, 2005 DefaultFilterOperations.java Copyright 2005
 *         Program of Computer Graphics, Cornell University
 */
public class DefaultFilterOperations {

  /**
   * Default behavior just point samples the original image.  That is the center of each output pixel
   * is looked up in the input grid and the color of each pixel that the sample falls in is chosen 
   * as the output color.
   * 
   * @param input the input image
   * @param output the output image
   * @param xSupportScale the horizontal radius of the filter
   * @param ySupportScale the vertical radius of the filter
   * @param filter the fitlering function
   */
  public static void filter(Image input, Image output, double xSupportScale, double ySupportScale, FilterFunction filter) {

    //The output width and height
    int width = output.getWidth();
    int height = output.getHeight();
    
    //The input width and height
    int inWidth = input.getWidth();
    int inHeight = input.getHeight();
    
    //The positions of the current output pixel center in the input pixel grid
    int xPos, yPos;
    
    //Color to use to perform lookups
    Color workColor = new Color();
    
    //Loop over all the pixels in the output and look up their positions in the input
    for (int x = 0; x < width; x++) {
      for (int y = 0; y < height; y++) {

        //Calculate the input positions of the current output pixel
        xPos = (int) ((x + 0.5)/width * inWidth);
        yPos = (int) ((y + 0.5)/height * inHeight);
        
        //Look up the output color in the input image
        input.getPixelColor(workColor, xPos, yPos);
        output.setPixelColor(workColor, x, y);

      }
    }
  }

  /**
   * Default bahavior just calls filter().
   * 
   * @param input the input image
   * @param output the output image
   * @param xSupportScale the xSupport scale
   * @param ySupportScale the ySupport scale
   * @param filter the filter function
   */
  public static void fastFilter(Image input, Image output, double xSupportScale, double ySupportScale, FilterFunction filter) {

    filter(input, output, xSupportScale, ySupportScale, filter);
    
  }

  /**
   * Just calls filter with filter scales 1.0 and 1.0 (they are ignored by the default implementation of 
   * filter anyway.
   * 
   * @param input the input image
   * @param output the output image
   * @param filter the filter function
   */
  public static void resample(Image input, Image output, FilterFunction filter) {

    filter(input, output, 1.0, 1.0, filter);
    
  }

  /**
   * Just calls resample().
   * 
   * @param input the input image
   * @param output the output image
   * @param filter the filter function
   */
  public static void fastResample(Image input, Image output, FilterFunction filter) {

    resample(input, output, filter);
    
  }
}
