/*
 * Copyright 2001 Program of Computer Graphics, Cornell University 580 Rhodes
 * Hall Cornell University Ithaca NY 14853 Web: http://www.graphics.cornell.edu/
 * Not for commercial use. Do not redistribute without permission.
 */

/*
 * PanelDisplay.java Created on March 5, 2000, 7:18 AM
 */

package ray1.viewer;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Window;
import java.awt.image.BufferedImage;
import java.io.File;

import javax.imageio.ImageIO;
import javax.swing.JPanel;

import ray1.misc.Image;


/**
 * JPanel for the display of images
 *
 * @author spf +latest $Author: arbree $
 * @version $Revision: 1.1 $, $Date: 2005/08/27 18:13:44 $
 */
public class PanelDisplay extends JPanel {

  /** An internal image for display* */
  private BufferedImage bufferedImage;

  /** The type of local image buffer* */
  private static final int BUFFER_TYPE = BufferedImage.TYPE_3BYTE_BGR;

  /** window to call when size changes* */
  final Window repackAncestor;

  /**
   * Default constructor
   */
  public PanelDisplay() {

    this(null);
  }

  /**
   * Constructor sets the ancerstor window
   *
   * @param ancestor
   */
  public PanelDisplay(java.awt.Window ancestor) {

    repackAncestor = ancestor;
    bufferedImage = new BufferedImage(128, 128, BUFFER_TYPE);
  }

  /**
   * @see java.awt.Component#getPreferredSize()
   */
  public Dimension getPreferredSize() {

    return new Dimension(bufferedImage.getWidth(), bufferedImage.getHeight());
  }

  /**
   * Saves the currently displayed image to the supplied file
   *
   * @param inFilename
   * @throws java.io.IOException if the file cannot be processed
   */
  public synchronized void saveCurrentImage(File inFilename) throws java.io.IOException {

    String fileName = inFilename.getName();
    String type = fileName.substring(fileName.lastIndexOf('.') + 1, fileName.length());
    ImageIO.write(bufferedImage, type, inFilename);
  }

  /**
   * Set the currently displayed image
   *
   * @param inImage the new image to display
   */
  public void setImage(Image inImage) {

    bufferImage(inImage);
    repaint(); // schedule repainting by event dispatching thread
    Thread.yield(); // give the update thread a chance to run
  }

  /**
   * @see javax.swing.JComponent#paintComponent(java.awt.Graphics)
   */
  public void paintComponent(Graphics g) {

    synchronized (this) { // make sure image doesn't change while we are drawing
                          // it
      g.drawImage(bufferedImage, 0, 0, null);
    }
  }

  /**
   * Copy the input image into the internal buffered image used for display
   * @param image the image to place in the buffer
   */
  private synchronized void bufferImage(Image image) {

    bufferedImage = image.asBufferedImage();

  }
}