package ray1.surface;

import java.io.BufferedReader;
import java.io.FileReader;

import ray1.material.Material;
import ray1.misc.IntersectionRecord;
import ray1.misc.Ray;
import ray1.surface.Surface;


/**
 * Basic packed triangle mesh. The triangle mesh is primarily data storage, all
 * geometric interaction is handled through MeshTriangle objects.
 *
 * @author arbree Aug 19, 2005 TriangleMesh.java Copyright 2005 Program of
 *         Computer Graphics, Cornell University
 */
public class Mesh extends Surface {

  /** The material for the mesh * */
  protected Material material = Material.DEFAULT_MATERIAL;

  /** The number of vertices in the mesh * */
  protected int numVertices;

  /** The number of triangles in the mesh * */
  protected int numTriangles;

  /** The vertex array -- always present in each mesh * */
  protected float[] verts;

  /** The texture coordinate array -- may be null * */
  protected float[] texcoords;

  /** The normal coordinate array -- may be null * */
  protected float[] normals;

  /** Mesh triangle objects for each triangle. */
  protected MeshTriangle[] triangles;

  /**
   * Default constructor creates an empty mesh
   */
  public Mesh() { }

  /**
   * Basic constructor. Sets mesh data array into the mesh structure. IMPORTANT:
   * The data array are not copies so changes to the input data array will
   * affect the mesh structure. The number of vertices and the number of
   * triangles are inferred from the lengths of the verts and tris array. If
   * either is not a multiple of three, an error is thrown.
   *
   * @param verts the vertex data
   * @param tris the triangle data
   * @param normals the normal data
   * @param texcoords the texture coordinate data
   */
  public Mesh(float[] verts, int[] tris, float[] normals, float[] texcoords, Material material) {

    // check the inputs
    if (verts.length % 3 != 0)
      throw new Error("Vertex array for a triangle mesh is not a multiple of 3.");
    if (tris.length % 3 != 0)
      throw new Error("Triangle array for a triangle mesh is not a multiple of 3.");

    // Set data
    this.material = material;
    setMeshData(verts, tris, normals, texcoords);

  }

  /**
   * Sets the mesh data and builds the triangle array.
   * @param verts the vertices
   * @param tris the triangles
   * @param normals the normals
   * @param texcoords the texture coordinates
   */
  private void setMeshData(float[] verts, int[] tris, float[] normals, float[] texcoords) {

    this.numVertices = verts.length / 3;
    this.numTriangles = tris.length / 3;
    this.verts = verts;
    this.normals = normals;
    this.texcoords = texcoords;

    // Build the mesh triangles
    triangles = new MeshTriangle[numTriangles];
    for (int i = 0; i < numTriangles; i++)
      triangles[i] = new MeshTriangle(this, tris[3 * i], tris[3 * i + 1], tris[3 * i + 2]);

  }

  /**
   * Returns the number of triangles
   *
   * @return the number of triangles
   */
  public int getNumTriangles() {

    return numTriangles;

  }

  /**
   * Returns the number of vertices
   *
   * @return the number of vertices
   */
  public int getNumVertices() {

    return numVertices;

  }

  /**
   * Note: returns the actual list of triangles changes will affect the mesh.
   *
   * @return Returns the triangles.
   */
  public MeshTriangle[] getTriangles() {

    return this.triangles;
  }

  /**
   * @see ray1.surface.Surface#getMaterial()
   */
  public Material getMaterial() {

    return material;

  }

  /**
   * @param material The material to set.
   */
  public void setMaterial(Material material) {

    this.material = material;
  }

  /**
   * Set the data in this mesh to the data in fileName
   * @param fileName the name of a .msh file
   */
  public void setData(String fileName) {

    readMesh(this, fileName);

  }

  /**
   * @see ray1.surface.Surface#intersect(ray1.misc.IntersectionRecord, ray1.misc.Ray)
   */
  public boolean intersect(IntersectionRecord outRecord, Ray ray) {

    throw new Error("Meshes cannot intersect rays.  Test for intersection against the individual mesh triangles.");

  }

  /**
   * Reads a .msh file into outputMesh.
   *
   * @param outputMesh the mesh to store the read data
   * @param fileName the name of the mesh file to read
   * @return the TriangleMesh from the file
   */
  public static final void readMesh(Mesh outputMesh, String fileName) {

    float[] vertices;
    int[] triangles;
    float[] normals;
    float[] texcoords;

    try {

      // Create a buffered reader for the mesh file
      BufferedReader fr = new BufferedReader(new FileReader(fileName));

      // Read the size of the file
      int nPoints = Integer.parseInt(fr.readLine());
      int nPolys = Integer.parseInt(fr.readLine());

      // Create arrays for mesh data
      vertices = new float[nPoints*3];
      triangles = new int[nPolys*3];
      normals = null;
      texcoords = null;

      // read vertices
      if (!fr.readLine().equals("vertices")) throw new RuntimeException("Broken file - vertices expected");
      for (int i=0; i<vertices.length; ++i) {
          vertices[i] = Float.parseFloat(fr.readLine());
      }

      // read triangles
      if (!fr.readLine().equals("triangles")) throw new RuntimeException("Broken file - triangles expected.");
      for (int i=0; i<triangles.length; ++i) {
          triangles[i] = Integer.parseInt(fr.readLine());
      }

      // read texcoords
      String line = fr.readLine();
      if (line != null && line.equals("texcoords")) {
          texcoords = new float[nPoints*2];
          line = null;
          for (int i=0; i<texcoords.length; ++i) {
              texcoords[i] = Float.parseFloat(fr.readLine());
          }
      }

      // Make sure that if tex coords were missing, but normals were
      // still there, we can still read in the normals, rather than losing
      // the "normals" keyword
      if (line == null) {
          line = fr.readLine();
      }
      if (line != null && line.equals("normals")) {
          normals = new float[nPoints*3];
          for (int i=0; i<normals.length; ++i) {
              normals[i] = Float.parseFloat(fr.readLine());
          }
      }
    }
    catch (Exception e) {
      throw new Error("Error reading mesh file: "+fileName);
    }

    //Set the data in the output Mesh
    outputMesh.setMeshData(vertices, triangles, normals, texcoords);

  }
}
