package ray1.misc;

import ray1.math.Point3;
import ray1.math.Vector3;

/**
 * A ray is simply an origin point and a direction vector.  By giving the 
 * ray a start and end t value, the class can represent both ray and line
 * segments.
 *
 * @author ags
 */
public class Ray {

  /**
   * This quantity represents a "small amount" to handle comparisons in
   * floating point computations.  It is often useful to have one global
   * value across the ray tracer that stands for a "small amount".
   */
  public static final double EPSILON = 1e-6;

  /** The starting point of the ray. */
  public final Point3 origin = new Point3();

  /** The normalized direction in which the ray travels. */
  public final Vector3 direction = new Vector3();

  /** Starting t value of the ray **/
  public double start;

  /** Ending t value of the ray **/
  public double end;

  /**
   * Default constructor generates a trivial ray.
   */
  public Ray() {};

  /**
   * The explicit constructor.  This is the only constructor with any real
   * code in it.  Values should be set here, and any variables that need to
   * be calculated should be done here.
   * @param newOrigin The origin of the new ray.
   * @param newDirection The direction of the new ray.
   */
  public Ray(Point3 newOrigin, Vector3 newDirection) {

    origin.set(newOrigin);
    direction.set(newDirection);
  }

  /**
   * Sets this ray with the given direction and origin.
   * @param newOrigin the new origin point
   * @param newDirection the new direction vector
   */
  public void set(Point3 newOrigin, Vector3 newDirection) {

    origin.set(newOrigin);
    direction.set(newDirection);
  }

  /**
   * Sets outPoint to the point on this ray t units from the origin.  Note that t can
   * be considered as distance along this ray only if the ray direction is normalized.
   * @param outPoint the output point
   * @param t The distance along the ray.
   */
  public void evaluate(Point3 outPoint, double t) {

    outPoint.set(origin);
    outPoint.scaleAdd(t, direction);

  }
}