package ray1.material;

import ray1.math.Vector3;
import ray1.misc.Color;
import ray1.misc.IntersectionRecord;

/**
 * A Phong material. Uses the Modified Blinn-Phong model which is energy
 * preserving and reciprocal.
 *
 * @author ags
 */
public class Phong implements Material {

  /** The diffuse color of the phong material. */
  protected final Color diffuseColor = new Color(1, 1, 1);

  /** The specular color of the phong material. */
  protected final Color specularColor = new Color(1, 1, 1);

  /** The phong exponent of the phong material. */
  protected double exponent = 1.0;

  /**
   * Default constructor, creates a phong material with the default values.
   */
  public Phong() {

  }

  /**
   * The explicit constructor. This is the only constructor with any real code
   * in it. Values should be set here, and any variables that need to be
   * calculated should be done here.
   *
   * @param newDiffuseColor The diffuse color of the new phong material.
   * @param newSpecularColor The specular color of the new phong material.
   * @param newExponent The phong exponent of the new phong material.
   */
  public Phong(Color newDiffuseColor, Color newSpecularColor, double newExponent) {

    diffuseColor.set(newDiffuseColor);
    specularColor.set(newSpecularColor);
    exponent = newExponent;
  }

  /**
   * Returns the diffuse color of this material in outColor
   *
   * @param outColor output space
   */
  public void getDiffuseColor(Color outColor) {

    outColor.set(diffuseColor);

  }

  /**
   * Sets the diffuse color of this material
   *
   * @param inColor
   */
  public void setDiffuseColor(Color inColor) {

    diffuseColor.set(inColor);

  }

  /**
   * @return Returns the exponent.
   */
  public double getExponent() {

    return this.exponent;
  }

  /**
   * @param exponent The exponent to set.
   */
  public void setExponent(double exponent) {

    this.exponent = exponent;
  }

  /**
   * Sets outColor to the specular color of this material
   *
   * @param outColor output space
   */
  public void getSpecularColor(Color outColor) {

    outColor.set(specularColor);

  }

  /**
   * Sets the specular color of this material to inColor
   *
   * @param inColor the input color
   */
  public void setSpecularColor(Color inColor) {

    specularColor.set(inColor);

  }

  /**
   * @see ray1.material.Material#shade(ray1.misc.Color, ray1.math.Vector3,
   *      ray1.math.Vector3, ray1.misc.IntersectionRecord)
   */
  public void shade(Color outColor, Vector3 toLight, Vector3 toEye, IntersectionRecord record) {

    //TODO: Implement this method

  }

  /**
   * @see Object#toString()
   */
  public String toString() {

    return "phong " + diffuseColor + " " + specularColor + " " + exponent + " end";
  }
}