package pipeline.triangle;


import javax.vecmath.Color3f;
import javax.vecmath.Vector2f;
import javax.vecmath.Vector3f;

import pipeline.Pipeline;
import pipeline.misc.Vertex;



/**
 * This class is a wrapper for the code that does triangle processing -- or vertex
 * processing, in the more traditional pipeline terminology.  Its mission is to transform
 * a triangle into screen space vertex coordinates for the rasterizer and to translate
 * the vertex attributes supplied by the caller into a set of attributes that will
 * provide the necessary information for the fragment processing stage.
 */ 
public abstract class TriangleProcessor {
    
    /** A list of valid Triangle Processors */
    public static Class[] classes = {
            ConstColorTP.class, TexturedTP.class, FlatShadedTP.class, SmoothShadedTP.class,
            FragmentShadedTP.class, TexturedFragmentShadedTP.class
    };
    
    /**
     * Returns the number of attributes this triangle processor will provide.
     * @return The number of attributes this triangle processor will provide.
     */
    public abstract int nAttr();
    
    /**
     * We can access everything we need to know about the pipeline state -- the current
     * transformation matrices and the lighting parameters -- via the Pipeline reference
     * above.  But for efficiency we may want to do some precomputation.  This function
     * will be called by the pipeline to notify this object whenever the transforms
     * parameters are changed.
     * @param pipe The reference to the pipeline object. Can be used to determine the
     * necessary matrices.
     */
    public abstract void updateTransforms(Pipeline pipe);

    /**
     * We can access everything we need to know about the pipeline state -- the current
     * transformation matrices and the lighting parameters -- via the Pipeline reference
     * above.  But for efficiency we may want to do some precomputation.  This function
     * will be called by the pipeline to notify this object whenever the lighting
     * parameters are changed.
     * @param pipe The reference to the pipeline object. Cna be used to determine the
     * viewing conditions.
     */
    public void updateLightModel(Pipeline pipe) {}

    /**
     * This is the main function of this class, which is called once for every triangle
     * in the scene.  As input we get all the attributes of the triangle's three
     * vertices, and as a result this function should call the rasterizer with
     * appropriate vertex positions and attribute values.
     * @param v The three vertices in 3D object coordinates.
     * @param c Colors associated with each of the vertices.
     * @param n A normal for each vertex.
     * @param t Texture coordinates for each vertex.
     * @param output The array of 3 vertices that serves as the output for this method.
     */
    public abstract void triangle(Vector3f[] v, Color3f[] c, Vector3f[] n, Vector2f[] t, Vertex[] output);

}