#pragma once

#include <cstdint>
#include <list>
#include <optional>
#include <queue>
#include <set>
#include <vector>

namespace sokoban {
typedef std::pair<const uint32_t, const uint32_t> move_t;

class sokoban_board {
public:
    sokoban_board(const uint32_t num_rows, const uint32_t num_cols,
                  const std::vector<bool>& texture);
    sokoban_board(const sokoban_board&) = delete;
    sokoban_board clone(
            const std::vector<uint32_t>& unreachable = {}) const;

    bool operator()(const uint32_t row_num,
                    const uint32_t col_num) const;
    uint32_t size() const;

    const uint32_t num_rows;
    const uint32_t num_cols;
    const std::vector<bool> texture;
};

class sokoban_state {
public:
    sokoban_state(const sokoban_board& board,
                  const std::vector<uint32_t>& box_positions,
                  const uint32_t player_position,
                  const std::list<move_t>& trace = {});
    sokoban_state(const sokoban_state& other);
    bool operator<(const sokoban_state& other) const;

    sokoban_state make_move(const move_t& move) const;
    std::vector<move_t> find_all_moves() const;
    bool is_final_state(
            const std::vector<uint32_t>& targets) const;
    bool is_dead(
            const std::vector<uint32_t>& targets) const;
    sokoban_board copy_board() const;
    sokoban_state reinitialize(const sokoban_board& board) const;
    std::list<move_t> get_trace();

private:
    const sokoban_board& board;
    std::vector<uint32_t> box_positions;
    uint32_t player_position;
    std::list<move_t> trace;
};

class sokoban_solver {
public:
    sokoban_solver(const sokoban_state& initial_state,
                   const std::vector<uint32_t>& targets);

    std::optional<std::list<move_t>> solve();

private:
    const sokoban_board board;

    std::queue<sokoban_state> states_to_explore;
    const std::vector<uint32_t> targets;
    std::set<sokoban_state> visited;
};
}  // namespace sokoban
