#include <chrono>
#include <cstdint>
#include <functional>
#include <iostream>
#include <list>
#include <vector>

#include "sokoban_solver/sokoban.hpp"

int main() {
    // start timer
    auto start_time = std::chrono::steady_clock::now();

    uint32_t num_rows;
    uint32_t num_cols;

    std::cin >> num_rows;
    std::cin >> num_cols;

    std::vector<bool> board_texture(num_rows * num_cols);
    std::vector<uint32_t> box_positions;
    uint32_t player_position = (uint32_t) -1;
    uint32_t board_index = 0;

    for(uint32_t i = 0; i < num_rows; ++i) {
        for(uint32_t j = 0; j < num_cols; ++j) {
            char cell_entry;
            std::cin >> cell_entry;
            switch(cell_entry) {
                case 'W':
                    board_texture[board_index] = false;
		    break;
                case 'N':
                    board_texture[board_index] = true;
		    break;
	        case 'B':
		    board_texture[board_index] = true;
		    box_positions.push_back(board_index);
		    break;
	        case 'P':
		    board_texture[board_index] = true;
		    player_position = board_index;
		    break;
	        default:
                std::cerr << "Invalid input\n";
                return 1;
            }
            board_index++;
        }
    }

    if(player_position == (uint32_t)-1) {
        std::cerr << "Invalid input\n";
        return 1;
    }

    uint32_t num_targets;
    std::cin >> num_targets;
    if (num_targets != box_positions.size()) {
        std::cerr << "Invalid input\n";
        return 1;
    }

    std::vector<uint32_t> targets(num_targets);
    for (uint32_t i = 0; i < num_targets; ++i) {
        uint32_t row_num;
	uint32_t col_num;
        std::cin >> row_num;
	std::cin >> col_num;

	targets[i] = row_num * num_cols + col_num;
    }

    sokoban::sokoban_board board(num_rows, num_cols,
                                 board_texture);
    sokoban::sokoban_state initial_state(
            board, box_positions, player_position);
    sokoban::sokoban_solver solver(initial_state, targets);
    
    std::optional<std::list<sokoban::move_t>> solution = solver.solve();
    if(solution) {
        std::function<uint32_t(const uint32_t)> get_row =
                [&](const uint32_t pos) {
                    return pos / num_cols;
                };
        std::function<uint32_t(const uint32_t)> get_col =
                [&](const uint32_t pos) {
                    return pos % num_cols;
                };

        // print the steps
        for(const sokoban::move_t& move : solution.value()) {
            std::cout << "(" << get_row(move.first)
                      << ", " << get_col(move.first)
                      << ") ---> ("
                      << get_row(move.second) << ", "
                      << get_col(move.second) << ")\n";
        }
    } else {
        std::cout << "No solution found" << std::endl;
    }

    // end timer
    auto end_time = std::chrono::steady_clock::now();

    uint64_t nanoseconds_elapsed = std::chrono::duration_cast<std::chrono::nanoseconds>(end_time - start_time).count();

    std::cout << (double)nanoseconds_elapsed / 1000000 << "ms" << std::endl;
}
